{ Numerical stability for stochastic differential equations                }
{ Exact solution, explicit and implicit Euler approximation                }
{ Considers the stochastic differential equation(SDE)                      }
{               dX(t) = A(t,X(t)) dt + B(t,X(t)) dW(t)                     }
{           with  X(T0) = X0   on the time interval [T0,T]                 }
{ where W(t) is a Wiener process                                           }
{ Equidistant approximation of X(t) by the explicit and implicit Euler     }
{ scheme with the time step size DELTA                                     }
{ Uses the Polar Marsaglia method to generate Gaussian random numbers      }
{ Plots the linearly interpolated trajectories of X(t) and the Euler ap-   }
{ proximations w.r.t. the highest possible resolution in the same graphic  }
{ Written by Henri Schurz, 9.10. 1991                                      }
{ Note : Using another SDE generally requires another implicit scheme.     }
{        Be careful if using time step sizes other than those indicated.   }
{        The maximum of the numbers of the time steps used can not be      }
{        larger than NUMINV. Otherwise the corresponding selection of data }
{        has to be reorganized in order to use AAGRAPHS-routines. The time }
{        step size DELTA can not be smaller than the DELTA_X used to eval- }
{        uate the exact solution.                                          }

PROGRAM EXX3X5X1; { Exercise 3.5.1 }

USES CRT,DOS,GRAPH,INIT,SETSCR,SERVICE,RANDNUMB,AAGRAPHS;

CONST
 T0=0.0;         { left end point                                       }
 T=1.0;          { right end point                                      }
 DELTA=(T-T0)/8; { time step size, divide by 8 or 16 as required        }
 X0=1.0;         { initial value                                        }
 ALPHA=-15.0;    { parameter(<< 0 for stiffness) in the function A(t,x) }
 BETA=0.5;       { parameter in the function B(t,x)                     }

VAR
 CR:STRING;          { help string                                  }
 I,J:INTEGER;        { time steps                                   }
 APPROXSTEP:INTEGER; { ratio of the time step sizes                 }
 TI:REAL;            { subinterval point                            }
 YT1OLD,YT2OLD:REAL; { last values of the approximations            }
 DELTA_X:REAL;       { time step size to plot the exact solution    }
 SQDELTA_X:REAL;     { square root of the time step size DELTA_X    }
 G1,G2:REAL;         { Gaussian random numbers                      }
 DWTI:REAL;          { Wiener process increment  W(t(i+1)) - W(ti)  }
 DWT:VECTOR;         { values of the Wiener process increments      }
 XT:VECTOR;          { values of the exact solution X(t)            }
 YT1:VECTOR;         { values of the explicit Euler approximation   }
 YT2:VECTOR;         { values of the implicit Euler approximation   }
 ABSCISSA:VECTOR;    { values of the subinterval points             }

{ Generates the drift function A(t,x) }

FUNCTION A(TI,XI:REAL):REAL;
BEGIN
 A:=ALPHA*XI;
END;{ A }

{ Generates the diffusion function B(t,x) }

FUNCTION B(TI,XI:REAL):REAL;
BEGIN
 B:=BETA;
END;{ B }

{ Generates the explicit solution X(TI) using X(TI-DELTA_X) }
{ Assumes i.i.d. Gaussian random numbers G1 and G2          }

FUNCTION EXPLSOL(G1,G2,XTOLD:REAL):REAL;
VAR
 EQ,Q,S22:REAL;
BEGIN
 Q:=ALPHA*DELTA_X; { integration by splitting and linear combination }
 EQ:=EXP(Q);
 S22:=EQ*EQ*(0.5/ALPHA-DELTA_X)+2.*EQ*(DELTA_X-1/ALPHA)+1.5/ALPHA;
 S22:=SQRT(ABS(S22));
 EXPLSOL:=EQ*XTOLD+BETA*(EQ*SQDELTA_X*G1+S22*G2); { approximate value }
END;{ EXPLSOL }

{ Main program : }

BEGIN

 INITIALIZE; { standard initialization }
 MAINWINDOW('Exercise 3.5.1');
 STATUSLINE('Be patient! This will take the computer some time!');
 APPROXSTEP:=TRUNC(NUMINV*DELTA/(T-T0)+0.0000001); { ratio of step sizes }

{ Generation of the Wiener process increments DWT[I]            }
{ Evaluation of the exact solution and initialization of data : }

 DELTA_X:=(T-T0)/NUMINV;   { time step size for the exact solution }
 SQDELTA_X:=SQRT(DELTA_X); { square root of DELTA_X                }
 TI:=T0;                   { initial time                          }
 XT[0]:=X0;                { initial value of the exact solution   }
 ABSCISSA[0]:=T0;          { value of the x-axis                   }
 FOR I:=1 TO NUMINV DO
  BEGIN
   TI:=TI+DELTA_X;  { time                        }
   GENERATE(G1,G2); { uses Polar Marsaglia method }
   DWT[I-1]:=G1*SQDELTA_X; { Wiener process increment  W(t(i+1)) - W(ti) }
   XT[I]:=EXPLSOL(G1,G2,XT[I-1]); { value of the exact solution          }
   ABSCISSA[I]:=TI;        { value of the x-axis                         }
  END;

{ Generation of the Euler approximations : }

 I:=0;
 TI:=T0;     { initial time                                      }
 YT1[0]:=X0; { initial value of the explicit Euler approximation }
 YT2[0]:=X0; { initial value of the implicit Euler approximation }
 WHILE TI<T DO
  BEGIN
   I:=I+1; { time step }
   TI:=TI+DELTA; { current time }
   YT1OLD:=YT1[(I-1)*APPROXSTEP]; { saves the old value }
   YT2OLD:=YT2[(I-1)*APPROXSTEP]; { saves the old value }
   DWTI:=0.0; { calculates Wiener process increment for Euler time step }
   FOR J:=1 TO APPROXSTEP DO DWTI:=DWTI+DWT[(I-1)*APPROXSTEP+J-1];

  { Explicit Euler scheme : }

   YT1[I*APPROXSTEP]:=YT1OLD+A(TI-DELTA,YT1OLD)*DELTA+B(TI-DELTA,YT1OLD)*DWTI;

  { Implicit Euler scheme(has to be replaced for other SDE's) : }

   YT2[I*APPROXSTEP]:=(YT2OLD+BETA*DWTI)/(1.0-ALPHA*DELTA);

  { Interpolation for the other values : }

   IF APPROXSTEP>1 THEN
    FOR J:=1 TO APPROXSTEP-1 DO
     BEGIN
      YT1[(I-1)*APPROXSTEP+J]:=YT1OLD+J*(YT1[I*APPROXSTEP]-YT1OLD)/APPROXSTEP;
      YT2[(I-1)*APPROXSTEP+J]:=YT2OLD+J*(YT2[I*APPROXSTEP]-YT2OLD)/APPROXSTEP;
     END;

  END;{ WHILE }

{ Printout : }

 CLEARDEVICE;
 GRAPH321(XT,YT1,ABSCISSA,'X(t), Y(t)','t','X(t)','Y(t)'); { plots the paths }
 CR:='Exact solution X(t) and explicit Euler approximation Y(t) using ';
 CR:=CR+'DELTA = '+CHCR(DELTA);
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 CLEARDEVICE;
 GRAPH321(XT,YT2,ABSCISSA,'X(t), Y(t)','t','X(t)','Y(t)'); { plots the paths }
 CR:='Exact solution X(t) and implicit Euler approximation Y(t) using ';
 CR:=CR+'DELTA = '+CHCR(DELTA);
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 MYEXITPROC; { closes graphics mode and sets the old procedure address }
END.{ EXX3X5X1 }