{ Numerical stability for stochastic differential equations                 }
{ Considers the stochastic differential equation   dX(t) = -16 X(t) dt      }
{           with  X(T0) = X0   on the time interval [T0,T]                  }
{ Equidistant approximation of X(t) by the Heun method and the implicit     }
{ trapezoidal method  with the time step size DELTA                         }
{ Plots the linearly interpolated trajectories of the both approximations   }
{ Written by Henri Schurz, 9.10. 1991                                       }
{ Note : If using time step sizes DELTA larger than 1/8 the plotting of the }
{        Heun approximation requires other scaling parameters.              }

PROGRAM PRX3X5X2;

USES CRT,DOS,GRAPH,INIT,SETSCR,SERVICE;

CONST
 NUM=8;            { number of time steps                 }
 T0=0.0;           { left end point                       }
 T=1.0;            { right end point                      }
 DELTA=(T-T0)/NUM; { time step size of the approximations }
 X0=1.0;           { initial value                        }
 ABSCMIN=T0;       { left end point                       }
 ABSCMAX=T;        { right end point                      }
 ORDMIN=0.0;       { minimum of the ordinate              }
 ORDMAX=+1.0;      { maximum of the ordinate              }
 ORDPOINT=0.5;     { significant ordinate point           }

TYPE
 VECTOR=ARRAY[0..NUM] OF REAL;

VAR
 CR:STRING;           { help string                                      }
 AXISX,AXISY:INTEGER; { location of the axes                             }
 DISTX,DISTY:INTEGER; { scale parameters                                 }
 NN:INTEGER;          { number of time steps                             }
 I:INTEGER;           { time step                                        }
 YT1:VECTOR;          { values of the Heun approximation                 }
 YT2:VECTOR;          { values of the implicit trapezoidal approximation }
 ABSCISSA:VECTOR;     { values of the subinterval points                 }

{ Prepares the screen for the printout of the trajectories       }
{ X-axis is placed in the bottom; CY,CX ... strings for the axes }

PROCEDURE COORDSYS(CY,CX:STRING);
VAR
 FACTOR,I0,I1,I2,IH,K:INTEGER;
 DX,DY:REAL;
BEGIN
 IH:=TEXTHEIGHT('M')+10;
 AXISX:=TRUNC(MAXY-4*IH);
 I0:=TEXTWIDTH(CHCR(ORDMAX))+7;I1:=TEXTWIDTH(CHCR(ORDMIN))+7;
 I2:=TEXTWIDTH(CHCR(ORDPOINT))+7;IF I0<I1 THEN I0:=I1;
 IF I0<I2 THEN I0:=I2;I1:=TRUNC(MAXX/10);
 IF I0>MAXX-NUMINV-40 THEN I0:=MAXX-NUMINV-40;
 IF I0<I1 THEN AXISY:=I1 ELSE AXISY:=I0;
 DISTX:=NUMINV;DISTY:=AXISX-2*IH-20;
 SETTEXTSTYLE(DEFAULTFONT,HORIZDIR,1);SETTEXTJUSTIFY(1,1);
 OUTTEXTXY(AXISY,AXISX-DISTY-IH,'^');OUTTEXTXY(AXISY,AXISX-DISTY-2*IH,CY);
 LINE(AXISY,AXISX-DISTY-IH,AXISY,AXISX+5);
 LINE(AXISY-5,AXISX,AXISY+30+DISTX,AXISX);
 SETTEXTJUSTIFY(2,1);I0:=AXISX-DISTY;
 LINE(AXISY-3,I0,AXISY+3,I0);OUTTEXTXY(AXISY-7,I0,CHCR(ORDMAX));
 DY:=(ORDPOINT-ORDMIN)/(ORDMAX-ORDMIN);I0:=AXISX-TRUNC(DY*DISTY);
 LINE(AXISY-3,I0,AXISY+3,I0);OUTTEXTXY(AXISY-7,I0,CHCR(ORDPOINT));
 OUTTEXTXY(AXISY-7,AXISX,CHCR(ORDMIN));SETTEXTJUSTIFY(1,1);
 OUTTEXTXY(AXISY+30+DISTX,AXISX+1,'>');OUTTEXTXY(AXISY+30+DISTX,AXISX+10,CX);
 IF ((ABSCMAX-ABSCMIN=2.0) OR (ABSCMAX+ABSCMIN=0.0)) THEN FACTOR:=2 ELSE
  IF ABSCMAX-ABSCMIN=3.0 THEN FACTOR:=3 ELSE
   IF ABSCMAX-ABSCMIN=4.0 THEN FACTOR:=4 ELSE
    IF ((ABSCMAX-ABSCMIN=5.0) OR ((ABSCMAX-ABSCMIN=1.0) AND (ABSCMIN=0.0)))
     THEN FACTOR:=5 ELSE FACTOR:=1;
 DX:=(ABSCMAX-ABSCMIN)/FACTOR;
 FOR K:=0 TO FACTOR DO
  BEGIN
   I1:=AXISY+TRUNC(K*DISTX/FACTOR+0.5);I2:=AXISX+IH;
   LINE(I1,AXISX+3,I1,AXISX-3);
   OUTTEXTXY(I1,I2,CHCR(ABSCMIN+K*DX));
  END;
END;{ COORDSYS }

{ Plots the trajectory on the screen assuming COORDSYS was called }
{ before and x-axis is placed in the bottom of the screen         }
{ N0,NN = indices of the first and last data picked               }
{ LTN = line thickness                                            }

PROCEDURE PLOTGRAPH1(LTN,N0,NN:INTEGER;ORDINATE,ABSCISSA:VECTOR);
VAR
 IL,IR,IHL,IHR:INTEGER; { screen coordinates }
 I:INTEGER;             { data index         }
 FACTORX,FACTORY:REAL;  { scaling parameters }
BEGIN
 SETLINESTYLE(0,0,LTN);
 FACTORX:=DISTX/(ABSCMAX-ABSCMIN);
 FACTORY:=DISTY/(ORDMAX-ORDMIN);
 IR:=AXISY+TRUNC((ABSCISSA[N0]-ABSCMIN)*FACTORX);
 IHR:=AXISX-TRUNC((ORDINATE[N0]-ORDMIN)*FACTORY);
 I:=N0;
 REPEAT
  I:=I+1;
  IL:=IR;IHL:=IHR;
  IR:=AXISY+TRUNC((ABSCISSA[I]-ABSCMIN)*FACTORX);
  IHR:=AXISX-TRUNC((ORDINATE[I]-ORDMIN)*FACTORY);
  LINE(IL,IHL,IR,IHR);
 UNTIL I=NN;
 SETLINESTYLE(0,0,1);
END;{ PLOTGRAPH1 }

{ Main program : }

BEGIN

 INITIALIZE; { standard initialization }
 MAINWINDOW('Problem 3.5.2');
 STATUSLINE('Be patient! This will take the computer some time!');
 NN:=TRUNC((T-T0)/DELTA+0.1);

{ Generation of the approximations : }

 YT1[0]:=X0;YT2[0]:=X0; { initial values }
 ABSCISSA[0]:=T0;
 I:=0;
 REPEAT
  I:=I+1; { time step }
  ABSCISSA[I]:=ABSCISSA[I-1]+DELTA; { value of the x-axis }

 { Heun scheme : }

  YT1[I]:=(1.0-16.*(1.0-8*DELTA)*DELTA)*YT1[I-1];

 { Implicit trapezoidal scheme : }

  YT2[I]:=(1.0-8.*DELTA)*YT2[I-1]/(1.0+8*DELTA);

 UNTIL I=NN;

{ Printout : }

 CLEARDEVICE;COORDSYS('X(t),Y(t)','t'); { draws the coordinate system       }
 PLOTGRAPH1(1,0,NN,YT1,ABSCISSA); { plots the Heun approximation for DELTA  }
 PLOTGRAPH1(3,0,NN,YT2,ABSCISSA); { plots the trapezoidal approx. for DELTA }
 CR:='Heun(thin) and implicit trapezoidal approximation(thick)';
 CR:=CR+' using DELTA='+CHCR(DELTA);
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 MYEXITPROC; { closes graphics mode and sets the old procedure address }
END.{ PRX3X5X2 }