{ Numerical stability for stochastic differential equations                 }
{ Considers the stochastic differential equation  dX(t) = - ALPHA * X(t) dt }
{           with  X(T0) = X0   on the time interval [T0,T]                  }
{ Equidistant approximation of X(t) by the explicit and implicit Euler      }
{ method with the time step size DELTA                                      }
{ Plots the linearly interpolated trajectories of the exact solution and    }
{ the explicit Euler approximation in the same graphic(problem 3.5.3) as    }
{ well as for the implicit approximation(problem 3.5.4), respectively       }
{ Written by Henri Schurz, 9.10. 1991                                       }
{ Note : Use the parameters ALPHA = 1.0 and 15.0 to illustrate the effects. }
{        If using time step sizes DELTA larger than 1/8 the plotting of the }
{        explicit approximation requires other scaling parameters.          }

PROGRAM PRX3X5X3; { Problem 3.5.3 and 3.5.4 }

USES CRT,DOS,GRAPH,INIT,SETSCR,SERVICE;

CONST
 NUM=16;           { number of time steps                                  }
 T0=0.0;           { left end point                                        }
 T=2.0;            { right end point                                       }
 DELTA=(T-T0)/NUM; { time step size of the approximations                  }
 X0=1.0;           { initial value                                         }
 ALPHA=15.0;       { parameter(>0) in the stochastic differential equation }
 ABSCMIN=T0;       { left end point                                        }
 ABSCMAX=T;        { right end point                                       }
 ORDMIN=0.0;       { minimum of the ordinate                               }
 ORDMAX=X0;        { maximum of the ordinate                               }
 ORDPOINT=X0/2;    { significant ordinate point                            }

TYPE
 VECTOR=ARRAY[0..NUMINV] OF REAL;

VAR
 CR:STRING;           { help string                                      }
 AXISX,AXISY:INTEGER; { location of the axes                             }
 DISTX,DISTY:INTEGER; { scale parameters                                 }
 I:INTEGER;           { time step                                        }
 DELTA_X:REAL;        { time step size of plotting exact solution        }
 XT:VECTOR;           { values of the exact solution                     }
 YT1:VECTOR;          { values of the explicit Euler approximation       }
 YT2:VECTOR;          { values of the implicit Euler approximation       }
 ABSCISSA0:VECTOR;    { values of the subinterval points(exact solution) }
 ABSCISSA1:VECTOR;    { values of the subinterval points(approximation)  }

{ Prepares the screen for the printout of the trajectories       }
{ X-axis is placed in the bottom; CY,CX ... strings for the axes }

PROCEDURE COORDSYS(CY,CX:STRING);
VAR
 FACTOR,I0,I1,I2,IH,K:INTEGER;
 DX,DY:REAL;
BEGIN
 IH:=TEXTHEIGHT('M')+10;
 AXISX:=TRUNC(MAXY-4*IH);
 I0:=TEXTWIDTH(CHCR(ORDMAX))+7;I1:=TEXTWIDTH(CHCR(ORDMIN))+7;
 I2:=TEXTWIDTH(CHCR(ORDPOINT))+7;IF I0<I1 THEN I0:=I1;
 IF I0<I2 THEN I0:=I2;I1:=TRUNC(MAXX/10);
 IF I0>MAXX-NUMINV-40 THEN I0:=MAXX-NUMINV-40;
 IF I0<I1 THEN AXISY:=I1 ELSE AXISY:=I0;
 DISTX:=NUMINV;DISTY:=AXISX-2*IH-20;
 SETTEXTSTYLE(DEFAULTFONT,HORIZDIR,1);SETTEXTJUSTIFY(1,1);
 OUTTEXTXY(AXISY,AXISX-DISTY-IH,'^');OUTTEXTXY(AXISY,AXISX-DISTY-2*IH,CY);
 LINE(AXISY,AXISX-DISTY-IH,AXISY,AXISX+5);
 LINE(AXISY-5,AXISX,AXISY+30+DISTX,AXISX);
 SETTEXTJUSTIFY(2,1);I0:=AXISX-DISTY;
 LINE(AXISY-3,I0,AXISY+3,I0);OUTTEXTXY(AXISY-7,I0,CHCR(ORDMAX));
 DY:=(ORDPOINT-ORDMIN)/(ORDMAX-ORDMIN);I0:=AXISX-TRUNC(DY*DISTY);
 LINE(AXISY-3,I0,AXISY+3,I0);OUTTEXTXY(AXISY-7,I0,CHCR(ORDPOINT));
 OUTTEXTXY(AXISY-7,AXISX,CHCR(ORDMIN));SETTEXTJUSTIFY(1,1);
 OUTTEXTXY(AXISY+30+DISTX,AXISX+1,'>');OUTTEXTXY(AXISY+30+DISTX,AXISX+10,CX);
 IF ((ABSCMAX-ABSCMIN=2.0) OR (ABSCMAX+ABSCMIN=0.0)) THEN FACTOR:=2 ELSE
  IF ABSCMAX-ABSCMIN=3.0 THEN FACTOR:=3 ELSE
   IF ABSCMAX-ABSCMIN=4.0 THEN FACTOR:=4 ELSE
    IF ((ABSCMAX-ABSCMIN=5.0) OR ((ABSCMAX-ABSCMIN=1.0) AND (ABSCMIN=0.0)))
     THEN FACTOR:=5 ELSE FACTOR:=1;
 DX:=(ABSCMAX-ABSCMIN)/FACTOR;
 FOR K:=0 TO FACTOR DO
  BEGIN
   I1:=AXISY+TRUNC(K*DISTX/FACTOR+0.5);I2:=AXISX+IH;
   LINE(I1,AXISX+3,I1,AXISX-3);
   OUTTEXTXY(I1,I2,CHCR(ABSCMIN+K*DX));
  END;
END;{ COORDSYS }

{ Plots the trajectory on the screen assuming COORDSYS was called }
{ before and x-axis is placed in the bottom of the screen         }
{ N0,NN = indices of the first and last data picked               }
{ LTN = line thickness                                            }

PROCEDURE PLOTGRAPH1(LTN,N0,NN:INTEGER;ORDINATE,ABSCISSA:VECTOR);
VAR
 IL,IR,IHL,IHR:INTEGER; { screen coordinates }
 I:INTEGER;             { data index         }
 FACTORX,FACTORY:REAL;  { scaling parameters }
BEGIN
 SETLINESTYLE(0,0,LTN);
 FACTORX:=DISTX/(ABSCMAX-ABSCMIN);
 FACTORY:=DISTY/(ORDMAX-ORDMIN);
 IR:=AXISY+TRUNC((ABSCISSA[N0]-ABSCMIN)*FACTORX);
 IHR:=AXISX-TRUNC((ORDINATE[N0]-ORDMIN)*FACTORY);
 I:=N0;
 REPEAT
  I:=I+1;
  IL:=IR;IHL:=IHR;
  IR:=AXISY+TRUNC((ABSCISSA[I]-ABSCMIN)*FACTORX);
  IHR:=AXISX-TRUNC((ORDINATE[I]-ORDMIN)*FACTORY);
  LINE(IL,IHL,IR,IHR);
 UNTIL I=NN;
 SETLINESTYLE(0,0,1);
END;{ PLOTGRAPH1 }

{ Main program : }

BEGIN

 INITIALIZE; { standard initialization }
 MAINWINDOW('Problem 3.5.3/4');
 STATUSLINE('Be patient! This will take the computer some time!');

{ Generation of the exact solution w.r.t. the highest screen resolution : }

 DELTA_X:=(T-T0)/NUMINV;
 I:=-1;
 REPEAT
  I:=I+1; { time step }
  ABSCISSA0[I]:=T0+I*DELTA_X; { value of the x-axis }

 { Exact solution : }

  XT[I]:=X0*EXP(-ALPHA*(ABSCISSA0[I]-T0));

 UNTIL I=NUMINV;

{ Generation of the approximations : }

 YT1[0]:=X0;YT2[0]:=X0; { initial values }
 ABSCISSA1[0]:=T0;
 I:=0;
 REPEAT
  I:=I+1; { time step }
  ABSCISSA1[I]:=ABSCISSA1[I-1]+DELTA; { value of the x-axis }

 { Explicit Euler scheme : }

  YT1[I]:=(1.0-ALPHA*DELTA)*YT1[I-1];

 { Implicit Euler scheme : }

  YT2[I]:=YT2[I-1]/(1.0+ALPHA*DELTA);

 UNTIL I=NUM;

{ Printout : }

 CLEARDEVICE;COORDSYS('X(t),Y(t)','t'); { draws the coordinate system      }
 PLOTGRAPH1(3,0,NUMINV,XT,ABSCISSA0); { plots the exact solution           }
 PLOTGRAPH1(1,0,NUM,YT1,ABSCISSA1); { plots the explicit Euler approximat. }
 CR:='Exact solution(thick) and explicit Euler approximation(thin)';
 CR:=CR+' using DELTA='+CHCR(DELTA);
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 CLEARDEVICE;COORDSYS('X(t),Y(t)','t'); { draws the coordinate system      }
 PLOTGRAPH1(3,0,NUMINV,XT,ABSCISSA0); { plots the exact solution           }
 PLOTGRAPH1(1,0,NUM,YT2,ABSCISSA1); { plots the implicit Euler approximat. }
 CR:='Exact solution(thick) and implicit Euler approximation(thin)';
 CR:=CR+' using DELTA='+CHCR(DELTA);
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 MYEXITPROC; { closes graphics mode and sets the old procedure address }
END.{ PRX3X5X3 }