{ Numerical stability for stochastic differential equations                }
{ Exact solution and Euler approximation                                   }
{ Considers the stochastic differential equation                           }
{               dX(t) = A(t,X(t)) dt + B(t,X(t)) dW(t)                     }
{           with  X(T0) = X0   on the time interval [T0,T]                 }
{ where W(t) is a Wiener process                                           }
{ Equidistant approximation of X(t) by the Euler scheme with the time step }
{ sizes DELTA                                                              }
{ Uses the Polar Marsaglia method to generate Gaussian random numbers      }
{ Plots the linearly interpolated trajectories of X(t) and the Euler ap-   }
{ proximation w.r.t. the highest possible resolution in the same graphic   }
{ Written by Henri Schurz, 9.10. 1991                                      }
{ Note : Be careful if changing the time step sizes. The maximum of the    }
{        numbers of the time steps used can not be larger than NUMINV.     }
{        Otherwise the corresponding selection of data has to be reorga-   }
{        nized in order to use AAGRAPHS-routines. The time step size DELTA }
{        can not be smaller than the DELTA_X used to evaluate the exact    }
{        solution.                                                         }

PROGRAM PRX3X5X5; { PC-Exercise 9.8.2 and 9.8.3 }

USES CRT,DOS,GRAPH,INIT,SETSCR,SERVICE,RANDNUMB,AAGRAPHS;

CONST
 T0=0.0;          { left end point                                        }
 T=1.0;           { right end point                                       }
 DELTA=(T-T0)/16; { time step size(>=1/NUMINV) of the Euler approximation }
 X0=1.0;          { initial value                                         }
 ALPHA=5.0;       { parameter(>> 0 for stiffness) in the function A(t,x)  }
 BETA=1.0;        { parameter in the function B(t,x)                      }

VAR
 CR:STRING;          { help string                                   }
 I,J:INTEGER;        { time steps                                    }
 APPROXSTEP:INTEGER; { ratio of the time step sizes                  }
 TI:REAL;            { subinterval point                             }
 YTOLD:REAL;         { last value of the approximation               }
 DELTA_X:REAL;       { time step size to plot the exact solution     }
 SQDELTA_X:REAL;     { square root of the time step size DELTA_X     }
 G1,G2:REAL;         { Gaussian random numbers                       }
 DWTI:REAL;          { Wiener process increment  W(t(i+1)) - W(ti)   }
 DWT:VECTOR;         { values of the Wiener process increments       }
 XT:VECTOR;          { values of the exact solution X(t)             }
 YT:VECTOR;          { values of the Euler approximation using DELTA }
 ABSCISSA:VECTOR;    { values of the subinterval points              }

{ Generates the drift function A(t,x) }

FUNCTION A(TI,XI:REAL):REAL;
BEGIN
 A:=ALPHA*XI;
END;{ A }

{ Generates the diffusion function B(t,x) }

FUNCTION B(TI,XI:REAL):REAL;
BEGIN
 B:=BETA;
END;{ B }

{ Generates the explicit solution X(TI) using X(TI-DELTA_X) }
{ Assumes i.i.d. Gaussian random numbers G1 and G2          }

FUNCTION EXPLSOL(G1,G2,XTOLD:REAL):REAL;
VAR
 EQ,Q,S22:REAL;
BEGIN
 Q:=ALPHA*DELTA_X; { integration by splitting and linear combination }
 EQ:=EXP(Q);
 S22:=EQ*EQ*(0.5/ALPHA-DELTA_X)+2.*EQ*(DELTA_X-1/ALPHA)+1.5/ALPHA;
 S22:=SQRT(ABS(S22));
 EXPLSOL:=EQ*XTOLD+BETA*(EQ*SQDELTA_X*G1+S22*G2); { approximate value }
END;{ EXPLSOL }

{ Main program : }

BEGIN

 INITIALIZE; { standard initialization }
 MAINWINDOW('Problem 3.5.5/6 (PC-Exercise 9.8.2/3)');
 STATUSLINE('Be patient! This will take the computer some time!');
 APPROXSTEP:=TRUNC(NUMINV*DELTA/(T-T0)+0.0000001); { ratio of step sizes }

{ Generation of the Wiener process increments DWT[I]            }
{ Evaluation of the exact solution and initialization of data : }

 DELTA_X:=(T-T0)/NUMINV;   { time step size for the exact solution }
 SQDELTA_X:=SQRT(DELTA_X); { square root of DELTA_X                }
 TI:=T0;                   { initial time                          }
 XT[0]:=X0;                { initial value of the exact solution   }
 ABSCISSA[0]:=T0;          { value of the x-axis                   }
 FOR I:=1 TO NUMINV DO
  BEGIN
   TI:=TI+DELTA_X;  { time                        }
   GENERATE(G1,G2); { uses Polar Marsaglia method }
   DWT[I-1]:=G1*SQDELTA_X; { Wiener process increment  W(t(i+1)) - W(ti) }
   XT[I]:=EXPLSOL(G1,G2,XT[I-1]); { value of the exact solution          }
   ABSCISSA[I]:=TI;        { value of the x-axis                         }
  END;

{ Generation of the Euler approximation : }

 I:=0;
 TI:=T0;    { initial time                       }
 YT[0]:=X0; { initial value of the approximation }
 WHILE TI<T DO
  BEGIN
   I:=I+1;  { time step }
   TI:=TI+DELTA; { time }
   YTOLD:=YT[(I-1)*APPROXSTEP]; { saves the old value }
   DWTI:=0.0; { calculates Wiener process increment for Euler time step }
   FOR J:=1 TO APPROXSTEP DO DWTI:=DWTI+DWT[(I-1)*APPROXSTEP+J-1];

  { Euler scheme : }

   YT[I*APPROXSTEP]:=YTOLD+A(TI-DELTA,YTOLD)*DELTA+B(TI-DELTA,YTOLD)*DWTI;

  { Interpolation for the other values : }

   IF APPROXSTEP>1 THEN
    FOR J:=1 TO APPROXSTEP-1 DO
     YT[(I-1)*APPROXSTEP+J]:=YTOLD+J*(YT[I*APPROXSTEP]-YTOLD)/APPROXSTEP;

  END;{ WHILE }

{ Printout : }

 CLEARDEVICE;
 GRAPH321(XT,YT,ABSCISSA,'X(t), Y(t)','t','X(t)','Y(t)'); { plots the paths }
 CR:='Exact solution X(t) and Euler approximation Y(t) using ';
 CR:=CR+'DELTA = '+CHCR(DELTA);
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 MYEXITPROC; { closes graphics mode and sets the old procedure address }
END.{ PRX3X5X5 }