{ Simulation studies with a nonlinear stochastic differential equation      }
{ Comparison of the efficiency between selected schemes : Euler, Milstein,  }
{ order 1.5 Taylor, explicit order 1.0 and explicit order 1.5 scheme which  }
{ are generally implemented (without any use of the particular structure of }
{ the given equation in the one-dimensional case at the time t = T          }
{ Considers the nonlinear stochastic differential equation                  }
{ dX(t) = -(sin(2*X(t)) + 0.25 sin(4*X(t)))dt + sqrt(2)*(cos(X(t)))^2 dW(t) }
{                           with  X(T0) = X0                                }
{ on the time interval [T0,T] where W(t) is a Wiener process                }
{ Equidistant approximation of X(t) with different time step sizes DELTA    }
{ Uses the Polar Marsaglia method to generate Gaussian random numbers       }
{ Estimation of the 90% confidence intervals for the absolute errors at the }
{ time T using M batches(M=20) with the sample size N { without printout!)  }
{ Plots the log2 of the absolute errors against the log2 of the step sizes  }
{ Plots the log2 of the elapsed time against the - log2 of these errors     }
{ Plots the log2 of the elapsed time against the log2 of the step sizes     }
{ Written by Henri Schurz, 9.10. 1991                                       }
{ Note : If changing the batch number remember that the corresponding per-  }
{        centage point of the t-distribution should also be changed.        }
{        For the measurement of the necessary time take the sample size     }
{        large enough. Confidence intervals can be plotted by calling the   }
{        routine CONFINV. See preparations at the end of this file.         }
{ Numerical schemes are generally implemented, so that this program can be  }
{ easily used for studies with other one-dimensional nonlinear equations!   }
{ For further studies, routines for order 2.0 Taylor scheme are included.   }

PROGRAM PRX4X4X6;

USES CRT,DOS,GRAPH,INIT,SETSCR,SERVICE,RANDNUMB,EXTGRAPH;

CONST
 NUM=9;         { number of different time step sizes                 }
 T0=0.0;        { left end point                                      }
 T=1.0;         { right end point                                     }
 DELTA=T-T0;    { largest time step size of the strong approximations }
 X0=1.0;        { initial value                                       }
 M=20;          { number of batches                                   }
 N=100;         { sample size of one batch                            }
 QUANTILE=1.73; { percentage point of the t-distribution              }

TYPE
 VECTOR0=ARRAY[1..M] OF REAL;

VAR
 CR:STRING;             { help string                                       }
 TIME:STRING;           { elapsed total computational time in seconds       }
 HOUR,MINUTE,SECOND,SEC100:WORD; { help time values                         }
 THOUR,TMINUTE,TSECOND,TSEC100:WORD; { help time values                     }
 G:INTEGER;             { index of the current time step size               }
 I:INTEGER;             { time step index                                   }
 J:INTEGER;             { batch index                                       }
 K:INTEGER;             { sample index within the current batch             }
 RTIME:REAL;            { elapsed time converted in a REAL type             }
 G0,G1,G2,G3:REAL;      { Gaussian independently distributed random numbers }
 Q11,Q21,Q22,Q31,Q32,Q33:REAL;{ coefficients for random number generation   }
 Q,SQRT2,SQRT3,YQ:REAL; { help values                                       }
 AAM,AAP,BBM,BBP,BPHIM,BPHIP,PSIM,PSIP:REAL;{ help values for explicit app. }
 TIOLD:REAL;            { left subinterval point                            }
 TI:REAL;               { right subinterval point                           }
 DELTA_Y:REAL;          { time step size of the strong approximation        }
 SQDELTA_Y:REAL;        { square root of the time step size DELTA_Y         }
 DWT:VECTOR;            { Wiener process increments  W(t(i+1)) - W(ti)      }
 DZT,DQT:VECTOR;        { random increments for generation of X(T)          }
 AA,DADX,BB,DBDX:REAL;  { function values for the Taylor schemes            }
 L0A11,L0B11,L1A11,L1B11,L1L1B11:REAL; { values of the functional operators }
 I1,I11,I10,I01,I00,I111:REAL; { multiple Ito integrals                     }
 XT:REAL;               { value of the exact solution X(t)                  }
 YT:REAL;               { value of the strong approximation using DELTA_Y   }
 AVERAGE,VARIANCE:REAL; { statistical parameters                            }
 EPSYLON:VECTOR0;       { sum of the absolute errors within a batch         }
 DEL:VECTOR1;           { time step sizes                                   }
 DIFFER:VECTOR1;        { half of the confidence interval lengths           }
 EPS:VECTOR1;           { absolute errors for different time step sizes     }
 INVEPS:VECTOR1;        { inverted errors EPS                               }
 TIMER:VECTOR1;         { time elapsed by the generation                    }

{ Generates the drift function A(t,x) }

FUNCTION A(TI,XI:REAL):REAL;
BEGIN
 A:=-SIN(2.*XI)-0.25*SIN(4.*XI);
END;{ A }

{ Generates the first partial derivatives of the drift function A(t,x) }

FUNCTION DAX(TI,XI:REAL):REAL; { x derivative }
BEGIN
 DAX:=-2.*COS(2.*XI)-COS(4.*XI);
END;{ DAX }

FUNCTION DAT(TI,XI:REAL):REAL; { t derivative }
BEGIN
 DAT:=0.0;
END;{ DAT }

{ Generates the second x partial derivative of the drift function A(t,x) }

FUNCTION DDAXX(TI,XI:REAL):REAL; { x derivative }
BEGIN
 DDAXX:=4.*(SIN(2.*XI)+SIN(4.*XI));
END;{ DDAXX }

{ Generates the diffusion function B(t,x) }

FUNCTION B(TI,XI:REAL):REAL;
BEGIN
 B:=SQRT2*COS(XI)*COS(XI);
END;{ B }

{ Generates the first partial derivatives of the diffusion function B(t,x) }

FUNCTION DBX(TI,XI:REAL):REAL; { x derivative }
BEGIN
 DBX:=-SQRT2*SIN(2.*XI);
END;{ DBX }

FUNCTION DBT(TI,XI:REAL):REAL; { t derivative }
BEGIN
 DBT:=0.0;
END;{ DBT }

{ Generates the second partial derivatives of the diffusion function B(t,x) }

FUNCTION DDBXX(TI,XI:REAL):REAL; { x derivative }
BEGIN
 DDBXX:=-2.*SQRT2*COS(2.*XI);
END;{ DDBXX }

{ Generates the functional form of the exact solution }

FUNCTION EXPLSOL(DWT,DZT,DQT:VECTOR):REAL;
VAR
 I:INTEGER;
 TI,V0,VT:REAL;
BEGIN
 V0:=SIN(X0)/COS(X0);VT:=0.0;I:=0;TI:=T0;
 REPEAT
  VT:=VT+EXP(TI)*(DWT[I]+DZT[I]+DQT[I]);
  I:=I+1;TI:=TI+DELTA_Y;
 UNTIL TI>=T;
 VT:=EXP(-T)*(V0+SQRT(2)*VT);
 EXPLSOL:=ARCTAN(VT); { exact solution at time T }
END;{ EXPLSOL }

{ Generates the error criterion formula }

FUNCTION ABSERR(XT,YT:REAL):REAL;
BEGIN
 ABSERR:=ABS(XT-YT); { absolute error }
END;{ ABSERR }

{ Computes the sample average and variance of a given data vector X }
{ with sample size NN                                               }

PROCEDURE COMPSAMPLEPARA(NN:INTEGER;X:VECTOR0;VAR SAVERAGE,SVARIANCE:REAL);
VAR
 J:INTEGER; { data index    }
 SQ:REAL;   { help variable }
BEGIN
 SAVERAGE:=0.0;SVARIANCE:=0.0;SQ:=0.0; { initialization }
 FOR J:=1 TO NN DO BEGIN SAVERAGE:=SAVERAGE+X[J];SQ:=SQ+X[J]*X[J]; END;
 SVARIANCE:=(SQ-SAVERAGE*SAVERAGE/NN)/(NN-1);
 SAVERAGE:=SAVERAGE/NN;
END;{ COMPSAMPLEPARA }

{ Takes the elapsed time in seconds for run times not exceeding one day }
{ Assumes GETTIME(HOUR,MINUTE,SECOND,SEC100) has been called before     }
{ Output time as a string TIME and as a real number RTIME               }

PROCEDURE TIMEINSEC(VAR TIME:STRING;VAR RTIME:REAL);
VAR
 ABSSEC,ABSSEC100,OLDHOUR,OLDMINUTE,OLDSEC100,OLDSECOND:WORD;
 CR:STRING;
BEGIN
 OLDHOUR:=HOUR;OLDMINUTE:=MINUTE;OLDSECOND:=SECOND;OLDSEC100:=SEC100;
 GETTIME(HOUR,MINUTE,SECOND,SEC100);
 IF SEC100<OLDSEC100 THEN
   BEGIN OLDSECOND:=OLDSECOND+1;ABSSEC100:=100-OLDSEC100+SEC100; END
  ELSE ABSSEC100:=SEC100-OLDSEC100;
 IF SECOND<OLDSECOND THEN
   BEGIN OLDMINUTE:=OLDMINUTE+1;ABSSEC:=60-OLDSECOND+SECOND; END
  ELSE ABSSEC:=SECOND-OLDSECOND;
 IF MINUTE<OLDMINUTE THEN
   BEGIN OLDHOUR:=OLDHOUR+1;ABSSEC:=ABSSEC+60*(60-OLDMINUTE+MINUTE); END
  ELSE ABSSEC:=ABSSEC+60*(MINUTE-OLDMINUTE);
 IF HOUR<OLDHOUR THEN ABSSEC:=ABSSEC+3600*(24-OLDHOUR+HOUR)
  ELSE ABSSEC:=ABSSEC+3600*(HOUR-OLDHOUR);
 STR(ABSSEC,CR);TIME:=CR+'.';STR(ABSSEC100,CR);
 IF ((LENGTH(CR)=1) AND (CR<>'0')) THEN CR:='0'+CR;
 TIME:=TIME+CR;
 RTIME:=ABSSEC+ABSSEC100/100;
END;{ TIMEINSEC }

{ Provides echo signals on the screen that parameters have been installed }
{ and schemes are just in progress or schemes have been done              }
{ Input NN determining the output line and scheme number                  }

PROCEDURE ECHO(NN:INTEGER;SCHEMETITLE:STRING);
VAR POS1,POS2,POS3,POS4,THM:INTEGER;
BEGIN
 THM:=TEXTHEIGHT('M');
 POS1:=TRUNC(MAXX/2);POS2:=3*(NN+5)*THM;POS3:=11*THM;POS4:=13*THM;
 CASE NN OF
  -4..-2   : OUTTEXTXY(POS1,POS2,SCHEMETITLE);
    -1     : OUTTEXTXY(POS1,POS2,SCHEMETITLE+' ... in progress');
   1..10   : BEGIN
              SETFILLSTYLE(0,0);BAR(1,POS3,MAXX-1,POS4);SETFILLSTYLE(0,1);
              OUTTEXTXY(POS1,POS2,SCHEMETITLE+' ... done');
             END;
   11,12   : OUTTEXTXY(POS1,POS2,SCHEMETITLE+' ... installed')
 END;
END;{ ECHO }

{ Main program : }

BEGIN

 INITIALIZE; { standard initialization }
 MAINWINDOW('Problem 4.4.6 (Efficiency with a nonlinear example)');
 STATUSLINE('Be patient! This will take the computer some time!');
 CR:='dX(t) = -(sin(2*X(t))+sin(4*X(t))/4) dt + sqrt(2)*cos(X(t))*cos(X(t)) dW(t)';
 ECHO(-4,CR); { delivers an echo signal to the screen equation implemented }
 CR:='<Parameters> : M='+CHCR(M)+'/N='+CHCR(N)+'/X(0)='+CHCR(X0);
 ECHO(11,CR); { delivers an echo signal to the screen on parameters used }
 ECHO(12,'Number of different time step sizes used : '+CHCR(NUM));
 SQRT2:=SQRT(2);SQRT3:=SQRT(3); { global constants }

{ Euler approximation : }

 ECHO(-1,'Euler'); { screen output scheme in progress }
 GETTIME(THOUR,TMINUTE,TSECOND,TSEC100); { gets the current time }
 DELTA_Y:=2.*DELTA;G:=0;
 REPEAT { for different time step sizes }
  GETTIME(HOUR,MINUTE,SECOND,SEC100); { gets the current time }
  G:=G+1;                   { index of the time step size used }
  DELTA_Y:=DELTA_Y/2;       { current time step size           }
  SQDELTA_Y:=SQRT(DELTA_Y); { square root of DELTA_Y           }
  Q11:=SQDELTA_Y;Q21:=SQDELTA_Y*DELTA_Y/2;Q22:=SQDELTA_Y*DELTA_Y*SQRT3/6;
  Q:=EXP(DELTA_Y)-1.0-DELTA_Y-DELTA_Y*DELTA_Y/2;Q31:=Q/SQDELTA_Y;
  Q32:=Q*(2.0-DELTA_Y)*SQRT3/(SQDELTA_Y*DELTA_Y)-SQDELTA_Y*DELTA_Y/SQRT3;
  Q:=(EXP(2*DELTA_Y)-1.0)/2-2.*DELTA_Y*EXP(DELTA_Y);
  Q33:=Q+DELTA_Y+DELTA_Y*DELTA_Y+DELTA_Y*DELTA_Y*DELTA_Y/3-Q31*Q31-Q32*Q32;
  Q33:=SQRT(Q33);

 { Generation for different batches : }

  J:=0;
  REPEAT
   J:=J+1;          { batch index                                  }
   EPSYLON[J]:=0.0; { sum of the absolute errors of the batch used }

  { Generation of different trajectories : }

   K:=0;
   REPEAT
    K:=K+1;  { index of the trajectory used          }

   { Generation of the Euler approximation and its absolute error : }

    I:=0;
    TI:=T0; { initial time                       }
    YT:=X0; { initial value of the approximation }
    WHILE TI<T DO
     BEGIN
      I:=I+1;         { time step index }
      TI:=TI+DELTA_Y; { current time    }
      IF I MOD 2 = 1 THEN { random number generation }
        BEGIN
         GENERATE(G1,G2);GENERATE(G3,G0); { uses Polar Marsaglia method }
        END
       ELSE
        BEGIN
         G1:=G0;GENERATE(G2,G3); { uses Polar Marsaglia method }
        END;
      DWT[I-1]:=Q11*G1; { Wiener process increment  W(t(i+1)) - W(ti) }
      DZT[I-1]:=Q21*G1+Q22*G2; { multiple integral I10 }
      DQT[I-1]:=Q31*G1+Q32*G2+Q33*G3; { random remainder term }

     { Euler scheme : }

      YT:=YT+A(TI-DELTA_Y,YT)*DELTA_Y+B(TI-DELTA_Y,YT)*DWT[I-1];

     END;{ WHILE }

   { Summation of the absolute errors : }

    XT:=EXPLSOL(DWT,DZT,DQT); { exact value of the solution at time T }
    EPSYLON[J]:=EPSYLON[J]+ABSERR(XT,YT);

   UNTIL K=N;{ REPEAT for different samples }
   EPSYLON[J]:=EPSYLON[J]/N; { estimate of the absolute error of the batch }
  UNTIL J=M;{ REPEAT for different batches }

{ Calculation of the confidence interval and initialization of data : }

  DEL[G]:=DELTA_Y; { current time step size }
  COMPSAMPLEPARA(M,EPSYLON,AVERAGE,VARIANCE);
  EPS[G]:=AVERAGE;           { midpoint of the confidence interval }
  DIFFER[G]:=QUANTILE*SQRT(VARIANCE/M); { half the interval length }
  INVEPS[G]:=1./AVERAGE; { inverted error EPS }
  TIMEINSEC(TIME,RTIME);TIMER[G]:=RTIME; { elapsed time for the step size }

 UNTIL G=NUM;{ REPEAT for different time step sizes }

 HOUR:=THOUR;MINUTE:=TMINUTE;SECOND:=TSECOND;SEC100:=TSEC100;
 TIMEINSEC(TIME,RTIME); { gets the total computational time elapsed }
 CR:='<1>  Euler in '+TIME+' seconds';
 ECHO(1,CR); { delivers an echo signal to the screen scheme ... done }

{ Milstein approximation : }

 ECHO(-1,'Milstein'); { screen output scheme in progress }
 GETTIME(THOUR,TMINUTE,TSECOND,TSEC100); { gets the current time }
 DELTA_Y:=2.*DELTA;G:=0;
 REPEAT { for different time step sizes }
  GETTIME(HOUR,MINUTE,SECOND,SEC100); { gets the current time }
  G:=G+1;                   { index of the time step size used }
  DELTA_Y:=DELTA_Y/2;       { current time step size           }
  SQDELTA_Y:=SQRT(DELTA_Y); { square root of DELTA_Y           }
  Q11:=SQDELTA_Y;Q21:=SQDELTA_Y*DELTA_Y/2;Q22:=SQDELTA_Y*DELTA_Y*SQRT3/6;
  Q:=EXP(DELTA_Y)-1.0-DELTA_Y-DELTA_Y*DELTA_Y/2;Q31:=Q/SQDELTA_Y;
  Q32:=Q*(2.0-DELTA_Y)*SQRT3/(SQDELTA_Y*DELTA_Y)-SQDELTA_Y*DELTA_Y/SQRT3;
  Q33:=(EXP(2*DELTA_Y)-1.0)/2-2.*DELTA_Y*EXP(DELTA_Y)+DELTA_Y+DELTA_Y*DELTA_Y;
  Q33:=SQRT(Q33+DELTA_Y*DELTA_Y*DELTA_Y/3-Q31*Q31-Q32*Q32);

 { Generation for different batches : }

  J:=0;
  REPEAT
   J:=J+1;          { batch index                                  }
   EPSYLON[J]:=0.0; { sum of the absolute errors of the batch used }

  { Generation of different trajectories : }

   K:=0;
   REPEAT
    K:=K+1;  { index of the trajectory used        }

   { Generation of the Milstein approximation and its absolute error : }

    I:=0;
    TI:=T0; { initial time                       }
    YT:=X0; { initial value of the approximation }
    WHILE TI<T DO
     BEGIN
      I:=I+1;         { time step index }
      TI:=TI+DELTA_Y; { current time    }
      IF I MOD 2 = 1 THEN { random number generation }
        BEGIN
         GENERATE(G1,G2);GENERATE(G3,G0); { uses Polar Marsaglia method }
        END
       ELSE
        BEGIN
         G1:=G0;GENERATE(G2,G3); { uses Polar Marsaglia method }
        END;
      DWT[I-1]:=Q11*G1; { Wiener process increment  W(t(i+1)) - W(ti) }
      DZT[I-1]:=Q21*G1+Q22*G2; { multiple integral I10 }
      DQT[I-1]:=Q31*G1+Q32*G2+Q33*G3; { random remainder term }

     { Milstein scheme(Ito version) : }

      I1:=DWT[I-1];I11:=0.5*(DWT[I-1]*DWT[I-1]-DELTA_Y);
      AA:=A(TI-DELTA_Y,YT);BB:=B(TI-DELTA_Y,YT);
      DBDX:=DBX(TI-DELTA_Y,YT);L1B11:=BB*DBDX;
      YT:=YT+AA*DELTA_Y       { drift part     }
            +BB*I1+L1B11*I11; { diffusion part }

     END;{ WHILE }

   { Summation of the absolute errors : }

    XT:=EXPLSOL(DWT,DZT,DQT); { exact value of the solution at time T }
    EPSYLON[J]:=EPSYLON[J]+ABSERR(XT,YT);

   UNTIL K=N;{ REPEAT for different samples }
   EPSYLON[J]:=EPSYLON[J]/N; { estimate of the absolute error of the batch }
  UNTIL J=M;{ REPEAT for different batches }

{ Calculation of the confidence interval and initialization of data : }

  DEL[G+NUM]:=DELTA_Y; { current time step size }
  COMPSAMPLEPARA(M,EPSYLON,AVERAGE,VARIANCE);
  EPS[G+NUM]:=AVERAGE;           { midpoint of the confidence interval }
  DIFFER[G+NUM]:=QUANTILE*SQRT(VARIANCE/M); { half the interval length }
  INVEPS[G+NUM]:=1./AVERAGE; { inverted error EPS }
  TIMEINSEC(TIME,RTIME);
  TIMER[G+NUM]:=RTIME; { elapsed time for the step size }

 UNTIL G=NUM;{ REPEAT for different time step sizes }

 HOUR:=THOUR;MINUTE:=TMINUTE;SECOND:=TSECOND;SEC100:=TSEC100;
 TIMEINSEC(TIME,RTIME); { gets the total computational time elapsed }
 CR:='<2>  Milstein in '+TIME+' seconds';
 ECHO(2,CR); { delivers an echo signal to the screen scheme ... done }

{ Explicit order 1.0 approximation proposed by Platen : }

 ECHO(-1,'Explicit 1.0'); { screen output scheme in progress }
 GETTIME(THOUR,TMINUTE,TSECOND,TSEC100); { gets the current time }
 DELTA_Y:=2.*DELTA;G:=0;
 REPEAT { for different time step sizes }
  GETTIME(HOUR,MINUTE,SECOND,SEC100); { gets the current time }
  G:=G+1;                   { index of the time step size used }
  DELTA_Y:=DELTA_Y/2;       { current time step size           }
  SQDELTA_Y:=SQRT(DELTA_Y); { square root of DELTA_Y           }
  Q11:=SQDELTA_Y;Q21:=SQDELTA_Y*DELTA_Y/2;Q22:=SQDELTA_Y*DELTA_Y*SQRT3/6;
  Q:=EXP(DELTA_Y)-1.0-DELTA_Y-DELTA_Y*DELTA_Y/2;Q31:=Q/SQDELTA_Y;
  Q32:=Q*(2.0-DELTA_Y)*SQRT3/(SQDELTA_Y*DELTA_Y)-SQDELTA_Y*DELTA_Y/SQRT3;
  Q33:=(EXP(2*DELTA_Y)-1.0)/2-2.*DELTA_Y*EXP(DELTA_Y)+DELTA_Y+DELTA_Y*DELTA_Y;
  Q33:=SQRT(Q33+DELTA_Y*DELTA_Y*DELTA_Y/3-Q31*Q31-Q32*Q32);

 { Generation for different batches : }

  J:=0;
  REPEAT
   J:=J+1;          { batch index                                  }
   EPSYLON[J]:=0.0; { sum of the absolute errors of the batch used }

  { Generation of different trajectories : }

   K:=0;
   REPEAT
    K:=K+1;  { index of the trajectory used        }

   { Generation of the explicit approximation and its absolute error : }

    I:=0;
    TI:=T0; { initial time                       }
    YT:=X0; { initial value of the approximation }
    WHILE TI<T DO
     BEGIN
      I:=I+1;         { time step index }
      TI:=TI+DELTA_Y; { current time    }
      IF I MOD 2 = 1 THEN { random number generation }
        BEGIN
         GENERATE(G1,G2);GENERATE(G3,G0); { uses Polar Marsaglia method }
        END
       ELSE
        BEGIN
         G1:=G0;GENERATE(G2,G3); { uses Polar Marsaglia method }
        END;
      DWT[I-1]:=Q11*G1; { Wiener process increment  W(t(i+1)) - W(ti) }
      DZT[I-1]:=Q21*G1+Q22*G2; { multiple integral I10 }
      DQT[I-1]:=Q31*G1+Q32*G2+Q33*G3; { random remainder term }

     { Explicit order 1.0 scheme(Ito version, proposed by Platen) : }

      TIOLD:=TI-DELTA_Y;
      I1:=DWT[I-1];I11:=0.5*(DWT[I-1]*DWT[I-1]-DELTA_Y);
      AA:=A(TIOLD,YT)*DELTA_Y;BB:=B(TIOLD,YT);YQ:=YT+AA+BB*SQDELTA_Y;
      YT:=YT+AA+BB*I1+(B(TIOLD,YQ)-BB)*I11/SQDELTA_Y;

     END;{ WHILE }

   { Summation of the absolute errors : }

    XT:=EXPLSOL(DWT,DZT,DQT); { exact value of the solution at time T }
    EPSYLON[J]:=EPSYLON[J]+ABSERR(XT,YT);

   UNTIL K=N;{ REPEAT for different samples }
   EPSYLON[J]:=EPSYLON[J]/N; { estimate of the absolute error of the batch }
  UNTIL J=M;{ REPEAT for different batches }

{ Calculation of the confidence interval and initialization of data : }

  DEL[G+2*NUM]:=DELTA_Y; { current time step size }
  COMPSAMPLEPARA(M,EPSYLON,AVERAGE,VARIANCE);
  EPS[G+2*NUM]:=AVERAGE;           { midpoint of the confidence interval }
  DIFFER[G+2*NUM]:=QUANTILE*SQRT(VARIANCE/M); { half the interval length }
  INVEPS[G+2*NUM]:=1./AVERAGE; { inverted error EPS }
  TIMEINSEC(TIME,RTIME);
  TIMER[G+2*NUM]:=RTIME; { elapsed time for the step size }

 UNTIL G=NUM;{ REPEAT for different time step sizes }

 HOUR:=THOUR;MINUTE:=TMINUTE;SECOND:=TSECOND;SEC100:=TSEC100;
 TIMEINSEC(TIME,RTIME); { gets the total computational time elapsed }
 CR:='<3>  Explicit order 1.0 in '+TIME+' seconds';
 ECHO(3,CR); { delivers an echo signal to the screen scheme ... done }

{ The order 1.5 Taylor approximation : }

 ECHO(-1,'Taylor 1.5'); { screen output scheme in progress }
 GETTIME(THOUR,TMINUTE,TSECOND,TSEC100); { gets the current time }
 DELTA_Y:=2.*DELTA;G:=0;
 REPEAT { for different time step sizes }
  GETTIME(HOUR,MINUTE,SECOND,SEC100); { gets the current time }
  G:=G+1;                   { index of the time step size used }
  DELTA_Y:=DELTA_Y/2;       { current time step size           }
  SQDELTA_Y:=SQRT(DELTA_Y); { square root of DELTA_Y           }
  Q11:=SQDELTA_Y;Q21:=SQDELTA_Y*DELTA_Y/2;Q22:=SQDELTA_Y*DELTA_Y*SQRT3/6;
  Q:=EXP(DELTA_Y)-1.0-DELTA_Y-DELTA_Y*DELTA_Y/2;Q31:=Q/SQDELTA_Y;
  Q32:=Q*(2.0-DELTA_Y)*SQRT3/(SQDELTA_Y*DELTA_Y)-SQDELTA_Y*DELTA_Y/SQRT3;
  Q33:=(EXP(2*DELTA_Y)-1.0)/2-2.*DELTA_Y*EXP(DELTA_Y)+DELTA_Y+DELTA_Y*DELTA_Y;
  Q33:=SQRT(Q33+DELTA_Y*DELTA_Y*DELTA_Y/3-Q31*Q31-Q32*Q32);

 { Generation for different batches : }

  J:=0;
  REPEAT
   J:=J+1;          { batch index                                  }
   EPSYLON[J]:=0.0; { sum of the absolute errors of the batch used }

  { Generation of different trajectories : }

   K:=0;
   REPEAT
    K:=K+1;  { index of the trajectory used        }

   { Generation of the Taylor approximation and its absolute error : }

    I:=0;
    TI:=T0; { initial time                       }
    YT:=X0; { initial value of the approximation }
    WHILE TI<T DO
     BEGIN
      I:=I+1;             { time step index                             }
      TI:=TI+DELTA_Y;     { current time                                }
      IF I MOD 2 = 1 THEN { random number generation }
        BEGIN
         GENERATE(G1,G2);GENERATE(G3,G0); { uses Polar Marsaglia method }
        END
       ELSE
        BEGIN
         G1:=G0;GENERATE(G2,G3); { uses Polar Marsaglia method }
        END;
      DWT[I-1]:=Q11*G1; { Wiener process increment  W(t(i+1)) - W(ti) }
      DZT[I-1]:=Q21*G1+Q22*G2; { multiple integral I10 }
      DQT[I-1]:=Q31*G1+Q32*G2+Q33*G3; { random remainder term }

     { Generation of the multiple Ito integrals : }

      I1:=DWT[I-1];I11:=0.5*(DWT[I-1]*DWT[I-1]-DELTA_Y);
      I00:=0.5*DELTA_Y*DELTA_Y;I10:=DZT[I-1];I01:=DWT[I-1]*DELTA_Y-I10;
      I111:=0.5*(DWT[I-1]*DWT[I-1]/3-DELTA_Y)*DWT[I-1];

     { 1.5 order strong Taylor scheme(Ito version) : }

      TIOLD:=TI-DELTA_Y;
      AA:=A(TIOLD,YT);DADX:=DAX(TIOLD,YT);
      BB:=B(TIOLD,YT);DBDX:=DBX(TIOLD,YT);
      {L0A11:=DAT(TIOLD,YT)+AA*DADX+0.5*BB*BB*DDAXX(TIOLD,YT); nonautonomous }
      L0A11:=AA*DADX+0.5*BB*BB*DDAXX(TIOLD,YT); { autonomous case }
      {L0B11:=DBT(TIOLD,YT)+AA*DBDX+0.5*BB*BB*DDBXX(TIOLD,YT); nonautonomous }
      L0B11:=AA*DBDX+0.5*BB*BB*DDBXX(TIOLD,YT); { autonomous case }
      L1A11:=BB*DADX;L1B11:=BB*DBDX;
      L1L1B11:=BB*(DBDX*DBDX+BB*DDBXX(TIOLD,YT));
      YT:=YT+AA*DELTA_Y+BB*I1+L1B11*I11+L1A11*I10+L0B11*I01+L0A11*I00
            +L1L1B11*I111;

     END;{ WHILE }

   { Summation of the absolute errors : }

    XT:=EXPLSOL(DWT,DZT,DQT); { exact value of the solution at time T }
    EPSYLON[J]:=EPSYLON[J]+ABSERR(XT,YT);

   UNTIL K=N;{ REPEAT for different samples }
   EPSYLON[J]:=EPSYLON[J]/N; { estimate of the absolute error of the batch }
  UNTIL J=M;{ REPEAT for different batches }

{ Calculation of the confidence interval and initialization of data : }

  DEL[G+3*NUM]:=DELTA_Y; { current time step size }
  COMPSAMPLEPARA(M,EPSYLON,AVERAGE,VARIANCE);
  EPS[G+3*NUM]:=AVERAGE;           { midpoint of the confidence interval }
  DIFFER[G+3*NUM]:=QUANTILE*SQRT(VARIANCE/M); { half the interval length }
  INVEPS[G+3*NUM]:=1./AVERAGE; { inverted error EPS }
  TIMEINSEC(TIME,RTIME);
  TIMER[G+3*NUM]:=RTIME; { elapsed time for the step size }

 UNTIL G=NUM;{ REPEAT for different time step sizes }

 HOUR:=THOUR;MINUTE:=TMINUTE;SECOND:=TSECOND;SEC100:=TSEC100;
 TIMEINSEC(TIME,RTIME); { gets the total computational time elapsed }
 CR:='<4>  Taylor 1.5 in '+TIME+' seconds';
 ECHO(4,CR); { delivers an echo signal to the screen scheme ... done }

{ Explicit order 1.5 approximation(autonomous case) : }

 ECHO(-1,'Explicit 1.5'); { screen output scheme in progress }
 GETTIME(THOUR,TMINUTE,TSECOND,TSEC100); { gets the current time }
 DELTA_Y:=2.*DELTA;G:=0;
 REPEAT { for different time step sizes }
  GETTIME(HOUR,MINUTE,SECOND,SEC100); { gets the current time }
  G:=G+1;                   { index of the time step size used }
  DELTA_Y:=DELTA_Y/2;       { current time step size           }
  SQDELTA_Y:=SQRT(DELTA_Y); { square root of DELTA_Y           }
  Q11:=SQDELTA_Y;Q21:=SQDELTA_Y*DELTA_Y/2;Q22:=SQDELTA_Y*DELTA_Y*SQRT3/6;
  Q:=EXP(DELTA_Y)-1.0-DELTA_Y-DELTA_Y*DELTA_Y/2;Q31:=Q/SQDELTA_Y;
  Q32:=Q*(2.0-DELTA_Y)*SQRT3/(SQDELTA_Y*DELTA_Y)-SQDELTA_Y*DELTA_Y/SQRT3;
  Q33:=(EXP(2*DELTA_Y)-1.0)/2-2.*DELTA_Y*EXP(DELTA_Y)+DELTA_Y+DELTA_Y*DELTA_Y;
  Q33:=SQRT(Q33+DELTA_Y*DELTA_Y*DELTA_Y/3-Q31*Q31-Q32*Q32);

 { Generation for different batches : }

  J:=0;
  REPEAT
   J:=J+1;          { batch index                                  }
   EPSYLON[J]:=0.0; { sum of the absolute errors of the batch used }

  { Generation of different trajectories : }

   K:=0;
   REPEAT
    K:=K+1;  { index of the trajectory used        }

   { Generation of the explicit approximation and its absolute error : }

    I:=0;
    TI:=T0; { initial time                       }
    YT:=X0; { initial value of the approximation }
    WHILE TI<T DO
     BEGIN
      I:=I+1;             { time step index                             }
      TI:=TI+DELTA_Y;     { current time                                }
      IF I MOD 2 = 1 THEN { random number generation }
        BEGIN
         GENERATE(G1,G2);GENERATE(G3,G0); { uses Polar Marsaglia method }
        END
       ELSE
        BEGIN
         G1:=G0;GENERATE(G2,G3); { uses Polar Marsaglia method }
        END;
      DWT[I-1]:=Q11*G1; { Wiener process increment  W(t(i+1)) - W(ti) }
      DZT[I-1]:=Q21*G1+Q22*G2; { multiple integral I10 }
      DQT[I-1]:=Q31*G1+Q32*G2+Q33*G3; { random remainder term }

     { Generation of the multiple Ito integrals : }

      I1:=DWT[I-1];I11:=0.5*(DWT[I-1]*DWT[I-1]-DELTA_Y);
      I00:=0.5*DELTA_Y*DELTA_Y;I10:=DZT[I-1];I01:=DWT[I-1]*DELTA_Y-I10;
      I111:=0.5*(DWT[I-1]*DWT[I-1]/3-DELTA_Y)*DWT[I-1];

     { Explicit order 1.5 scheme(Ito version, valid for autonomous case) : }

      TIOLD:=TI-DELTA_Y;
      AA:=A(TIOLD,YT);BB:=B(TIOLD,YT);PSIP:=YT+AA*DELTA_Y+BB*SQDELTA_Y;
      PSIM:=YT+AA*DELTA_Y-BB*SQDELTA_Y;BBP:=B(TIOLD,PSIP);BBM:=B(TIOLD,PSIM);
      AAP:=A(TIOLD,PSIP);AAM:=A(TIOLD,PSIM);
      BPHIP:=B(TIOLD,PSIP+BBP*SQDELTA_Y);BPHIM:=B(TIOLD,PSIP-BBP*SQDELTA_Y);
      YT:=YT+AA*DELTA_Y+BB*I1+0.5*(BBP-BBM)*I11/SQDELTA_Y;
      YT:=YT+0.5*(AAP-AAM)*I10/SQDELTA_Y+0.5*(BBP-2.*BB+BBM)*I01/DELTA_Y;
      YT:=YT+0.5*(AAP-2.*AA+AAM)*I00/DELTA_Y
            +0.5*(BPHIP-BPHIM-BBP+BBM)*I111/DELTA_Y;

     END;{ WHILE }

   { Summation of the absolute errors : }

    XT:=EXPLSOL(DWT,DZT,DQT); { exact value of the solution at time T }
    EPSYLON[J]:=EPSYLON[J]+ABSERR(XT,YT);

   UNTIL K=N;{ REPEAT for different samples }
   EPSYLON[J]:=EPSYLON[J]/N; { estimate of the absolute error of the batch }
  UNTIL J=M;{ REPEAT for different batches }

{ Calculation of the confidence interval and initialization of data : }

  DEL[G+4*NUM]:=DELTA_Y; { current time step size }
  COMPSAMPLEPARA(M,EPSYLON,AVERAGE,VARIANCE);
  EPS[G+4*NUM]:=AVERAGE;           { midpoint of the confidence interval }
  DIFFER[G+4*NUM]:=QUANTILE*SQRT(VARIANCE/M); { half the interval length }
  INVEPS[G+4*NUM]:=1./AVERAGE; { inverted error EPS }
  TIMEINSEC(TIME,RTIME);
  TIMER[G+4*NUM]:=RTIME; { elapsed time for the step size }

 UNTIL G=NUM;{ REPEAT for different time step sizes }

 HOUR:=THOUR;MINUTE:=TMINUTE;SECOND:=TSECOND;SEC100:=TSEC100;
 TIMEINSEC(TIME,RTIME); { gets the total computational time elapsed }
 CR:='<5>  Explicit order 1.5 in '+TIME+' seconds';
 ECHO(5,CR); { delivers an echo signal to the screen scheme ... done }
 STATUSLINE('Ready. Press any key except <ESC> !');
 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

{ Printout of the error graphs in the logarithmic graphic table : }

 CLEARDEVICE;
 GRAPH441(5,NUM,EPS,DEL,'log2(EPS)','log2(DELTA)');
 CR:='Linearly interpolated (log2(EPS),log2(DELTA)) - graph at t=';
 CR:=CR+CHCR(T)+'<Taylor,Der.free>';
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

{ Printout of the logarithmic graphic table for the efficiency comparison : }

 CLEARDEVICE;
 GRAPH441(5,NUM,TIMER,INVEPS,'log2(TIME)','-log2(EPS)');
 CR:='Linearly interpolated (-log2(EPS),log2(TIME)) - graph at t=';
 CR:=CR+CHCR(T)+'<Taylor,Der.free>';
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

{ Printout of the time consumption in the logarithmic graphic table : }

 CLEARDEVICE;
 GRAPH441(5,NUM,TIMER,DEL,'log2(TIME)','log2(DELTA)');
 CR:='Linearly interpolated (log2(TIME),log2(DELTA)) - graph at t=';
 CR:=CR+CHCR(T)+'<Taylor,Der.free>';
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 MYEXITPROC; { closes graphics mode and sets the old procedure address }
END.{ PRX4X4X6 }

{ If desired
    choose the corresponding confidence intervals by EPS, DIFFER and DEL
   and add the following :

  Printout of the confidence intervals :

 CLEARDEVICE;
 CONFINV(NUM,EPS,DIFFER,DEL,'EPS','DELTA');
 CR:='90% Confidence intervals for the absolute error EPS(t=';
 CR:=CR+CHCR(T)+')'+'<Taylor,Der.free>';
 STATUSLINE(CR);
}
{---- To reduce computational time ----}

{ Generates help values for the scheme evaluation                       }
{ Reduces the number of frequently appearing expressions(function calls }
{ Variable SCHEME controls the number of the necessary help values.     }

PROCEDURE HELPSCHEME(SCHEME:INTEGER;XI:REAL);
BEGIN
 SIN2XI:=SIN(2.*XI);SIN4XI:=SIN(4.*XI);COSXI2:=COS(XI)*COS(XI);
 IF SCHEME>1 THEN COS2XI:=COS(2.*XI);
END; { HELPSCHEME }

{---- Preparation for order 2.0 Taylor approximation ----}

 DDADXX,DDBDXX:REAL;    { function values for the Taylor schemes            }
 DDBDXT,DDDBDXXX:REAL;  { function values for the order 2.0 Taylor scheme   }
 CORRDRIFT:REAL;        { value of the corresponding Stratonovich drift     }
 L1L1A11,L1L0B11,L0L1B11,L1L1L1B11:REAL;{ values of the functional operators}
 DL1B11DX:REAL;         { help variable(derivative of the operator L1B11)   }
 J1,J11,J00,J111,J1111:REAL; { multiple Stratonovich integrals              }
 J10,J01,J110,J101,J011:REAL;{ multiple Stratonovich integrals approximated }
...
FUNCTION DDBXT(TI,XI:REAL):REAL; { derivative w.r.t. x and t }
BEGIN
 DDBXT:=0.0;
END;{ DDBXT }

FUNCTION DDBTX(TI,XI:REAL):REAL; { derivative w.r.t. t and x }
BEGIN
 DDBTX:=0.0;
END;{ DDBTX }

{ Generates the third x partial derivative of the diffusion function B(t,x) }

FUNCTION DDDBXXX(TI,XI:REAL):REAL; { x derivative }
BEGIN
 DDDBXXX:=4.*SQRT2*SIN(2.*XI);
END;{ DDDBXXX }
...
{ The order 2.0 Taylor approximation generally implemented : }

 ECHO(-1,'Taylor 2.0'); { screen output scheme in progress }
 GETTIME(THOUR,TMINUTE,TSECOND,TSEC100); { gets the current time }
 SETUPPARAMULTIINT(P,ALPHAP,ROP); { sets the parameters for MULTIINT }
 DELTA_Y:=2.*DELTA;G:=0;
 REPEAT { for different time step sizes }
  GETTIME(HOUR,MINUTE,SECOND,SEC100); { gets the current time }
  G:=G+1;                   { index of the time step size used }
  DELTA_Y:=DELTA_Y/2;       { current time step size           }
  SQDELTA_Y:=SQRT(DELTA_Y); { square root of DELTA_Y           }
  Q11:=SQDELTA_Y;Q21:=SQDELTA_Y*DELTA_Y/2;Q22:=SQDELTA_Y*DELTA_Y*SQRT3/6;
  Q:=EXP(DELTA_Y)-1.0-DELTA_Y-DELTA_Y*DELTA_Y/2;Q31:=Q/SQDELTA_Y;
  Q32:=Q*(2.0-DELTA_Y)*SQRT3/(SQDELTA_Y*DELTA_Y)-SQDELTA_Y*DELTA_Y/SQRT3;
  Q33:=(EXP(2*DELTA_Y)-1.0)/2-2.*DELTA_Y*EXP(DELTA_Y)+DELTA_Y+DELTA_Y*DELTA_Y;
  Q33:=SQRT(Q33+DELTA_Y*DELTA_Y*DELTA_Y/3-Q31*Q31-Q32*Q32);

 { Generation for different batches : }

  J:=0;
  REPEAT
   J:=J+1;          { batch index                                  }
   EPSYLON[J]:=0.0; { sum of the absolute errors of the batch used }

  { Generation of different trajectories : }

   K:=0;
   REPEAT
    K:=K+1;  { index of the trajectory used        }
    WT:=0.0; { value of the Wiener process at t=T0 }

   { Generation of the Taylor approximation and its absolute error : }

    I:=0;
    TI:=T0; { initial time                       }
    YT:=X0; { initial value of the approximation }
    WHILE TI<T DO
     BEGIN
      I:=I+1;         { time step index }
      TI:=TI+DELTA_Y; { current time    }
      IF I MOD 2 = 1 THEN GENERATE(G1,G2) { uses the Polar Marsaglia method }
       ELSE G1:=G2;
      DWT[I]:=G1*SQDELTA_Y; { Wiener process increment  W(t(i+1)) - W(ti) }
      HELPSCHEME(4,YT);

     { Generation of the multiple integrals : }

      MULTIINT(P,G1,J1,J01,J10,J11,J011,J101,J110);DZT[I]:=J10;
      J00:=0.5*DELTA_Y*DELTA_Y;J111:=J11*DWT[I]/3;J1111:=J111*DWT[I]/4;

     { 2.0 order strong Taylor scheme(Stratonovich version) : }

      TIOLD:=TI-DELTA_Y;
      AA:=A(TIOLD,YT);DADX:=DAX(TIOLD,YT);DDADXX:=DDAXX(TIOLD,YT);
      BB:=B(TIOLD,YT);DBDX:=DBX(TIOLD,YT);DDBDXX:=DDBXX(TIOLD,YT);
      DDBDXT:=DDBXT(TIOLD,YT);DDDBDXXX:=DDDBXXX(TIOLD,YT);
      L1B11:=BB*DBDX;CORRDRIFT:=AA-0.5*L1B11;DL1B11DX:=DBDX*DBDX+BB*DDBDXX;
      L0A11:=DAT(TIOLD,YT)-0.5*(DBT(TIOLD,YT)*DBDX+BB*DDBDXT);
      L0A11:=L0A11+CORRDRIFT*(DADX-0.5*DL1B11DX);
      L1A11:=BB*(DADX-0.5*DL1B11DX);
      L1L1A11:=DBDX*L1A11+BB*BB*(DDADXX-0.5*(3.*DBDX*DDBDXX+BB*DDDBDXXX));
      L0B11:=DBT(TIOLD,YT)+CORRDRIFT*DBDX;
      L0L1B11:=DBT(TIOLD,YT)*DBDX+BB*DDBDXT+CORRDRIFT*DL1B11DX;
      L1L0B11:=BB*(DDBTX(TIOLD,YT)+(DADX-0.5*DL1B11DX)*DBDX+CORRDRIFT*DDBDXX);
      L1L1B11:=BB*DL1B11DX;
      L1L1L1B11:=DBDX*L1L1B11+BB*BB*(3.*DBDX*DDBDXX+BB*DDDBDXXX);
      YT:=YT+CORRDRIFT*DELTA_Y+BB*J1+L1B11*J11+L1A11*J10+L0B11*J01;
      YT:=YT+L0A11*J00+L1L1B11*J111+L1L1L1B11*J1111;
      YT:=YT+L0L1B11*J011+L1L0B11*J101+L1L1A11*J110;

     END;{ WHILE }

   { Summation of the absolute errors : }

    XT:=EXPLSOL(DWT,DZT,DQT); { exact value of the solution }
    EPSYLON[J]:=EPSYLON[J]+ABSERR(XT,YT);

   UNTIL K=N;{ REPEAT for different samples }
   EPSYLON[J]:=EPSYLON[J]/N; { estimate of the absolute error of the batch }
  UNTIL J=M;{ REPEAT for different batches }

{ Calculation of the confidence interval and initialization of data : }

  DEL[G+3*NUM]:=DELTA_Y; { current time step size }
  COMPSAMPLEPARA(M,EPSYLON,AVERAGE,VARIANCE);
  EPS[G+3*NUM]:=AVERAGE;           { midpoint of the confidence interval }
  DIFFER[G+3*NUM]:=QUANTILE*SQRT(VARIANCE/M); { half the interval length }
  INVEPS[G+3*NUM]:=1./AVERAGE; { inverted error EPS }
  TIMEINSEC(TIME,RTIME);
  TIMER[G+3*NUM]:=RTIME; { elapsed time for the step size }

 UNTIL G=NUM;{ REPEAT for different time step sizes }

 HOUR:=THOUR;MINUTE:=TMINUTE;SECOND:=TSECOND;SEC100:=TSEC100;
 TIMEINSEC(TIME,RTIME); { gets the total computational time elapsed }
 CR:='<4>  Taylor 2.0 in '+TIME+' seconds';
 ECHO(4,CR); { delivers an echo signal to the screen scheme ... done }

{---- Preparation for multiple integrals generation which is necessary ----}
{----            for order 2.0 strong Taylor approximation             ----}

 P=5;           { truncation parameter for the multiple integrals     }
...
 ALPHAP,ROP:REAL;       { approximation parameter of the multiple integrals }
...
{ Sets the parameters for the procedure MULTIINT }

PROCEDURE SETUPPARAMULTIINT(P:INTEGER;VAR ALPHAP,ROP:REAL);
VAR
 R:INTEGER;
 INCRE:REAL;
BEGIN
 ALPHAP:=0.0;ROP:=0.0;
 FOR R:=1 TO P DO
  BEGIN
   INCRE:=1/(R*R);ROP:=ROP+INCRE;ALPHAP:=ALPHAP+INCRE*INCRE;
  END;
 ALPHAP:=(PI*PI*PI*PI/90-ALPHAP)/(2*PI*PI);
 ROP:=PI*PI/6-ROP;ROP:=ROP/(2*PI*PI);
END;{ SETUPPARAMULTIINT }

{ Generates the multiple Stratonovich integrals                         }
{ J1, J01, J10, J11, J011, J101, and J110                               }
{ Uses the time step size DELTA_Y; P = accuracy parameter               }
{ Assumes the Gaussian random number DWT has already been generated and }
{ SETUPPARAMULTIINT has been called before                              }

PROCEDURE MULTIINT(P:INTEGER;DWT:REAL;VAR J1,J01,J10,J11,J011,J101,J110:REAL);
VAR
 R,L:INTEGER;
 A10,B11P,B1P,C11P,FI1P,GSI,MUE1P:REAL;
 ETA1,FI1:VECTOR;
BEGIN
 GENERATE(FI1P,MUE1P);
 A10:=0.0;B1P:=0.0;B11P:=0.0;
 FOR R:=1 TO P DO
  BEGIN
   GENERATE(FI1[R],ETA1[R]);
   A10:=A10+FI1[R]/R;
   B1P:=B1P+ETA1[R]/(R*R);
   B11P:=B11P+(FI1[R]*FI1[R]+ETA1[R]*ETA1[R])/(R*R);
  END;
 A10:=-(1./PI)*A10*SQRT(2.0)*SQDELTA_Y;
 A10:=A10-2.*SQDELTA_Y*SQRT(ROP)*MUE1P;
 B1P:=B1P*SQDELTA_Y/SQRT(2.0)+SQDELTA_Y*SQRT(ALPHAP)*FI1P;
 B11P:=B11P/(4.*PI*PI);
 GSI:=DWT;
 J1:=GSI*SQDELTA_Y; { increment of the Wiener trajectory = J1 }
 J10:=0.5*DELTA_Y*(J1+A10); { multiple integral J10 }
 J01:=J1*DELTA_Y-J10; { multiple integral J01 }
 J11:=0.5*J1*J1; { multiple integral J11 }
 C11P:=0.0;
 R:=0;
 REPEAT
  R:=R+1;
  L:=0;
  REPEAT
   L:=L+1;
   IF L<>R THEN
    C11P:=C11P+(R/(R*R-L*L))*(FI1[R]*FI1[L]/L-ETA1[R]*ETA1[L]*L/R);
  UNTIL L=P;
 UNTIL R=P;
 C11P:=-C11P/(2.*PI*PI);
 J101:=SQR(DELTA_Y*GSI)/6-DELTA_Y*A10*A10/4;
 J101:=J101+SQDELTA_Y*DELTA_Y*GSI*B1P/PI;
 J101:=J101-DELTA_Y*DELTA_Y*B11P; { multiple integral J101 }
 J110:=SQR(DELTA_Y*GSI)/6+DELTA_Y*A10*A10/4;
 J110:=J110-SQDELTA_Y*DELTA_Y*GSI*B1P/(2.*PI);
 J110:=J110+SQDELTA_Y*DELTA_Y*A10*GSI/4;
 J110:=J110-DELTA_Y*DELTA_Y*C11P; { multiple integral J110 }
 J011:=J11*DELTA_Y-J101-J110; { multiple integral J011 }
END;{ MULTIINT }

