{ Estimation of probabilities that X(T) lies in the interval [C1,C2] using  }
{ the variance reducing unbiased Euler estimator                            }
{ Considers the stochastic differential equation                            }
{       dX(t) = 0.5*(1-sin(X(t)) dt + dW(t)        with X(T0) = X0          }
{  on the time interval [T0,T] where W(t) is a Wiener process               }
{ Equidistant approximation Y(t) of X(t) by the Euler scheme with different }
{ time step sizes DELTA                                                     }
{ Uses the Polar Marsaglia method to generate Gaussian random numbers       }
{ Estimation of the 90% confidence intervals for the estimates E ( g(Y(T) ) }
{ based on the Euler approximations at the time T using M batches(M=20)     }
{ with the sample size N = 200 (g(.) is the indicator function of [C1,C2])  }
{ Plots the confidence intervals for different time step sizes              }
{ Displays the set of the estimates for these probabilities                 }
{ Written by Henri Schurz, 9.10. 1991                                       }
{ Note : If changing the batch numbers remember that the corresponding per- }
{        centage points should also be changed. Run time errors may occur if}
{        the step sizes too small (This numerical problem requires the re-  }
{        arrangement at the calculation of the densities to be estimated).  }

PROGRAM PRX5X5X5; { PC-Exercise 16.4.3 }

USES CRT,DOS,GRAPH,INIT,SETSCR,SERVICE,RANDNUMB,AAGRAPHS;

CONST
 NUM=4;          { number of different time step sizes                }
 T0=0.0;         { initial time                                       }
 T=1.0;          { final time                                         }
 DELTA=(T-T0)/8; { largest time step size of the Euler approximations }
 X0=0.0;         { initial value                                      }
 C1=0.3;         { left interval endpoint                             }
 C2=0.4;         { right interval endpoint ( C1 < C2 )                }
 A=0.0;          { parameter in the density PD = ptilde               }
 SIGMA=1.0;      { parameter in the density PD = ptilde ( SIGMA > 0 ) }
 OA=0.5;         { parameter in the optimal density POD = dtilde      }
 OSIGMA=1.0;     { parameter (>0) in the optimal density POD = dtilde }
 M=20;           { number of batches                                  }
 N=200;          { batch size                                         }
 QUANTILE=1.73;  { percentage point of the t-distribution             }

TYPE
 VECTOR0=ARRAY[1..M] OF REAL;
 VECTOR2=ARRAY[0..NUMINV] OF INTEGER;

VAR
 CR:STRING;             { help string                                     }
 G:INTEGER;             { index of the current time step size             }
 I:INTEGER;             { time step index                                 }
 J:INTEGER;             { batch index                                     }
 K:INTEGER;             { sample index within the current batch           }
 TAOI,TAOOLD:REAL;      { subinterval point                               }
 TAO:VECTOR;            { intermediate random points in [TAOOLD,TAOI]     }
 LI:INTEGER;            { current random index l[i]                       }
 LJ,LK:INTEGER;         { help indexes ( LJ = j, LK = k )                 }
 DELTA_Y:REAL;          { time step size of the Euler approximation       }
 SQDELTA_Y:REAL;        { sqrt of the current time step size              }
 NUMSTEPS:INTEGER;      { number of approximation steps                   }
 G1,G2,U1,U2:REAL;      { Gaussian random numbers                         }
 U:REAL;                { random number to generate the points TAO[I]     }
 XI,XIOLD:REAL;         { values of the Gaussian increments for DTILDE    }
 XT:REAL;               { end value of the path for the optimal density   }
 X:VECTOR;              { vector of the auxiliary points x(i,j) = x[LJ]   }
 DTILDE:REAL;           { value of the corresponding optimal density      }
 P:REAL;                { transition density based on the densities q     }
 Q:REAL;                { unbiased estimator for the transition density q }
 SUM,PROD:REAL;         { help values for the calculation of density q    }
 ETAEUL:REAL;           { variance reducing unbiased Euler estimate       }
 AVERAGE,VARIANCE:REAL; { statistical parameters                          }
 MUYLON:VECTOR0;        { sum of the estimates within a batch             }
 DEL:VECTOR1;           { time step sizes                                 }
 DIFFER:VECTOR1;        { half of the confidence interval lengths         }
 MU:VECTOR1;            { Euler estimates for different time step sizes   }

{ Generates the optimal density for the parameters a=0.5 and sigma=1 }

FUNCTION POD(S1,X1,S2,X2:REAL):REAL;
VAR QQ:REAL;
BEGIN
 QQ:=X2-X1-OA*(S2-S1);
 POD:=(1./SQRT(2.*OSIGMA*PI*(S2-S1)))*EXP(-QQ*QQ/(2.*OSIGMA*(S2-S1)));
END;{ POD }

{ Generates the density D = ptilde with parameters a = 0 and sigma = 1 }

FUNCTION PD(S1,X1,S2,X2:REAL):REAL;
VAR QQ:REAL;
BEGIN
 QQ:=X2-X1-A*(S2-S1);
 PD:=(1./SQRT(2.*PI*SIGMA*(S2-S1)))*EXP(-QQ*QQ/(2.*SIGMA*(S2-S1)));
END;{ PD }

{ Generates the density PHI(s,y,t,x) }

FUNCTION PHI(S1,X1,S2,X2:REAL):REAL;
VAR QQ:REAL;
BEGIN
 QQ:=X2-X1-0.5*(1.-SIN(X2))*(S2-S1);
 PHI:=(1./SQRT(2.*PI*(S2-S1)))*EXP(-QQ*QQ/(2.*(S2-S1)));
END;{ PHI }

{ Generates the transition density K(s,y,t,x) }

FUNCTION PK(S1,X1,S2,X2:REAL):REAL;
VAR AX,AY,QX:REAL;
BEGIN
 AX:=0.5*(1.-SIN(X2));AY:=0.5*(1.-SIN(X1));QX:=X2-X1-AX*(S2-S1);
 PK:=(1./SQRT(2.*PI*(S2-S1)))*((AY-AX)*QX/(S2-S1))*EXP(-QX*QX/(2.*(S2-S1)));
END;{ PK }

{ Generates the functional form of g(.) }

FUNCTION GXT(X:REAL):REAL;
BEGIN
 IF ((X<C1) OR (X>C2)) THEN GXT:=0.0
  ELSE GXT:=1.0; { provides the indicator function of the interval [C1,C2] }
END;{ GXT }

{ Computes the sample average and variance of a given data vector X }
{ with sample size NN                                               }

PROCEDURE COMPSAMPLEPARA(NN:INTEGER;X:VECTOR0;VAR SAVERAGE,SVARIANCE:REAL);
VAR
 J:INTEGER; { data index    }
 SQ:REAL;   { help variable }
BEGIN
 SAVERAGE:=0.0;SVARIANCE:=0.0;SQ:=0.0; { initialization }
 FOR J:=1 TO NN DO BEGIN SAVERAGE:=SAVERAGE+X[J];SQ:=SQ+X[J]*X[J]; END;
 SVARIANCE:=(SQ-SAVERAGE*SAVERAGE/NN)/(NN-1);
 SAVERAGE:=SAVERAGE/NN;
END;{ COMPSAMPLEPARA }

{ Main program : }

BEGIN

 INITIALIZE; { standard initialization }
 MAINWINDOW('Problem 5.5.5 (PC-Exercise 16.4.3)');
 STATUSLINE('Be patient! This will take the computer some time!');

{ Estimation of the confidence intervals for ETAEUL }
{ for different time step sizes :                   }

 DELTA_Y:=2.*DELTA;
 G:=0;
 REPEAT
  G:=G+1;                   { index of the time step size used }
  DELTA_Y:=DELTA_Y/2;       { current time step size           }
  SQDELTA_Y:=SQRT(DELTA_Y); { sqrt of the time step size       }
  NUMSTEPS:=TRUNC((T-T0)/DELTA_Y+0.0000001); { number of steps }

 { Generation for different batches : }

  J:=0;
  REPEAT
   J:=J+1;         { batch index                            }
   MUYLON[J]:=0.0; { sum of the estimates of the batch used }

  { Generation of different trajectories : }

   K:=0;
   REPEAT
    K:=K+1; { index of the trajectory used }

   { Generation of the variance reducing unbiased Euler estimate : }

    TAOI:=T0;XI:=X0;
    XT:=RANDOM*(C2-C1)+C1; { random end value for the optimal density }
    DTILDE:=1.0; { initial value of the optimal density }
    P:=1.0; { initial value of the transition density p }
    FOR I:=1 TO NUMSTEPS DO { calculation along the time steps }
     BEGIN
      TAOOLD:=TAOI;TAOI:=TAOI+DELTA_Y; { subinterval end points }
      XIOLD:=XI; { Gaussian increment at the left subinterval point }
      IF I<NUMSTEPS THEN
        BEGIN
         IF I MOD 2=1 THEN GENERATE(U1,U2) ELSE U1:=U2; { by Polar Marsaglia }
         XI:=SQRT(OSIGMA*(T-TAOI)*DELTA_Y/(T-TAOOLD))*U1; { increment }
         XI:=XI+(XIOLD*(T-TAOI)+XT*DELTA_Y)/(T-TAOOLD); { for DTILDE }
        END
       ELSE
        XI:=XT;
      LI:=0;LK:=0;
      TAO[0]:=TAOI;
      WHILE LI=0 DO { generates the random intermediate points TAO[I] }
       BEGIN
        LK:=LK+1;
        U:=RANDOM;
        IF U<=EXP(-(TAO[LK-1]-TAOOLD)) THEN
          BEGIN
           LI:=LK; { random index l(i) }
           TAO[LK]:=TAOOLD;
          END
         ELSE
          TAO[LK]:=LN(U*(EXP(TAO[LK-1]-TAOOLD)-1.0)+1.0)+TAOOLD;
       END; { WHILE }
      X[0]:=XIOLD;X[LI]:=XI;TAO[LI]:=TAOOLD;
      SUM:=0.0; { for the calculation of unbiased estimator Q }
      IF LI>1 THEN { calculation for the estimate Q }
       BEGIN
        FOR LK:=1 TO LI-1 DO { generates random intermediate points x(i,j) }
         BEGIN
          IF LK MOD 2=1 THEN GENERATE(G1,G2) ELSE G1:=G2; { uses Polar Marsaglia }
          X[LK]:=G1*SQRT(SIGMA*(TAOI-TAO[LI-LK])
                      *(TAO[LI-LK]-TAO[LI-LK+1])/(TAOI-TAO[LI-LK+1]));
          X[LK]:=X[LK]+X[LK-1]*(TAOI-TAO[LI-LK])/(TAOI-TAO[LI-LK+1]);
          X[LK]:=X[LK]+XI*(TAO[LI-LK]-TAO[LI-LK+1])/(TAOI-TAO[LI-LK+1]);
         END;
        FOR LK:=1 TO LI-1 DO { calculates help values for the estimate Q }
         BEGIN
          PROD:=1.0;LJ:=0;
          REPEAT
           LJ:=LJ+1;
           PROD:=PROD*PK(TAO[LJ],X[LI-LJ],TAO[LJ-1],X[LI-LJ+1])
                     /PD(TAO[LJ],X[LI-LJ],TAO[LJ-1],X[LI-LJ+1]);
          UNTIL LJ=LK;
          SUM:=SUM+PROD*PHI(TAOOLD,XIOLD,TAO[LK],X[LI-LK])
                /(PD(TAOOLD,XIOLD,TAO[LK],X[LI-LK])*EXP(TAO[LK]-TAOOLD));
         END;
       END; { IF LI>1 }

     { Generation of the unbiased estimator Q : }

      Q:=PHI(TAOOLD,XIOLD,TAOI,XI)
         +PD(TAOOLD,XIOLD,TAOI,XI)*EXP(TAOI-TAOOLD)*SUM;

     { Estimation of the current transition density P : }

      P:=P*Q;

     { Estimation of the optimal density DTILDE : }

      IF I<NUMSTEPS THEN
        DTILDE:=DTILDE*POD(TAOOLD,XIOLD,TAOI,XI)
                      *POD(TAOI,XI,T,XT)/POD(TAOOLD,XIOLD,T,XT)
       ELSE
        DTILDE:=DTILDE/(C2-C1);

     END; { end of the loop for the time step indices I }

   { Variance reducing unbiased Euler estimate : }

    ETAEUL:=(P/DTILDE)*GXT(XT); { GXT(XT) = 1 is trivially satisfied }

   { Summation of the estimates within the batch : }

    MUYLON[J]:=MUYLON[J]+ETAEUL;

   UNTIL K=N;{ REPEAT for different samples }
   MUYLON[J]:=MUYLON[J]/N; { estimate for the simulated batch }
  UNTIL J=M;{ REPEAT for different batches }

{ Calculation of the confidence interval and initialization of data : }

  DEL[G]:=DELTA_Y; { current time step size }
  COMPSAMPLEPARA(M,MUYLON,AVERAGE,VARIANCE);
  MU[G]:=AVERAGE;            { midpoint of the confidence interval }
  DIFFER[G]:=QUANTILE*SQRT(VARIANCE/M); { half the interval length }

 UNTIL G=NUM;{ REPEAT for different time step sizes }

{ Printout of the results : }

 CLEARDEVICE;
 CR:='Prob(X(T) in ['+CHCR(C1)+','+CHCR(C2)+'])';
 CONFINV(NUM,MU,DIFFER,DEL,CR,'DELTA');
 CR:='90% Confidence intervals for Prob(X(T) in ['+CHCR(C1)+',';
 CR:=CR+CHCR(C2)+']) (Var.red. unb. Euler,T=';
 CR:=CR+CHCR(T)+')';
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

{ Printout of the midpoints of the confidence intervals : }

 CLEARDEVICE;SETTEXTJUSTIFY(0,1);
 G:=0;
 REPEAT
  G:=G+1;
  CR:='Delta = '+CHCR(DEL[G])+'   Prob(X(T) in ['+CHCR(C1)+','+CHCR(C2)+'])';
  CR:=CR+' = '+CHCR(MU[G])+' +/- '+CHCR(DIFFER[G]);
  OUTTEXTXY(10,TRUNC(MAXY/NUM)+3*G*TEXTHEIGHT('M'),CR);
 UNTIL G=NUM;
 CR:='Var. red. unb. Euler est. for the probability ';
 CR:=CR+ 'P(X(T) in ['+CHCR(C1)+','+CHCR(C2)+']) at time T='+CHCR(T);
 STATUSLINE(CR);

 WAITTOGO; { waits for <ENTER> to be pressed }
           { ! <ESC> terminates the program  }

 MYEXITPROC; { closes graphics mode and sets the old procedure address }
END.{ PRX5X5X5 }