 
function p = sampled_signal(in1,in2,varargin)

% function p = sampled_signal(in1,in2,varargin)
%
% Constructor for sampled signal class
%
% sampled_signal(<sample_structure>)
% sampled_signal(<sampled_signal>)
% sampled_signal(struct(<sampled_signal>))
% sampled_signal(<matrix>,<sample_rate>)
%
% sampled_signal(<name>,<sample_structure>,varargin)
% sampled_signal(<name>,<sample_signal>,   varargin) % uses signal's 
%                                                      sample structure
% Possible names/varargins:
%
%-----------------%----------------------------%-------------------
%     name        %    optional argument(s)    %  argument default
%-----------------%----------------------------%-------------------
%                 %                            %
%   'sinc'        %                            %
%   'impulse'     %                            %
%   'box'         % half width                 % 1/2
%   'sine'        %                            %
%   'exp'         %                            %
%   'indicator'   % interval end points (a,b)  % (0,1)
%   'packet'      %                            %
%   'opacket'     %                            %
%   '2packet'     %                            %
%   'chirp'       % chirp rate, upshift        % 1/2, none
%   'chirp2'      % chirp rates                % [1 3]/2
%   'chirp10'     % chirp rate (upshift by 10) % 1/2
%   'noise'       % randn distributed signal   % 
%                 %                            %
%-----------------%----------------------------%-------------------

  p=class(initial_struct,'sampled_signal');
  
  if nargin==0, return, end
  
  switch class(in1)
  
    case 'sampled_signal'
      p = in1;
      for ii=1:length(p(:))
	p(ii).name = sprintf('%s_{%d}',inputname(1),ii);
      end
      
    case 'double'
      p.sig = in1;
      if nargin<2
	if length(in1)==2
	  p.d   = sample_structure(in1);
	  p.sig = zeros(round(2*p.d(2)/p.d(1)),1);
	else
	  p.d = sample_structure(1,in1);
	end
      else
	if isa(in2,'sampled_signal')
	  p.d = sample_structure(in2(1).d,in1);
	else
	  p.d = sample_structure(in2(1),in1);
	end
	  
      end
      
      p.dy = sample_structure(1,p.sig(1,:)');  % y-sample struct
      
    case 'struct'
      p=class(in1,'sampled_signal');
      
    case 'char'
      
      switch class(in2)
	case 'sampled_signal'
	  p.d  = in2.d;
	  p.d0 = in2.d0;
      
	case 'double'
	  if length(in2)==2
	    p.d = sample_structure(in2);
	  else
	    p.d = in2;
	  end
      end
      
      p = signal(p,in1,varargin);
      
  end
  
  if rem(size(p.sig,1),2) 
    disp('Warning: last sample in odd length signal truncated!')
    p.sig(end,:)=[];
    p.d = sample_structure(p.d(1),p.sig);
  end    
    
%
%
%
    
function p=initial_struct

% signal attributes

   p.sig  = [];         % data
   p.d    = [1 0];      % sample structure(s)
   p.d0   = [0];        % 1 if support interval starts at 0?

   p.orig = [];         % sampled signal (room for original version)
   
   p.dy   = [1 0];
   p.dy0  = [0];
   
   p.SNR  = inf;        % signal to noise ratio
   
   p.interp = '*cubic'; % interpolation method used in resampling

% desired axis attributes   

   p.xlabel = GetGlobal('TimeLabel','default','Time (\mus)');
   p.ylabel = [];
   
   p.buttondown = 'mouse_limits';
   
   p.grid = 'off';
   p.box  = 'off';
   
   p.colormap  = 'hot';

% name management

   p.name      = 'NoName';
   p.nametrack = 'on';
   
%
%
%
   
function p=signal(p,name,varargin)

%
% Description:
% 	Make a sampled signal by name
%

   tt=lattice(p);
   
   switch(lower(name))
   
   case 'sinc'
      tt(n0(p))=1; % avoid divide by zero
      p.sig       = sin(pi*tt)./(pi*tt);
      p.sig(n0(p))=1;

   case 'impulse'
      p.sig = zeros(size(tt));
      p.sig(n0(p)) = 1/p.d(1);

   case 'box'
      if length(varargin{1})>0
	aa=varargin{1}{1}(:).';
      else
	aa=1/2;
      end
      p.sig = indicator(tt,-aa,aa,'[]');
      name  = sprintf('%s_{%s}',name,num2str(aa));
      
   case 'cosine'
      freq=varargin{1}{1}(:).';
      if length(varargin{1})>1
	aa=varargin{1}{2}(:).';
	p.sig = sum(repmat(aa,length(tt),1).*cos(2*pi*tt*freq),2);
      else
	p.sig = sum(cos(2*pi*tt*freq),2);
      end
      
   case 'sine'

      p = make_sine(p,[10; 1; -0.5; 0.5]);
	    
   case 'exp'
      if length(varargin{1})>0
	v=varargin{1}{1}(:).';
      else
	v=1;
      end
      
      p.sig = exp(j*2*pi*tt*v);

   case 'expwin'
      aa=varargin{1}{1};
      bb=varargin{1}{2};
      
      tta =  -aa(2).*(tt-aa(1)).^2;
      ttb =  -bb(2).*(tt-bb(1)).^2;

      p.sig  = exp(tta) .* indicator(tt,-inf,aa(1)) + ...
	       exp(ttb) .* indicator(tt,bb(1),inf)  + ...
	                   indicator(tt,aa(1),bb(1));
      
   case 'sineon'
     aa=varargin{1}{1};
     bb=varargin{1}{2};
     
     p.sig = sineon(tt,aa,bb);
      
    case 'bell'
     aa=varargin{1}{1};
     bb=varargin{1}{2};
     
     p.sig = bell(tt,aa,bb);
      
   case 'indicator'
     if length(varargin{1})>0
       aa=varargin{1}{1};
       bb=varargin{1}{2};
     else
       aa=0;
       bb=1;
     end
       
     p.sig  = indicator(tt,aa,bb);
     name   = sprintf('{\\bf 1}_{[%s,%s)}',num2str(aa),num2str(bb)); %]
      
   case 'bell_1'
     
     aa=varargin{1}{1};
     bb=varargin{1}{2};
     
     a0 = -0.5; b0=1.5;
     
     mm = (b0-a0)/(bb-aa);
     pp = a0 - mm*aa;
     
     tt = mm*tt + pp;
     p.sig  = indicator(tt,a0,b0) .* sin(pi/4 *(1 + sin(pi*tt)));
     
   case 'packet'
     freq = [ 6	11	20];
     amp  = [ 1	1	1   ];
     t0   = [ -45	-15	15  ]/20;
     t1   = t0 + 1.25;
     
     p = make_sine(p,[freq; amp; t0; t1]);
     
   case 'opacket'
     freq = [ 6	20      11];
     amp  = [ 1	1	1   ];
     t0   = [ -30	-20	-5  ]/20;
     t1   = t0 + 1.25;
     
     p = make_sine(p,[freq; amp; t0; t1]);
     
   case '2packet'
     freq = [ 5  10 ];
     amp  = [ 1  1  ];
     t0   = [ -8 2  ]/5;
     t1   = [  0 8  ]/5;
     
     p = make_sine(p,[freq; amp; t0; t1]);
     
   case 'gaussian'
      if length(varargin{1})>0
	rr = cat(2,varargin{:}{:}); % variance and mean
      else
	rr=[];
      end
      if length(rr)<1
	rr=[1 0];
      elseif length(rr)<2
	rr(2)=0;
      end
      
      p.sig = 1/sqrt(2*pi*rr(1)) * exp(-((tt-rr(2)).^2/(2*rr(1))));
      name = sprintf('Gaussian_{%s,%s}',num2str(rr(2)),num2str(rr(1)));
     
     
   case 'chirp'
      if length(varargin{1})>0
	rr=varargin{1}{1}(:).';
	name = sprintf('%s@%s',name,num2str(rr));
      else
	rr=1;
      end
      if isempty(rr)
	rr=1;
      end
      
      p = chirp(p,rr);
      
      if length(varargin{1})>1
	g0=varargin{1}{2}(:).';
      
	p.name = name;
	p = upshift(p,g0);
	name = p.name;
      end
	     
    case 'chirp10'
     p=sampled_signal('chirp',p.d,varargin{1});
     p=upshift(p,10);
	     
   case 'chirp2'
      if length(varargin{1})>0
	rr=varargin{1}{1}(:).';
      else
	rr=[1 3];
      end
      p = chirp(p,rr);
      
   case 'noise'
      p.sig = randn(size(tt));
     
   otherwise
     command = sprintf('p = %s(p,varargin);',name);
     eval(command)
     
 end
 
 p.name = name;

%
%
%

function g=sineon(xx,aa,bb)


% aa length 2 onset  interval
% bb length 2 offset interval
   

   g=zeros(size(xx));
   
   aii=find(xx<=aa(2) & xx>aa(1));
   
   g(aii) = sin( pi/(2*diff(aa)) .* (xx(aii)-aa(1)));
   
   
   bii=find(xx<bb(2) & xx>bb(1));
   g(bii) = cos( pi/(2*diff(bb)) .* (xx(bii)-bb(1)));
   
   ones_ii = find(xx>=aa(2) & xx<=bb(1));
   g(ones_ii) = ones(size(ones_ii));

 

%
%
%

function sig=bell(tt,aa,bb,fac)

%
% aa(1:2,:) = onset  region
% bb(1:2,:) = offset region
%
   if nargin<4, fac=0.2; end

   
   dt = fac*(bb(:,1)-aa(:,1));
   
   if size(aa,2)==1
     aa(:,2)=aa(:,1)+dt;
   end
   
   if size(bb,2)==1
     bb(:,2)=bb(:,1);
     bb(:,1)=bb(:,1)-dt;
   end
   
   nsigs=size(aa,1);

   sig = zeros(length(tt),nsigs);
   for ii=1:nsigs
     tta =  (tt-aa(ii,1))./(aa(ii,2)-aa(ii,1));
     ttb =  (tt-bb(ii,1))./(bb(ii,2)-bb(ii,1));
   
     sig(:,ii) = ...
	 sin(pi/2*tta).^2 .* indicator(tt,aa(ii,1),aa(ii,2)) + ...
	 cos(pi/2*ttb).^2 .* indicator(tt,bb(ii,1),bb(ii,2)) + ...
	 indicator(tt,aa(ii,2),bb(ii,1));
   end
   

%
% function p = make_sine(parameters, pd)
%
% yd = sample structure
%
% parameters row
%	      1	frequencies
%	      2	magnitudes
%	      3	start support
%	      4	end   support

function p = make_sine(p,parameters)

% init

  T = p.d(2);
  
  freqs = parameters(1,:);
  sp    = size(parameters,1);

% magnitudes

  if sp>1
    magnitude = parameters(2,:);
  else
    magnitude = ones( size(parameters,2) );
  end

% start times

  if sp>2
    start = parameters(3,:);
  else
    start = -0.9*T*ones( size(parameters,2) );
  end

% end times

  if sp>3
    endt = parameters(4,:);
  else
    endt = 0.9*T*ones( size(parameters,2) );
  end

% build the function

   tt = lattice(p);
   p.sig  = zeros(size(tt));

   for ii=1:length(freqs)
       p.sig = p.sig + magnitude(ii)*exp(j*2*pi*freqs(ii)*tt) .* ...
	   window1(tt,start(ii),endt(ii),0.2);
   end

%
%
%

function p = chirp(p,rate)

% Make superposition of "analytic" chirp signals
%
% rate = vector of chirp rates


% arg processing

  if nargin<2, rate = 10; end
	
% specify chirp signal 

  p.d0 = 1;
  tt   = lattice(p); 
  p.d0 = 0;
  
  nn = length(tt);
  
  nz = min(nn/4,64); % number of leading and trailing zeros

%  tt = tt - tt(nz);
  
  p.sig=0;
  for ii=1:length(rate)
    if rate(ii)<0
      fm = 2*pi*(1+cos(rate(ii)*2*pi*tt)).*tt;
    else
      fm = pi*rate(ii)* tt.*tt;
    end
    p.sig = p.sig + exp(j*fm); % analytic chirp
  end
  
% window

  p.sig = p.sig .* bell(tt,tt(nz),tt(length(tt)-nz));


  
%
%
%


% function Xab = indicator(x, a, b, endpt)
%
%	indicator function for one of
%		[a,b)  (default)
%		(a,b],
%		(a,b), or
%		[a,b]
% Interval inclusion is controlled by the 2 character string
% endpt, e.g.,
% 		indicator(x, 0,1, '(]') 
% means (0,1]

function Xab = indicator(x, a, b, endpt)

% arg process

	if nargin<2, a=0;   end
	if nargin<3, b=inf; end
	if nargin<4, endpt='[)'; end

% indicate

%	Xab = ( x>=a & x<=b );

% left end point

	if endpt(1)=='['
		Xab = x>=a;
	else
		Xab = x>a;
	end

% right end point

	if endpt(2)==']'
		Xab = Xab & (x<=b);
	else
		Xab = Xab & (x<b);
	end

  
%
%
%


function ww = window(tt, aa, bb, wtype)

% function ww = window(tt, aa, bb, wtype)
%
% Window function on the interval (aa,bb)
%
% Input:
%	tt	a lattice or sample structure 
%	aa	interval start
%	bb	interval end
% 	wtype   'bell' | 'sine'

% arg process

	if length(tt)==2
		tt=lattice(tt);
	end

	if nargin<2, aa= -tt(2); end
	if nargin<3, bb=  tt(2); end
	if nargin<4, wtype='bell'; end

% compute window fct

	if strcmp(wtype,'bell')
	  a0 = -0.5; b0=1.5;

	  mm = (b0-a0)/(bb-aa);
	  pp = a0 - mm*aa;

	  tt = mm*tt + pp;
	  ww  = indicator(tt,a0,b0) .* sin(pi/4 *(1 + sin(pi*tt)));
		
	elseif strcmp(wtype,'exp') % exp win
	  
	  ww  = exp(-(tt-aa).^2) .* indicator(tt,-inf,aa, '[)') + ...
	                            indicator(tt,aa,  bb, '[)') + ...
		exp(-(tt-bb).^2) .* indicator(tt,bb,  inf,'[)');
		  

	else % sine window

	  ww = indicator(tt,aa,bb) .* sin(pi/(bb-aa) .* (tt-aa));
	end
  
%
%
%

function ww = window1(tt, aa, bb, onset)

% function ww = window1(tt, aa, bb, onset)
%
% Window function on the interval (aa,bb)
%
% Input:
%	tt	a lattice or sample structure 
%	aa	interval start
%	bb	interval end
% 	wtype   'bell' | 'sine'

% arg process

	if length(tt)==2
		tt=lattice(tt);
	end

	if nargin<2, aa= -tt(2); end
	if nargin<3, bb=  tt(2); end
	if nargin<4, wtype='bell'; end

% compute window fct

        ww=ones(size(tt));
	
	ww = window(tt,aa,aa+2*onset).*indicator(tt,aa,aa+onset) + ...
	                               indicator(tt,aa+onset,bb-onset) +...
	     window(tt,bb-2*onset,bb).*indicator(tt,bb-onset,bb);	       

  
%
%
%


% function p = FM(p,FMs)
%
% Make a frequency modulated signal 
%
% FMs is a cell array containing line data:
%    xdata, ydata, interp


function p = FM(p,FMs)

% derefrence varargin (could be cell array itself)

   FMs=FMs{1};
   if ~isa(FMs{1},'struct') % is arg a cell array?
     FMs=FMs{1};
   end
 
   sample_rate = p.d(1);
   if isfield(FMs{1},'d0')
     p.d0 = FMs{1}.d0;
   else
     p.d0=1;
   end
   
   tt=lattice(p);
   
   p.sig=0;
   
   for ii=1:length(FMs)
     
     xdata=FMs{ii}.xdata;
     ydata=FMs{ii}.ydata;
     
     mx=min(xdata);
     Mx=max(xdata);
             
     if strcmp(FMs{ii}.interp,'linear') | length(xdata)<3 
       fm = LinearInterp(tt,xdata,ydata);
       p.SNR=[];
     else
       fm = interp1(xdata,ydata,tt,FMs{ii}.interp);
       p.SNR{ii} =fm;
     end
      
     p.sig = p.sig + exp(j*2*pi*cumsum(nan2zero(fm))*sample_rate).*bell(tt,mx,Mx,0.1);
     
     
   end
   






