(*-----------------------------------------------------------------------*)
(* JUP200: Jupiter; ecliptic coordinates L,B,R (in deg and AU)           *)
(*         equinox of date                                               *)
(*         T: time in Julian centuries since J2000                       *)
(*            = (JED-2451545.0)/36525                                    *)
(*-----------------------------------------------------------------------*)
PROCEDURE JUP200(T:REAL;VAR L,B,R:REAL);
  CONST P2=6.283185307;
  VAR C5,S5:        ARRAY [-1..5] OF REAL;
      C,S:          ARRAY [-10..0] OF REAL;
      M5,M6,M7:     REAL;
      U,V,DL,DR,DB: REAL;
      I:            INTEGER;

  FUNCTION FRAC(X:REAL):REAL;
    BEGIN  X:=X-TRUNC(X); IF (X<0) THEN X:=X+1.0; FRAC:=X  END;

  PROCEDURE ADDTHE(C1,S1,C2,S2:REAL; VAR C,S:REAL);
    BEGIN  C:=C1*C2-S1*S2; S:=S1*C2+C1*S2; END;

  PROCEDURE TERM(I5,I,IT:INTEGER;DLC,DLS,DRC,DRS,DBC,DBS:REAL);
    BEGIN
      IF IT=0 THEN ADDTHE(C5[I5],S5[I5],C[I],S[I],U,V)
              ELSE BEGIN U:=U*T; V:=V*T END;
      DL:=DL+DLC*U+DLS*V; DR:=DR+DRC*U+DRS*V; DB:=DB+DBC*U+DBS*V;
    END;

  PROCEDURE PERTSAT;  (* Kepler terms and perturbations by Saturn *)
    VAR I: INTEGER;
    BEGIN
      C[0]:=1.0; S[0]:=0.0; C[-1]:=COS(M6); S[-1]:=-SIN(M6);
      FOR I:=-1 DOWNTO -9 DO ADDTHE(C[I],S[I],C[-1],S[-1],C[I-1],S[I-1]);
      TERM(-1, -1,0,  -0.2,    1.4,     2.0,   0.6,    0.1, -0.2);
      TERM( 0, -1,0,   9.4,    8.9,     3.9,  -8.3,   -0.4, -1.4);
      TERM( 0, -2,0,   5.6,   -3.0,    -5.4,  -5.7,   -2.0,  0.0);
      TERM( 0, -3,0,  -4.0,   -0.1,     0.0,   5.5,    0.0,  0.0);
      TERM( 0, -5,0,   3.3,   -1.6,    -1.6,  -3.1,   -0.5, -1.2);
      TERM( 1,  0,0,-113.1,19998.6,-25208.2,-142.2,-4670.7,288.9);
      TERM( 1,  0,1, -76.1,   66.9,   -84.2, -95.8,   21.6, 29.4);
      TERM( 1,  0,2,  -0.5,   -0.3,     0.4,  -0.7,    0.1, -0.1);
      TERM( 1, -1,0,  78.8,  -14.5,    11.5,  64.4,   -0.2,  0.2);
      TERM( 1, -2,0,  -2.0, -132.4,    28.8,   4.3,   -1.7,  0.4);
      TERM( 1, -2,1,  -1.1,   -0.7,     0.2,  -0.3,    0.0,  0.0);
      TERM( 1, -3,0,  -7.5,   -6.8,    -0.4,  -1.1,    0.6, -0.9);
      TERM( 1, -4,0,   0.7,    0.7,     0.6,  -1.1,    0.0, -0.2);
      TERM( 1, -5,0,  51.5,  -26.0,   -32.5, -64.4,   -4.9,-12.4);
      TERM( 1, -5,1,  -1.2,   -2.2,    -2.7,   1.5,   -0.4,  0.3);
      TERM( 2,  0,0,  -3.4,  632.0,  -610.6,  -6.5, -226.8, 12.7);
      TERM( 2,  0,1,  -4.2,    3.8,    -4.1,  -4.5,    0.2,  0.6);
      TERM( 2, -1,0,   5.3,   -0.7,     0.7,   6.1,    0.2,  1.1);
      TERM( 2, -2,0, -76.4, -185.1,   260.2,-108.0,    1.6,  0.0);
      TERM( 2, -3,0,  66.7,   47.8,   -51.4,  69.8,    0.9,  0.3);
      TERM( 2, -3,1,   0.6,   -1.0,     1.0,   0.6,    0.0,  0.0);
      TERM( 2, -4,0,  17.0,    1.4,    -1.8,   9.6,    0.0, -0.1);
      TERM( 2, -5,0,1066.2, -518.3,    -1.3, -23.9,    1.8, -0.3);
      TERM( 2, -5,1, -25.4,  -40.3,    -0.9,   0.3,    0.0,  0.0);
      TERM( 2, -5,2,  -0.7,    0.5,     0.0,   0.0,    0.0,  0.0);
      TERM( 3,  0,0,  -0.1,   28.0,   -22.1,  -0.2,  -12.5,  0.7);
      TERM( 3, -2,0,  -5.0,  -11.5,    11.7,  -5.4,    2.1, -1.0);
      TERM( 3, -3,0,  16.9,   -6.4,    13.4,  26.9,   -0.5,  0.8);
      TERM( 3, -4,0,   7.2,  -13.3,    20.9,  10.5,    0.1, -0.1);
      TERM( 3, -5,0,  68.5,  134.3,  -166.9,  86.5,    7.1, 15.2);
      TERM( 3, -5,1,   3.5,   -2.7,     3.4,   4.3,    0.5, -0.4);
      TERM( 3, -6,0,   0.6,    1.0,    -0.9,   0.5,    0.0,  0.0);
      TERM( 3, -7,0,  -1.1,    1.7,    -0.4,  -0.2,    0.0,  0.0);
      TERM( 4,  0,0,   0.0,    1.4,    -1.0,   0.0,   -0.6,  0.0);
      TERM( 4, -2,0,  -0.3,   -0.7,     0.4,  -0.2,    0.2, -0.1);
      TERM( 4, -3,0,   1.1,   -0.6,     0.9,   1.2,    0.1,  0.2);
      TERM( 4, -4,0,   3.2,    1.7,    -4.1,   5.8,    0.2,  0.1);
      TERM( 4, -5,0,   6.7,    8.7,    -9.3,   8.7,   -1.1,  1.6);
      TERM( 4, -6,0,   1.5,   -0.3,     0.6,   2.4,    0.0,  0.0);
      TERM( 4, -7,0,  -1.9,    2.3,    -3.2,  -2.7,    0.0, -0.1);
      TERM( 4, -8,0,   0.4,   -1.8,     1.9,   0.5,    0.0,  0.0);
      TERM( 4, -9,0,  -0.2,   -0.5,     0.3,  -0.1,    0.0,  0.0);
      TERM( 4,-10,0,  -8.6,   -6.8,    -0.4,   0.1,    0.0,  0.0);
      TERM( 4,-10,1,  -0.5,    0.6,     0.0,   0.0,    0.0,  0.0);
      TERM( 5, -5,0,  -0.1,    1.5,    -2.5,  -0.8,   -0.1,  0.1);
      TERM( 5, -6,0,   0.1,    0.8,    -1.6,   0.1,    0.0,  0.0);
      TERM( 5, -9,0,  -0.5,   -0.1,     0.1,  -0.8,    0.0,  0.0);
      TERM( 5,-10,0,   2.5,   -2.2,     2.8,   3.1,    0.1, -0.2);
    END;

  PROCEDURE PERTURA;  (* perturbations by Uranus *)
    BEGIN
      C[-1]:=COS(M7); S[-1]:=-SIN(M7);
      ADDTHE(C[-1],S[-1],C[-1],S[-1],C[-2],S[-2]);
      TERM( 1, -1,0,   0.4,    0.9,     0.0,   0.0,    0.0,  0.0);
      TERM( 1, -2,0,   0.4,    0.4,    -0.4,   0.3,    0.0,  0.0);
    END;

  PROCEDURE PERTSUR;  (* perturbations by Saturn and Uranus *)
    VAR PHI,X,Y: REAL;
    BEGIN
      PHI:=(2*M5-6*M6+3*M7); X:=COS(PHI); Y:=SIN(PHI);
      DL:=DL-0.8*X+8.5*Y; DR:=DR-0.1*X;
      ADDTHE(X,Y,C5[1],S5[1],X,Y);
      DL:=DL+0.4*X+0.5*Y; DR:=DR-0.7*X+0.5*Y; DB:=DB-0.1*X;
    END;

  BEGIN  (* JUP200 *)

    DL:=0.0; DR:=0.0; DB:=0.0;
    M5:=P2*FRAC(0.0565314+8.4302963*T); M6:=P2*FRAC(0.8829867+3.3947688*T);
    M7:=P2*FRAC(0.3969537+1.1902586*T);
    C5[0]:=1.0;     S5[0]:=0.0;
    C5[1]:=COS(M5); S5[1]:=SIN(M5);  C5[-1]:=C5[1]; S5[-1]:=-S5[1];
    FOR I:=2 TO 5 DO ADDTHE(C5[I-1],S5[I-1],C5[1],S5[1],C5[I],S5[I]);
    PERTSAT; PERTURA; PERTSUR;
    L:= 360.0*FRAC(0.0388910 + M5/P2 + ((5025.2+0.8*T)*T+DL)/1296.0E3 );
    R:= 5.208873 + 0.000041*T  +  DR*1.0E-5;
    B:= ( 227.3 - 0.3*T + DB ) / 3600.0;

  END;   (* JUP200 *)

