(*-----------------------------------------------------------------------*)
(* NEP200: Neptune; ecliptic coordinates L,B,R (in deg and AU)           *)
(*         equinox of date                                               *)
(*         (T: time in Julian centuries since J2000)                     *)
(*         (   = (JED-2451545.0)/36525             )                     *)
(*-----------------------------------------------------------------------*)
PROCEDURE NEP200(T:REAL;VAR L,B,R:REAL);
  CONST P2=6.283185307;
  VAR C8,S8:         ARRAY [ 0..6] OF REAL;
      C,S:           ARRAY [-6..0] OF REAL;
      M5,M6,M7,M8:   REAL;
      U,V, DL,DR,DB: REAL;
      I: INTEGER;

  FUNCTION FRAC(X:REAL):REAL;
    BEGIN  X:=X-TRUNC(X); IF (X<0) THEN X:=X+1.0; FRAC:=X  END;

  PROCEDURE ADDTHE(C1,S1,C2,S2:REAL; VAR C,S:REAL);
    BEGIN  C:=C1*C2-S1*S2; S:=S1*C2+C1*S2; END;

  PROCEDURE TERM(I1,I,IT:INTEGER;DLC,DLS,DRC,DRS,DBC,DBS:REAL);
    BEGIN
      IF IT=0 THEN ADDTHE(C8[I1],S8[I1],C[I],S[I],U,V)
              ELSE BEGIN U:=U*T; V:=V*T END;
      DL:=DL+DLC*U+DLS*V; DR:=DR+DRC*U+DRS*V; DB:=DB+DBC*U+DBS*V;
    END;

  PROCEDURE PERTJUP;  (* perturbations by Jupiter *)
    BEGIN
      C[0]:=1.0; S[0]:=0.0; C[-1]:=COS(M5); S[-1]:=-SIN(M5);
      ADDTHE(C[-1],S[-1],C[-1],S[-1],C[-2],S[-2]);
      TERM(0,-1,0,  0.1,   0.1,    -3.0,   1.8,   -0.3, -0.3);
      TERM(1, 0,0,  0.0,   0.0,   -15.9,   9.0,    0.0,  0.0);
      TERM(1,-1,0,-17.6, -29.3,   416.1,-250.0,    0.0,  0.0);
      TERM(1,-2,0, -0.4,  -0.7,    10.4,  -6.2,    0.0,  0.0);
      TERM(2,-1,0, -0.2,  -0.4,     2.4,  -1.4,    0.4, -0.3);
    END;

  PROCEDURE PERTSAT;  (* perturbations by Saturn *)
    BEGIN
      C[0]:=1.0; S[0]:=0.0; C[-1]:=COS(M6); S[-1]:=-SIN(M6);
      ADDTHE(C[-1],S[-1],C[-1],S[-1],C[-2],S[-2]);
      TERM(0,-1,0, -0.1,   0.0,     0.2,  -1.8,   -0.1, -0.5);
      TERM(1, 0,0,  0.0,   0.0,    -8.3, -10.4,    0.0,  0.0);
      TERM(1,-1,0, 13.6, -12.7,   187.5, 201.1,    0.0,  0.0);
      TERM(1,-2,0,  0.4,  -0.4,     4.5,   4.5,    0.0,  0.0);
      TERM(2,-1,0,  0.4,  -0.1,     1.7,  -3.2,    0.2,  0.2);
      TERM(2,-2,0, -0.1,   0.0,    -0.2,   2.7,    0.0,  0.0);
    END;

  PROCEDURE PERTURA;  (* Kepler terms and perturbations by Uranus *)
    VAR I: INTEGER;
    BEGIN
      C[0]:=1.0; S[0]:=0.0; C[-1]:=COS(M7); S[-1]:=-SIN(M7);
      FOR I:=-1 DOWNTO -5 DO ADDTHE(C[I],S[I],C[-1],S[-1],C[I-1],S[I-1]);
      TERM(1, 0,0, 32.3,3549.5,-25880.2, 235.8,-6360.5,374.0);
      TERM(1, 0,1, 31.2,  34.4,  -251.4, 227.4,   34.9, 29.3);
      TERM(1, 0,2, -1.4,   3.9,   -28.6, -10.1,    0.0, -0.9);
      TERM(2, 0,0,  6.1,  68.0,  -111.4,   2.0,  -54.7,  3.7);
      TERM(2, 0,1,  0.8,  -0.2,    -2.1,   2.0,   -0.2,  0.8);
      TERM(3, 0,0,  0.1,   1.0,    -0.7,   0.0,   -0.8,  0.1);
      TERM(0,-1,0, -0.1,  -0.3,    -3.6,   0.0,    0.0,  0.0);
      TERM(1, 0,0,  0.0,   0.0,     5.5,  -6.9,    0.1,  0.0);
      TERM(1,-1,0, -2.2,  -1.6,  -116.3, 163.6,    0.0, -0.1);
      TERM(1,-2,0,  0.2,   0.1,    -1.2,   0.4,    0.0, -0.1);
      TERM(2,-1,0,  4.2,  -1.1,    -4.4, -34.6,   -0.2,  0.1);
      TERM(2,-2,0,  8.6,  -2.9,   -33.4, -97.0,    0.2,  0.1);
      TERM(3,-1,0,  0.1,  -0.2,     2.1,  -1.2,    0.0,  0.1);
      TERM(3,-2,0, -4.6,   9.3,    38.2,  19.8,    0.1,  0.1);
      TERM(3,-3,0, -0.5,   1.7,    23.5,   7.0,    0.0,  0.0);
      TERM(4,-2,0,  0.2,   0.8,     3.3,  -1.5,   -0.2, -0.1);
      TERM(4,-3,0,  0.9,   1.7,    17.9,  -9.1,   -0.1,  0.0);
      TERM(4,-4,0, -0.4,  -0.4,    -6.2,   4.8,    0.0,  0.0);
      TERM(5,-3,0, -1.6,  -0.5,    -2.2,   7.0,    0.0,  0.0);
      TERM(5,-4,0, -0.4,  -0.1,    -0.7,   5.5,    0.0,  0.0);
      TERM(5,-5,0,  0.2,   0.0,     0.0,  -3.5,    0.0,  0.0);
      TERM(6,-4,0, -0.3,   0.2,     2.1,   2.7,    0.0,  0.0);
      TERM(6,-5,0,  0.1,  -0.1,    -1.4,  -1.4,    0.0,  0.0);
      TERM(6,-6,0, -0.1,   0.1,     1.4,   0.7,    0.0,  0.0);
    END;

  BEGIN  (* NEP200 *) 

    DL:=0.0; DR:=0.0; DB:=0.0;
    M5:=P2*FRAC(0.0563867+8.4298907*T); M6:=P2*FRAC(0.8825086+3.3957748*T);
    M7:=P2*FRAC(0.3965358+1.1902851*T); M8:=P2*FRAC(0.7214906+0.6068526*T);
    C8[0]:=1.0; S8[0]:=0.0; C8[1]:=COS(M8); S8[1]:=SIN(M8);
    FOR I:=2 TO 6 DO ADDTHE(C8[I-1],S8[I-1],C8[1],S8[1],C8[I],S8[I]);
    PERTJUP; PERTSAT; PERTURA;
    L:= 360.0*FRAC(0.1254046 + M8/P2 + ((4982.8-21.3*T)*T+DL)/1296.0E3 );
    R:= 30.072984 + (0.001234+0.000003*T) * T  +  DR*1.0E-5;
    B:= (  54.77 + ( 0.26 + 0.06*T) * T  +  DB ) / 3600.0;

  END;   (* NEP200 *)

