(*-----------------------------------------------------------------------*)
(* SAT200: Saturn; ecliptic coordinates L,B,R (in deg and AU)            *)
(*         equinox of date                                               *)
(*         (T: time in Julian centuries since J2000)                     *)
(*         (   = (JED-2451545.0)/36525             )                     *)
(*-----------------------------------------------------------------------*)
PROCEDURE SAT200(T:REAL;VAR L,B,R:REAL);
  CONST P2=6.283185307;
  VAR C6,S6:         ARRAY [ 0..11] OF REAL;
      C,S:           ARRAY [-6.. 1] OF REAL;
      M5,M6,M7,M8:   REAL;
      U,V, DL,DR,DB: REAL;
      I:             INTEGER;

  FUNCTION FRAC(X:REAL):REAL;
    BEGIN  X:=X-TRUNC(X); IF (X<0) THEN X:=X+1.0; FRAC:=X  END;

  PROCEDURE ADDTHE(C1,S1,C2,S2:REAL; VAR C,S:REAL);
    BEGIN  C:=C1*C2-S1*S2; S:=S1*C2+C1*S2; END;

  PROCEDURE TERM(I6,I,IT:INTEGER;DLC,DLS,DRC,DRS,DBC,DBS:REAL);
    BEGIN
      IF IT=0 THEN ADDTHE(C6[I6],S6[I6],C[I],S[I],U,V)
              ELSE BEGIN U:=U*T; V:=V*T END;
      DL:=DL+DLC*U+DLS*V; DR:=DR+DRC*U+DRS*V; DB:=DB+DBC*U+DBS*V;
    END;

  PROCEDURE PERTJUP;  (* Kepler terms and perturbations by Jupiter *)
    VAR I: INTEGER;
    BEGIN
      C[0]:=1.0; S[0]:=0.0; C[1]:=COS(M5); S[1]:=SIN(M5);
      FOR I:=0 DOWNTO -5 DO ADDTHE(C[I],S[I],C[1],-S[1],C[I-1],S[I-1]);
      TERM( 0,-1,0,   12.0,   -1.4,   -13.9,    6.4,    1.2,  -1.8);
      TERM( 0,-2,0,    0.0,   -0.2,    -0.9,    1.0,    0.0,  -0.1);
      TERM( 1, 1,0,    0.9,    0.4,    -1.8,    1.9,    0.2,   0.2);
      TERM( 1, 0,0, -348.3,22907.7,-52915.5, -752.2,-3266.5,8314.4);
      TERM( 1, 0,1, -225.2, -146.2,   337.7, -521.3,   79.6,  17.4);
      TERM( 1, 0,2,    1.3,   -1.4,     3.2,    2.9,    0.1,  -0.4);
      TERM( 1,-1,0,   -1.0,  -30.7,   108.6, -815.0,   -3.6,  -9.3);
      TERM( 1,-2,0,   -2.0,   -2.7,    -2.1,  -11.9,   -0.1,  -0.4);
      TERM( 2, 1,0,    0.1,    0.2,    -1.0,    0.3,    0.0,   0.0);
      TERM( 2, 0,0,   44.2,  724.0, -1464.3,  -34.7, -188.7, 459.1);
      TERM( 2, 0,1,  -17.0,  -11.3,    18.9,  -28.6,    1.0,  -3.7);
      TERM( 2,-1,0,   -3.5, -426.6,  -546.5,  -26.5,   -1.6,  -2.7);
      TERM( 2,-1,1,    3.5,   -2.2,    -2.6,   -4.3,    0.0,   0.0);
      TERM( 2,-2,0,   10.5,  -30.9,  -130.5,  -52.3,   -1.9,   0.2);
      TERM( 2,-3,0,   -0.2,   -0.4,    -1.2,   -0.1,   -0.1,   0.0);
      TERM( 3, 0,0,    6.5,   30.5,   -61.1,    0.4,  -11.6,  28.1);
      TERM( 3, 0,1,   -1.2,   -0.7,     1.1,   -1.8,   -0.2,  -0.6);
      TERM( 3,-1,0,   29.0,  -40.2,    98.2,   45.3,    3.2,  -9.4);
      TERM( 3,-1,1,    0.6,    0.6,    -1.0,    1.3,    0.0,   0.0);
      TERM( 3,-2,0,  -27.0,  -21.1,   -68.5,    8.1,  -19.8,   5.4);
      TERM( 3,-2,1,    0.9,   -0.5,    -0.4,   -2.0,   -0.1,  -0.8);
      TERM( 3,-3,0,   -5.4,   -4.1,   -19.1,   26.2,   -0.1,  -0.1);
      TERM( 4, 0,0,    0.6,    1.4,    -3.0,   -0.2,   -0.6,   1.6);
      TERM( 4,-1,0,    1.5,   -2.5,    12.4,    4.7,    1.0,  -1.1);
      TERM( 4,-2,0, -821.9,   -9.6,   -26.0, 1873.6,  -70.5,  -4.4);
      TERM( 4,-2,1,    4.1,  -21.9,   -50.3,   -9.9,    0.7,  -3.0);
      TERM( 4,-3,0,   -2.0,   -4.7,   -19.3,    8.2,   -0.1,  -0.3);
      TERM( 4,-4,0,   -1.5,    1.3,     6.5,    7.3,    0.0,   0.0);
      TERM( 5,-2,0,-2627.6,-1277.3,   117.4, -344.1,  -13.8,  -4.3);
      TERM( 5,-2,1,   63.0,  -98.6,    12.7,    6.7,    0.1,  -0.2);
      TERM( 5,-2,2,    1.7,    1.2,    -0.2,    0.3,    0.0,   0.0);
      TERM( 5,-3,0,    0.4,   -3.6,   -11.3,   -1.6,    0.0,  -0.3);
      TERM( 5,-4,0,   -1.4,    0.3,     1.5,    6.3,   -0.1,   0.0);
      TERM( 5,-5,0,    0.3,    0.6,     3.0,   -1.7,    0.0,   0.0);
      TERM( 6,-2,0, -146.7,  -73.7,   166.4, -334.3,  -43.6, -46.7);
      TERM( 6,-2,1,    5.2,   -6.8,    15.1,   11.4,    1.7,  -1.0);
      TERM( 6,-3,0,    1.5,   -2.9,    -2.2,   -1.3,    0.1,  -0.1);
      TERM( 6,-4,0,   -0.7,   -0.2,    -0.7,    2.8,    0.0,   0.0);
      TERM( 6,-5,0,    0.0,    0.5,     2.5,   -0.1,    0.0,   0.0);
      TERM( 6,-6,0,    0.3,   -0.1,    -0.3,   -1.2,    0.0,   0.0);
      TERM( 7,-2,0,   -9.6,   -3.9,     9.6,  -18.6,   -4.7,  -5.3);
      TERM( 7,-2,1,    0.4,   -0.5,     1.0,    0.9,    0.3,  -0.1);
      TERM( 7,-3,0,    3.0,    5.3,     7.5,   -3.5,    0.0,   0.0);
      TERM( 7,-4,0,    0.2,    0.4,     1.6,   -1.3,    0.0,   0.0);
      TERM( 7,-5,0,   -0.1,    0.2,     1.0,    0.5,    0.0,   0.0);
      TERM( 7,-6,0,    0.2,    0.0,     0.2,   -1.0,    0.0,   0.0);
      TERM( 8,-2,0,   -0.7,   -0.2,     0.6,   -1.2,   -0.4,  -0.4);
      TERM( 8,-3,0,    0.5,    1.0,    -2.0,    1.5,    0.1,   0.2);
      TERM( 8,-4,0,    0.4,    1.3,     3.6,   -0.9,    0.0,  -0.1);
      TERM( 9,-4,0,    4.0,   -8.7,   -19.9,   -9.9,    0.2,  -0.4);
      TERM( 9,-4,1,    0.5,    0.3,     0.8,   -1.8,    0.0,   0.0);
      TERM(10,-4,0,   21.3,  -16.8,     3.3,    3.3,    0.2,  -0.2);
      TERM(10,-4,1,    1.0,    1.7,    -0.4,    0.4,    0.0,   0.0);
      TERM(11,-4,0,    1.6,   -1.3,     3.0,    3.7,    0.8,  -0.2);
    END;

  PROCEDURE PERTURA;  (* perturbations by Uranus *)
    VAR I: INTEGER;
    BEGIN
      C[-1]:=COS(M7); S[-1]:=-SIN(M7);
      FOR I:=-1 DOWNTO -4 DO ADDTHE(C[I],S[I],C[-1],S[-1],C[I-1],S[I-1]);
      TERM( 0,-1,0,    1.0,    0.7,     0.4,   -1.5,    0.1,   0.0);
      TERM( 0,-2,0,    0.0,   -0.4,    -1.1,    0.1,   -0.1,  -0.1);
      TERM( 0,-3,0,   -0.9,   -1.2,    -2.7,    2.1,   -0.5,  -0.3);
      TERM( 1,-1,0,    7.8,   -1.5,     2.3,   12.7,    0.0,   0.0);
      TERM( 1,-2,0,   -1.1,   -8.1,     5.2,   -0.3,   -0.3,  -0.3);
      TERM( 1,-3,0,  -16.4,  -21.0,    -2.1,    0.0,    0.4,   0.0);
      TERM( 2,-1,0,    0.6,   -0.1,     0.1,    1.2,    0.1,   0.0);
      TERM( 2,-2,0,   -4.9,  -11.7,    31.5,  -13.3,    0.0,  -0.2);
      TERM( 2,-3,0,   19.1,   10.0,   -22.1,   42.1,    0.1,  -1.1);
      TERM( 2,-4,0,    0.9,   -0.1,     0.1,    1.4,    0.0,   0.0);
      TERM( 3,-2,0,   -0.4,   -0.9,     1.7,   -0.8,    0.0,  -0.3);
      TERM( 3,-3,0,    2.3,    0.0,     1.0,    5.7,    0.3,   0.3);
      TERM( 3,-4,0,    0.3,   -0.7,     2.0,    0.7,    0.0,   0.0);
      TERM( 3,-5,0,   -0.1,   -0.4,     1.1,   -0.3,    0.0,   0.0);
    END;

  PROCEDURE PERTNEP;  (* perturbations by Neptune *)
    BEGIN
      C[-1]:=COS(M8); S[-1]:=-SIN(M8);
      ADDTHE(C[-1],S[-1],C[-1],S[-1],C[-2],S[-2]);
      TERM( 1,-1,0,   -1.3,   -1.2,     2.3,   -2.5,    0.0,   0.0);
      TERM( 1,-2,0,    1.0,   -0.1,     0.1,    1.4,    0.0,   0.0);
      TERM( 2,-2,0,    1.1,   -0.1,     0.2,    3.3,    0.0,   0.0);
    END;

  PROCEDURE PERTJUR;  (* perturbations by Jupiter and Uranus *)
    VAR PHI,X,Y: REAL;
    BEGIN
      PHI:=(-2*M5+5*M6-3*M7); X:=COS(PHI); Y:=SIN(PHI);
      DL:=DL-0.8*X-0.1*Y; DR:=DR-0.2*X+1.8*Y; DB:=DB+0.3*X+0.5*Y;
      ADDTHE(X,Y,C6[1],S6[1],X,Y);
      DL:=DL+(+2.4-0.7*T)*X+(27.8-0.4*T)*Y; DR:=DR+2.1*X-0.2*Y;
      ADDTHE(X,Y,C6[1],S6[1],X,Y);
      DL:=DL+0.1*X+1.6*Y; DR:=DR-3.6*X+0.3*Y; DB:=DB-0.2*X+0.6*Y;
    END;

  BEGIN  (* SAT200 *)

    DL:=0.0; DR:=0.0; DB:=0.0;
    M5:=P2*FRAC(0.0565314+8.4302963*T); M6:=P2*FRAC(0.8829867+3.3947688*T);
    M7:=P2*FRAC(0.3969537+1.1902586*T); M8:=P2*FRAC(0.7208473+0.6068623*T);
    C6[0]:=1.0; S6[0]:=0.0;  C6[1]:=COS(M6); S6[1]:=SIN(M6);
    FOR I:=2 TO 11 DO ADDTHE(C6[I-1],S6[I-1],C6[1],S6[1],C6[I],S6[I]);
    PERTJUP; PERTURA; PERTNEP; PERTJUR;
    L:= 360.0*FRAC(0.2561136 + M6/P2 + ((5018.6+T*1.9)*T +DL)/1296.0E3 );
    R:= 9.557584 - 0.000186*T  +  DR*1.0E-5;
    B:= ( 175.1 - 10.2*T + DB ) / 3600.0;

  END;   (* SAT200 *)

