/*
 *	File: ad2YuSi.c
 *
 *      (C) IWTS
 *          KU Nijmegen
 *          The Netherlands
 *
 *      Author: R. Harald Baayen
 *
 *      History:
 *
 *      - jul 1997, version 1.0
 *
 *      Description:
 *
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <malloc.h>
#include <math.h>
#include "lex_cons.h"


/* EXTERN FUNCTIONS */

/* functions for numerical procedures */

extern double	expV ();
extern double	expVm ();
extern double	funcV ();
extern double	funcVm ();
extern double	getlexprop ();
extern double	adjustV ();
extern double	adjustVm ();
extern double	qromb ();

/* argument reading, file manipulation, and help function */

extern int	leesgetal ();
extern double	leesdouble ();
extern void	change_extension ();
extern void	help ();


/* GLOBAL VARIABLES */

double   N, V, n1, n2,              /* number of tokens, types, hapaxes, disleg */
         E,                         /* extrapolation sample size */
         pstar,                     /* maximum relative frequency */
         mmax,                      /* highest frequency rank */
         mrank,                     /* word frequency, used in expVm() */
         SPECTRUM[MAXM1][3],        /* frequency spectrum m Vm EVm */
         Z, VZ,                     /* Zipf size, and V(Z) */
         VZparam,                   /* VZ as a third parameter of the model */
         Nzero, Vzero,              /* original sample size N0 */
         eV, eV1, eV2N, S,          /* E[V(N)], E[V(1,N)], E[V(2N)], S */
         beta,                      /* second parameter of Yule-Simon model */
         OBSERVED[MAXCHUNKS3+1][6], /* V and V1..5 */
         EXPVOCINT[MAXCHUNKS3+1],   /* interpolated vocabulary size */
         lexprop,                   /* proportion specialized words */
         mse,                       /* MSE for specialization fit */
         CHUNKS[MAXCHUNKS3],        /* the chunk sizes */
         chunksize, remainDer,      /* chunk variables */
         x, x1, y, y2, y3, y4, y5,
	 u_bound,		    /* upper integration intervals */
         logt, t;                   /* log(t), t = N/Z */

FILE     *fpspectrum,               /* fpspectrum: datafile: m, Vm */
         *fpexpspect,               /* expected spectrum */
         *fpexpspect2N,             /* spectrum at 2N */
         *fpprofile,                /* observed developmental profile */
         *fpVN,                     /* file with E[V(N)] and E[V(2N)] */
         *fpsum,                    /* file with summary of model */
         *fpsum2,                   /* file with summary of unadjusted model */
         *fpint,                    /* interpolation results */
         *fpext,                    /* extrapolation results */
         *fpE;                      /* spectrum at sample size E */

int      nranks,                    /* number of different ranks in spectrum */
         maxrank,                   /* largest rank for fit, default 15 */
         i, j,                      /* counter */
         header,                    /* boolean for presence of header */
         k,                         /* variable for chunks */
         nchunks,                   /* number of chunks for interpolation */
         enchunks,                  /* number of chunks for extrapolation */
         token, type,               /* auxiliary variables for scanf */
         infinite,                  /* boolean on value of S */
         again,                     /* boolean for manual search for params */
         aantal;                    /* for command line options */

char     woord[MAXWORDLENGTH],   
         new_name[MAXWORDLENGTH],
         base_name[MAXWORDLENGTH],
         c,                          /* for reading from stdin */
         *fs;                        /* variable for scanning options */


/* MAIN () */

int main (argc, argv)
int     argc;
char    *argv[];

{ 
   /* DEFAULTS */

   maxrank = DEF_MAXRANK;
   nchunks = DEF_CHUNKS;
   enchunks = DEF_CHUNKS;
   header = 1;
   E = NULL_F;
   Z = NULL_F;
   infinite = 0;  /* default: S is not infinite */
   u_bound = U_BOUND;

   /* COMMAND LINE OPTIONS */

   while ((--argc > 0) && ((*++argv)[0] == '-')) {
        for (fs = argv[0] + 1; *fs != '\0'; fs++) {
            switch (*fs) {
            case 'h':
                help();
                break;
            case 'Z':
                Z =  leesdouble (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'E':
                i =  leesgetal (fs, &aantal);
                E = (double) i;
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'k':
                nchunks = leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'K':
                enchunks = leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'i':
                u_bound = (double) leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'm':
                maxrank = leesgetal (fs, &aantal);
                for (; aantal > 0; aantal--){
                   fs++;
                }
                break;
            case 'H':      /* input files without headers! */
                header = 0;
                break;
            default:
                fprintf (stderr, "ad2YuSi: illegal option %c\n", *fs);
                exit (1);
                break;
            }
        }
   } /* of while */


   /* FILE HANDLING */

   if (argc == 0) {
     help ();
   }

   /* load input spectrum, should have .spc extension */

   if ((fpspectrum = fopen(*argv, "r")) == NULL) {
       fprintf(stderr, "ad2YuSi: can't open %s\n", *argv);
       exit(1);
   }

   strncpy (base_name, *argv, strlen(*argv) - 4);

   change_extension (base_name, new_name, ".obs");

   if ((fpprofile = fopen (new_name, "r")) == NULL) {
       fprintf (stderr, "ad2YuSi: can't open profile file %s\n", new_name);
       exit (1);
   }

   /* file name handling output files */

   change_extension(base_name, new_name, "_bY.spc");
   if ((fpexpspect = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2YuSi: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bY.sp2");
   if ((fpexpspect2N = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2YuSi: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bY.ev2");
   if ((fpVN = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2YuSi: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bY.sum");
   if ((fpsum = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2YuSi: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bY.int");
   if ((fpint = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2YuSi: can't open output file %s\n", new_name);
      exit(1);
   }
   change_extension (base_name, new_name, "_bY.ext");
   if ((fpext = fopen(new_name, "w")) == NULL){
      fprintf(stderr, "ad2YuSi: can't open output file %s\n", new_name);
      exit(1);
   }

   if (E > NULL_F){
     change_extension (base_name, new_name, "_bY.sp3");
     if ((fpE = fopen(new_name, "w")) == NULL){
        fprintf(stderr, "ad2YuSi: can't open output file %s\n", new_name);
        exit(1);
     }
   }

   change_extension (base_name, new_name, "_Y.sum");
   if ((fpsum2 = fopen(new_name, "r")) == NULL){
      fprintf(stderr, "ad2YuSi: can't open input file %s\n", new_name);
      exit(1);
   }

   /* LOAD SUMMARY FILE OF UNADJUSTED MODEL */

   /* Z beta VZ: 10, 11, 12 in summary files */
   /* scan title line of summary file */
   for (i = 1; i <= 11; i++) fscanf(fpsum2, "%s ", woord);
   /* scan 8 lines of summary data */
   for (i = 1; i <= 8; i++) {
       fscanf(fpsum2, "%s %s %s", woord, woord, woord);
       fprintf(stderr, "%s\n", woord);
   }
   fscanf(fpsum2, "%s ", woord);
   if (strcmp("Z", woord)!=0){
     fprintf(stderr, "Yule-Simon summary file has wrong format\n");
     fprintf(stderr, "Possibly, cost function C2 was used instead of C1\n");
     exit(1);
   }
   fscanf(fpsum2, "%s %lf",  woord, &Z);
   fscanf(fpsum2, "%s %s %lf", woord, woord, &beta);
   fscanf(fpsum2, "%s %s %lf", woord, woord, &VZ);
   fclose(fpsum2);
   fprintf(stdout, "read summary file, Z = %f, beta = %f, VZ = %f\n", \
	Z, beta, VZ);
   fflush(stdout);

/*

Yule-Simon Zipfian model for alice.book.spc
N         =        26505
V(N)      =         2647
E[V(N)]   =    2647.0029
V(1,N)    =         1171
E[V(1,N)] =    1170.9997
V(2,N)    =          403
E[V(2,N)] =     372.5773
S         = infinite
Z         =    747.51898193
beta      =    0.6302334666
VZ        =          415.29     

*/
   /* LOAD SPECTRUM FILE */

   
   nranks = 0; n1 = 0; n2 = 0;
   if (header){
      fscanf(fpspectrum, "%s ", woord);  /* m */
      fscanf(fpspectrum, "%s ", woord);  /* Vm */
   }

   while (fscanf(fpspectrum, "%d %d", &token, &type) != EOF)  {
        nranks++;
        SPECTRUM[nranks][0] = (double) token;
        SPECTRUM[nranks][1] = (double) type;
        if (token == 1) n1 = type;
        if (token == 2) n2 = type;
        N+= (double) token * (double) type;
        V+= (double) type;
   }
   mmax = SPECTRUM[nranks][0];
   pstar = mmax/N;
   Nzero = N; Vzero = V;


   /* CALCULATE E[V(N)] AND E[V(m,N)] ;  NOTE: S IS INFINITE */

   fprintf (stdout, "computing expected spectrum at N\n");
   fflush(stdout);
   /* VZ = Z/(beta *log(pstar*Z)); */
   eV = expV (N);
   S = NULL_F;
   for (i = 1; i <= maxrank; i++) {
     fprintf (stdout, "[%d]\n", i);
     fflush (stdout);
     SPECTRUM[i][2] = expVm ((double) i, Nzero);
     /*     if (i <= 5) VMZERO[i] = SPECTRUM[i][2]; */
   }
   fprintf (stdout, "\n");
   fflush (stdout);


   /* AND DETERMINE THE INTERPOLATED VALUED OF E[V(N)] */

   /* the chunksizes */

   chunksize = floor(Nzero/(nchunks*1.0));
   remainDer = Nzero - ((nchunks*1.0) * chunksize);
   for (k = 1; k <= nchunks; k++)   CHUNKS[k] = chunksize;
   for (k = 1; k <= remainDer; k++) CHUNKS[k]++;
   for (k = 2; k <= nchunks; k++)   CHUNKS[k] += CHUNKS[k-1];

   /* interpolating */
   fprintf(stdout, "computing interpolation statistics unadjusted model\n");

   fflush(stdout);
   for (k = 1; k <= nchunks; k++){
       fprintf(stdout, "[%d]\n", k);
       fflush(stdout);
       EXPVOCINT[k] = expV(CHUNKS[k]);
   }
   fprintf(stdout, "\n");

   /* read observed profile */

   fprintf(stdout, "reading %s.obs\n",base_name);
   fflush(stdout);
   if (header){
       for (i = 1; i <= NMEASURES2; i++){
          if (fscanf(fpprofile, "%s ", woord) == EOF){
            fprintf(stderr, "incomplete header for %s.obs\n", base_name);
          }
       }
   }

   for (i = 1; i <= nchunks; i++){
       if (fscanf(fpprofile, "%d ", &type) != EOF){
           ;
       }
       else{
           fprintf(stderr, "%s.obs is incompatible with required number of chunks\n", base_name);
           exit(1);
       }
       for (j = 2; j <= NMEASURES2; j++){
           if (fscanf(fpprofile, "%lf ", &x) != EOF){
               switch (j){
                 case 4:
                   OBSERVED[i][0] = x;
                   break;
                 case 5:
                   OBSERVED[i][1] = x;
                   break;
                 case 6:
                   OBSERVED[i][2] = x;
                   break;
                 case 7:
                   OBSERVED[i][3] = x;
                   break;
                 case 8:
                   OBSERVED[i][4] = x;
                   break;
                 case 9:
                   OBSERVED[i][5] = x;
                   break;
                 default:
                   break;
              }
           }
           else{
               fprintf(stderr,
                "%s.obs is incompatible with required number of chunks\n",
                base_name);
               exit(1);
           }
       }
   }

   /* determine optimal specialization proportion */

   fprintf(stdout, "starting estimation of p\n");
   fflush(stdout);

   lexprop = getlexprop ();

   /* PRINT SUMMARY */

   fprintf(fpsum, "Partition-adjusted Yule-Simon model for %s\n", *argv);
   fprintf(fpsum, "N         = %12d\n", (int) N);
   fprintf(fpsum, "V(N)      = %12d\n", (int) V);
   fprintf(fpsum, "E[V(N)]   = %12.4f\n", eV);
   fprintf(fpsum, "V(1,N)    = %12d\n", (int) n1);
   fprintf(fpsum, "E[V(1,N)] = %12.4f\n", SPECTRUM[1][2]);
   fprintf(fpsum, "Z         = %15.8f\n",  Z);
   fprintf(fpsum, "beta      = %15.10f\n",  beta);
   fprintf(fpsum, "VZ        = %15.10f\n",  VZ);

   if (beta < 1){
       fprintf(fpsum, "S         = infinite\n");
   }
   else{
       S = VZ * beta/(beta-1.0);
       fprintf(fpsum, "S         = %12.5f\n", S);
   }

   fprintf(fpsum, "p         = %12.8f\n", lexprop);
   fprintf(fpsum, "MSE       = %12.8f\n", mse);
   fclose(fpsum);

   /* PRINT SPECTRUM */

   fprintf (fpexpspect, "         m         Vm        EVm     alphaM    EalphaM\n");
   for (i = 1; i <= maxrank; i++) {
    fprintf (fpexpspect, "%10d %10d ",(int) SPECTRUM[i][0],(int) SPECTRUM[i][1]);
    fprintf (fpexpspect, "%10.4f %10.4f %10.4f\n", SPECTRUM[i][2],
             SPECTRUM[i][1] / Vzero, SPECTRUM[i][2] / eV);
   }
   fclose (fpexpspect);

   /* PRINT SPECTRUM AT 2N */

   fprintf (stdout, "computing expected spectrum at 2N\n");
   fflush (stdout);
   N = 2 * Nzero;
   eV2N = adjustV (expV (N), N, lexprop, Nzero, Vzero);
   fprintf (fpexpspect2N, "         m      EVm2N\n");
   for (i = 1; i <= 2 * maxrank; i++) {
     fprintf (stdout, "[%d]\n", i);
     fflush (stdout);
     fprintf (fpexpspect2N, "%10d %10.2f\n", i, 
              adjustVm (expVm ((double) i, N), N, lexprop, Nzero, SPECTRUM[i][2]));
   }
   fprintf (stdout, "\n");
   fflush (stdout);
   fclose (fpexpspect2N);

   /* PRINT VOCABULARY SIZES */

   fprintf (fpVN, "       V       EV     EV2N\n");
   fprintf (fpVN, "%8.2f %8.2f %8.2f\n", V, eV, adjustV (eV2N, 2 * Nzero, lexprop, Nzero, Vzero));
   fclose (fpVN);


   /* INTERPOLATION */

   if (nchunks > 0){

     /* CALCULATE THE TEXT CHUNKS */

     chunksize = floor(Nzero/(nchunks*1.0));
     remainDer = Nzero - ((nchunks*1.0) * chunksize);
     for (k = 1; k <= nchunks; k++)   CHUNKS[k] = chunksize;
     for (k = 1; k <= remainDer; k++) CHUNKS[k]++;
     for (k = 2; k <= nchunks; k++)   CHUNKS[k] += CHUNKS[k-1];

     /* AND PRINT THE CORRESPONDING STATISTICS */

     fprintf (stdout, "computing interpolation statistics\n");
     fflush (stdout);
     fprintf (fpint, "       N       EV   Alpha1      EV1      EV2      EV3      EV4      EV5       GV\n");
     for (k = 1; k <= nchunks; k++) {
        fprintf (stdout, "[%d]\n", k);
        fflush (stdout);
        x = adjustV (expV (CHUNKS[k]), CHUNKS[k], lexprop, Nzero, Vzero);
        x1 = adjustV (expV (CHUNKS[k] + 1.0), CHUNKS[k], lexprop, Nzero, Vzero);
        y =  adjustVm (expVm(1.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[1][2]);
        y2 = adjustVm (expVm(2.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[2][2]);
        y3 = adjustVm (expVm(3.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[3][2]);
        y4 = adjustVm (expVm(4.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[4][2]);
        y5 = adjustVm (expVm(5.0, CHUNKS[k]), CHUNKS[k], lexprop, Nzero, SPECTRUM[5][2]);
        fprintf (fpint, "%8.2f %8.2f %8.4f %8.2f %8.2f %8.2f %8.2f %8.2f %8.4f\n", 
                        CHUNKS[k], x, y / x, y, y2, y3, y4, y5, x1 - x);
     }
     fprintf (stdout, "\n");
     fflush (stdout);
   }

   /* EXTRAPOLATION */
  
   if (E == NULL_F) {  /* extrapolate to 2N */
     fprintf (stdout, "computing extrapolation statistics to 2N\n");
     fflush (stdout);
     fprintf (fpext, "         N         EV        EV1      EV2      EV3      EV4      EV5       GV\n");

     for (k = 1; k <= nchunks; k++) {
        fprintf (stdout, "[%d]\n", k);
        fflush (stdout);
        x = adjustV (expV (Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, Vzero);
        x1 = adjustV (expV (Nzero + CHUNKS[k]+1.0), Nzero + CHUNKS[k] + 1.0, lexprop, Nzero, Vzero);
        y = adjustVm (expVm (1.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[1][2]);
        y2 = adjustVm (expVm (2.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[2][2]);
        y3 = adjustVm (expVm (3.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[3][2]);
        y4 = adjustVm (expVm (4.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[4][2]);
        y5 = adjustVm (expVm (5.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[5][2]);
        fprintf (fpext, "%10.2f %10.2f %10.4f ", Nzero + CHUNKS[k],  x, y);
        fprintf (fpext, "%8.2f %8.2f %8.2f %8.2f %8.4f\n", y2, y3, y4, y5,x1-x);
     }
     fprintf (stdout, "\n");
     fflush (stdout);
   }
   else {
     fprintf (stdout, "ad2YuSi: warning: this part is not yet implemented\n");
     fflush (stdout);
     exit (1);

     /* FIND NEW CHUNKSIZES */

     fprintf(stderr, "computing extrapolation statistics to E\n");
     chunksize = floor((E-Nzero)/(enchunks*1.0));
     remainDer = (E-Nzero) - ((enchunks*1.0) * chunksize);
     for (k = 1; k <= enchunks; k++)   CHUNKS[k] = chunksize;
     for (k = 1; k <= remainDer; k++)  CHUNKS[k]++;
     for (k = 2; k <= enchunks; k++)   CHUNKS[k] += CHUNKS[k-1];

     /* PRINT THE GROWTH CURVE */

	      fprintf (stdout, "computing extrapolation statistics to 2N\n");
     fflush (stdout);
     fprintf (fpext, "         N         EV        EV1      EV2      EV3      EV4      EV5       GV\n");

     for (k = 1; k <= enchunks; k++) {
        fprintf (stdout, "[%d]\n", k);
        fflush (stdout);
        x = adjustV (expV (Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, Vzero);
        x1 = adjustV (expV (Nzero + CHUNKS[k]+1.0), Nzero + CHUNKS[k] + 1.0, lexprop, Nzero, Vzero);
        y = adjustVm (expVm (1.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[1][2]);
        y2 = adjustVm (expVm (2.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[2][2]);
        y3 = adjustVm (expVm (3.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[3][2]);
        y4 = adjustVm (expVm (4.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[4][2]);
        y5 = adjustVm (expVm (5.0, Nzero + CHUNKS[k]), Nzero + CHUNKS[k], lexprop, Nzero, SPECTRUM[5][2]);
        fprintf (fpext, "%10.2f %10.2f %10.4f ", Nzero + CHUNKS[k],  x, y);
        fprintf (fpext, "%8.2f %8.2f %8.2f %8.2f %8.4f\n", y2, y3, y4, y5,x1-x);
     }
     fprintf (stdout, "\n");
     fflush (stdout);


     /* AND SHOW THE SPECTRUM AT E */

     eV2N = expV(E);
     fprintf(fpE, "         m      EVmXN\n");
     for (i = 1; i <= maxrank; i++){
       fprintf(fpE, "%10d %10.2f\n", i, expVm((double) i, E));
     }
   }

   return (0);
} /* end of main */



double expV (n)
double n;
{
  N = n;
  t = N / Z;
  return (qromb (funcV, L_BOUND, u_bound) * VZ * beta);
}


double expVm (m, n)
double m, n;  
{
  mrank = m;
  N = n;
  t = N / Z;
  logt = log (t);
  return (qromb (funcVm, L_BOUND, u_bound) * VZ * beta);

}


double funcVm (x)
double x;
{
 return( (exp(mrank*logt) * x) / (exp((mrank+1.0)*log(t+x)) * exp(beta*log(1+x))));
}


double funcV (x)
double x;
{
 return( t / ((t+x) * exp(beta*log(1+x))));
}


void help ()
{
  fprintf (stderr, "ad2YuSi text.spc\n");
  fprintf (stderr, "OPTIONS:\n");
  fprintf (stderr, "     -h: display help\n");
  fprintf (stderr, "     -m: number of ranks in fit (default: 15)\n");
  fprintf (stderr, "     -k: number of chunks for interpolation (default: 20)\n");
  fprintf (stderr, "     -K: number of chunks for extrapolation (default: 20)\n");
  fprintf (stderr, "     -E: extrapolation sample size (default: 2N)\n");
  fprintf (stderr, "     -H: input files lack header (default: with header)\n");
  fprintf (stderr, "INPUT:\n");
  fprintf (stderr, "     text.spc:  m Vm\n");
  fprintf (stderr, "     text.obs should be available for reading \n");
  fprintf (stderr, "OUTPUT:\n");
  fprintf (stderr, "     text_bY.spc:      expected spectrum\n");
  fprintf (stderr, "     text_bY.sp2:      expected spectrum at 2N\n");
  fprintf (stderr, "     text_bY.ev2:      E[V(N)] and E[V(2N)]\n");
  fprintf (stderr, "     text_bY.sum:      summary, fitted parameters\n");
  fprintf (stderr, "     text_bY.int:      interpolation results\n");
  fprintf (stderr, "     text_bY.ext:      extrapolation results\n");
  exit (1);
}


double getlexprop ()
{
     double expvoc, adjvoc, optp, p, t, diff;
     int i;

     /* use vocabulary growth curve for fit */

     mse = MAXX;
     optp = NULL_F;
     for (p = 0.01; p <= 1.0; p += 0.01){
       x = NULL_F;
       for (i = 1; i <= nchunks; i++){
          N = CHUNKS[i]; t = N/CHUNKS[nchunks];
          expvoc = EXPVOCINT[i];
          adjvoc = (p * t * Vzero) + (1.0-p)*expvoc;
          diff = (adjvoc-OBSERVED[i][0]);
          x += (diff*diff);
       }
       x = (x/((double) nchunks));
       if (x < mse) {
            mse = x;
            optp = p;
       }
     }
     return(optp);
}
