// ----------------------------------------------------------------------------
// (c) Copyright 2005, Springer.  All Rights Reserved. The code in this CD-ROM is
// distributed by Springer with ABSOLUTELY NO SUPPORT and NO WARRANTY from
// Springer. Use or reproduction of the information provided on this code for
// commercial gain is strictly prohibited. Explicit permission is given for the
// reproduction and use of this information in an instructional setting provided
// proper reference is given to the original source.
//
// Authors and Springer shall not be liable for damage in connection with, or
// arising out of, the furnishing, performance or use of the contents of the
// CD-ROM.
// ----------------------------------------------------------------------------

#include "pl3RxGen.h"
#include "TestBuilder.h"


// *****************************************************************
//                         pp3RxGen Task
// *****************************************************************

//
// Tvm pp3RGen Task pp3RxGen Constructor & Destructor
//
pp3RxGenTaskT::pp3RxGenTaskT ( pl3RxGenTvmT & tvm ) :
    tbvTaskTypeSafeT<tbvCellT> ( &tvm, "pp3RxGen" ),

    // TVM reference
    pp3RGen ( tvm )
{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

pp3RxGenTaskT::~pp3RxGenTaskT () {}

//
// Tvm pp3RGen Task pp3RxGen body method
//
void pp3RxGenTaskT::body ( tbvCellT *pp3RxGenArgP )
{
  tbvOut << pp3RGen.getInstanceNameP() << " is called @ ";
  tbvOut << tbvGetInt64Time(TBV_NS) << " ns\n";

  //
  // Lock mutex to do one transaction at a time
  //
  pp3RGen.pp3RxGenMutex.lock();

  vector<uint32> data = pp3RxGenArgP->getDataBy32();
  bool prty = 0;

  tbvSmartUnsignedT badPrtyRand;
  badPrtyRand.keepOnly(1,100);

  // Signals change only on rising edge of clk
  if (pp3RGen.pp3rfclk == 0) {
    tbvWaitCycle(pp3RGen.pp3rfclk, tbvThreadT::POSEDGE);
    tbvWait(1.5, TBV_NS); // Minimum hold time
  }

  // Pl3 Enable controls the sender
  if (pp3RGen.pp3renb_n != 0)
    pp3RGen.handleBackpressure();

  tbvSignalT dataSig(31,0); // Used because easier to calculate parity

  if (pp3RxGenArgP->getValid())
    pp3RGen.pp3RxGenFiber.beginTransactionH("Valid Cell");
  else
    pp3RGen.pp3RxGenFiber.beginTransactionH("Invalid Cell");

  // If port has changed from last time, send new port
  dataSig = pp3RxGenArgP->getInputPort().getUnsignedValue();
  if (dataSig.getValue() != pp3RGen.curr_port) {
    pp3RGen.curr_port = dataSig.getValue();
    pp3RGen.pp3rsx = 1;
    pp3RGen.pp3rval = 0;
    randBit.randomize();
    dataSig = (pp3RxGenArgP->getInputPort().getUnsignedValue()) + ((randBit % 16777216) << 8); // Only lsb 8-bits should be considered
    pp3RGen.pp3rdat = dataSig.getValue();

    prty = dataSig.reductionXor();
    badPrtyRand.randomize();
    if (badPrtyRand <= pp3RGen.badParity)
      prty = prty ^ 1;
    if (pp3RGen.prtySense)
      pp3RGen.pp3rprty = prty ^ 1;
    else
      pp3RGen.pp3rprty = prty;

    // Randomize don't-care signals
    randBit.randomize();
    pp3RGen.pp3rmod = randBit % 4;
    randBit.randomize();
    pp3RGen.pp3rsop = randBit % 2;
    randBit.randomize();
    pp3RGen.pp3reop = randBit % 2;
    randBit.randomize();
    pp3RGen.pp3rerr = randBit % 2;

    tbvWaitCycle(pp3RGen.pp3rfclk, tbvThreadT::POSEDGE);
    tbvWait(1.5, TBV_NS); // Minimum hold time
  }

  // Send data
  for(unsigned i = 0; i < data.size(); i++) {
    if (tbv4StateNotEqual(pp3RGen.pp3renb_n, 0)) {
      pp3RGen.handleBackpressure();
    }

    pp3RGen.pp3rdat = data[i];

    dataSig = data[i];
    if (pp3RxGenArgP->getValid()) {
      prty = dataSig.reductionXor();
      badPrtyRand.randomize();
      if (badPrtyRand <= pp3RGen.badParity)
	prty = prty ^ 1;
      if (pp3RGen.prtySense)
	pp3RGen.pp3rprty = prty ^ 1;
      else
	pp3RGen.pp3rprty = prty;
    }
    else { // parity ignored when rval, rsx disabled so randomize it
      randBit.randomize();
      pp3RGen.pp3rprty = randBit % 2;
    }      
     
    // Start of burst
    if (i == 0) {
      pp3RGen.pp3rval = pp3RxGenArgP->getValid();
      pp3RGen.pp3rsop = pp3RxGenArgP->getSOP(); 
      if (data.size() >= 2) { // Not end of burst
	pp3RGen.pp3reop = 0;
	pp3RGen.pp3rerr = 0;
	pp3RGen.pp3rmod = 0;
      }
    }

    if (i == 1)  pp3RGen.pp3rsop = 0;

    if (i == (data.size()-1)) {
      pp3RGen.pp3reop = pp3RxGenArgP->getEOP();
      pp3RGen.pp3rmod = (pp3RGen.badMode) ? ((4 - (pp3RxGenArgP->getSize()%4)) % 4) : ((pp3RxGenArgP->getEOP()) ? ((4 - (pp3RxGenArgP->getSize()%4)) % 4) : 0);
      // rerr asserted only when reop is asserted
      if(pp3RxGenArgP->getEOP())
	pp3RGen.pp3rerr = pp3RxGenArgP->getBad();
    }

    // Randomize rsx when rval = 1
    if (pp3RxGenArgP->getValid()) {
      randBit.randomize();
      pp3RGen.pp3rsx = randBit % 2;
    }
    else
      pp3RGen.pp3rsx = 0;
	
    tbvWaitCycle(pp3RGen.pp3rfclk, tbvThreadT::POSEDGE);
    tbvWait(1.5, TBV_NS); // Minimum hold time
 }

  // In case this was the last cell, then, we need to get rval and rsx to 0.
  // However, if renb_n != 0, we first have to wait until its 1 before
  // we do so because rval and rsx have to stay the same while renb_n != 0
  if (pp3RGen.pp3renb_n != 0)
    pp3RGen.handleBackpressure();
  pp3RGen.pp3rval = 0; 
  pp3RGen.pp3rsx = 0; // If data.size() was zero

  // Record the values in the argument block
  pp3RGen.pp3RxGenFiber.recordAttribute(*pp3RxGenArgP, "Cell");

  // Finish Transaction Recording
  pp3RGen.pp3RxGenFiber.endTransaction();

  // release the semaphore
  pp3RGen.pp3RxGenMutex.unlock();

  tbvOut << pp3RGen.getInstanceNameP() << " done @ " << tbvGetInt64Time(TBV_NS) << " ns\n";

}

// *****************************************************************
//                      pp3RGen TVM
// *****************************************************************

//
// TVM pp3RGen Constructor & Destructor
//
pl3RxGenTvmT::pl3RxGenTvmT ( ) :
  tbvTvmT (),

  // Initialize the TVM Signals
        pp3rfclk (getFullInterfaceHdlNameP("pp3rfclk") ),
        rst_l (getFullInterfaceHdlNameP("rst_l") ),
        pp3renb_n (getFullInterfaceHdlNameP("pp3renb_n_r") ),
        pp3rsx (getFullInterfaceHdlNameP("pp3rsx_o") ),
        pp3rsop (getFullInterfaceHdlNameP("pp3rsop_o") ),
        pp3reop (getFullInterfaceHdlNameP("pp3reop_o") ),
        pp3rdat (getFullInterfaceHdlNameP("pp3rdat_o") ),
        pp3rmod (getFullInterfaceHdlNameP("pp3rmod_o") ),
        pp3rprty (getFullInterfaceHdlNameP("pp3rprty_o") ),
        pp3rerr (getFullInterfaceHdlNameP("pp3rerr_o") ),
        pp3rval (getFullInterfaceHdlNameP("pp3rval_o") ),


  // Initialize the Mutex
  pp3RxGenMutex( "pp3RxGenMutex" ),
  pp3RxGenFiber( this, "pp3RxGenFiber"),

  // Create the task objects
  pp3RxGen(* this),

  // Create the Event Expressions
  posedge_clk ( pp3rfclk, tbvThreadT::POSEDGE ),
  pp3renb_n_0(posedge_clk, !pp3renb_n()),

  prtySense(0),
  badParity(0),
  badMode(0),

  curr_port(0),
  backPressureTimeoutNSec(20000)
{
 //add any construtor code needed for tvm pp3RGen
 pp3RxGenFiber.setRecording(TRUE);
 tbvExceptionT::setUserExceptionTypeString("LINK CORE BACKPRESSURE");
 tbvExceptionT::setUserExceptionTypeString("LINK ENABLE LOW TOO LONG");
}

//destructor for tvm pp3RGen
pl3RxGenTvmT::~pl3RxGenTvmT()
{
  //add any clean up code needed for tvm pp3RGen

}

//
// Method called from $tbv_tvm_connect, to create the TVM object.
//
void pl3RxGenTvmT::create ( )
{
  new pl3RxGenTvmT ( );
};


