// ----------------------------------------------------------------------------
// (c) Copyright 2005, Springer.  All Rights Reserved. The code in this CD-ROM is
// distributed by Springer with ABSOLUTELY NO SUPPORT and NO WARRANTY from
// Springer. Use or reproduction of the information provided on this code for
// commercial gain is strictly prohibited. Explicit permission is given for the
// reproduction and use of this information in an instructional setting provided
// proper reference is given to the original source.  
// 
// Authors and Springer shall not be liable for damage in connection with, or
// arising out of, the furnishing, performance or use of the contents of the
// CD-ROM. 
// ----------------------------------------------------------------------------

#include "trafficScheduler.h"

TVM_VERILOG_REGISTER(trafficSchedulerTvmT::create, "trafficSchedulerTvm");

/**************************************************************************/
trafficSchedulerTaskT::trafficSchedulerTaskT( trafficSchedulerTvmT & tvm) :
  backgroundTaskT ( "traffic scheduler"),
  trafficSchedulerTvm(tvm),
  flowOnPort(64, FALSE)
{
  //disable the mutex built into the fiber and turn off automatic
  // transaction recording
    //setSynchronization(FALSE, FALSE);
} 


/**************************************************************************/
void trafficSchedulerTaskT::addTask( interfaceDriverTaskT * taskP) 
{
  tasks.push_back(taskP);
}


/**************************************************************************/
list<interfaceDriverTaskT *> & trafficSchedulerTaskT::getTasks() 
{
  return tasks;
}


/**************************************************************************/
void trafficSchedulerTaskT::body() 
{
//  trafficSchedulerTvm.trafficMutex.lock();
  
  flowMemT & memory = dataProvider->getFlowMem();
  list<unsigned int> portList = memory.getPortList();
  list<unsigned int>::iterator portIter;

//  while (!dataProvider->done()) { 
    bagMutex.lock();
    portList = memory.getPortList();
    for (portIter = portList.begin(); portIter != portList.end(); portIter++) {
      port = *portIter;
      portCredit = memory.getPortParam(port).weight;
      //total time the scheduler can spend on this port
      portBag.add(port, portCredit);
      portBag.resetPeek();
      
      
      tbvBagT<unsigned int> &flowByPortBag_port = flowByPortBag[port];
      if(flowByPortBag_port.empty()){
	  list<unsigned int> & fidList = memory.getFidListOnPort(port);
	  list<unsigned int>::iterator fidIter;	  
	  
	  for (fidIter = fidList.begin(); fidIter != fidList.end(); fidIter++) {
	    fid = *fidIter;
	    //record the fid number
	    fidCredit = memory.getFlow(fid).weight;
	    //total time the scheduler can spend on this fid
	    flowByPortBag_port.add(fid, fidCredit);   
	    flowByPortBag_port.resetPeek();
	  }
      }
    }
    while (!portBag.empty()) {
      port = trafficSchedulerTvm.getUseRandomPeekPort() ? portBag.peekRandom() : portBag.peekNext();
      tbvBagT<unsigned int> & flowByPortBag_port = flowByPortBag[port];
      portBag.remove();
      if (!flowOnPort[port]) {
	if (!flowByPortBag_port.empty()) {
	  fid = trafficSchedulerTvm.getUseRandomPeekFid() ? flowByPortBag_port.peekRandom(TRUE) : flowByPortBag_port.peekNext('u');
	  flowByPortBag_port.remove();
	} else {
	  if  (!portBag.empty()) { portBag.remove(); }
	}
      } else {
	fid = prevFlow[port];

      }
      sendCell(fid, port);      
    }
    
    // Hack -- Since max chip processing rate is 1 cell every 16 clocks
    // This wait time should be used for addCredit and addSystemCredit instead of the interface
    // cycles.  ALso, we may want to associate the EventExpr posEdge with the trafficSchedulerTvm.
    // Or maybe not.
    tbvEventExprT posEdge(trafficSchedulerTvm.clk, tbvThreadT::POSEDGE);
    tbvWait(posEdge*16, tbvEventExprT::RWSYNC);

    bagMutex.unlock();
    
//  trafficSchedulerTvm.trafficMutex.unlock();

  if (dataProvider->done()) {
      stop();
  }
}

/***********************************************************************************************/
void trafficSchedulerTaskT::sendCell(unsigned int fid, unsigned int port) {
    dualLeakyBucketT & leakyBucket = trafficSchedulerTvm.getDualLeakyBucket(fid);
    bool useDualLeakyBucket = trafficSchedulerTvm.getUseDualLeakyBucket();
    
    dataGramT * dataGramP = NULL;
    unsigned int cycles = 0;
    
    dataGramP = dataProvider->frontCell(fid);
    unsigned int dataSize = (dataGramP->getDataComP())->getDataSize();
    cycles = (unsigned int) ceil(dataSize/float(trafficSchedulerTvm.databusWidth));
    
    debugOutput() << dec
        << (leakyBucket.hasCredit(cycles, useDualLeakyBucket) ? "GOOD " : "BAD  ")
        << "@" << tbvGetInt64Time()
        << " fid = " << fid
        << " pk = " << leakyBucket.peakBucket.getCredit()
        << " avg = " << leakyBucket.averageBucket.getCredit()
        << endl;

    if (leakyBucket.hasCredit(cycles, useDualLeakyBucket)) {
        cout << *dataGramP->getDataComP() << endl;
        trafficSchedulerTvm.addCredit(cycles + trafficSchedulerTvm.validOverheadCycles, useDualLeakyBucket);
        leakyBucket.removeCredit(cycles, useDualLeakyBucket);
        if (dataProvider->rotate(dataGramP)) { // Normally, rotate on EOP.
            flowOnPort[port] = false;
        } else {
            flowOnPort[port] = true;
            prevFlow[port] = fid;
        }
        if (dataGramP != NULL)  {
              dataProvider->popCell(fid);
        }

        //update fid, port credit
        fidCredit -= cycles;
        portCredit -= cycles;
    
        // needs to split up into find the right task to push cell to, not all tasks
        list<interfaceDriverTaskT *>::iterator taskIter;
        if (dataGramP != NULL)  {
            for (taskIter = tasks.begin(); taskIter != tasks.end(); taskIter++) {
                (*taskIter)->pushIntoInterfaceDriverQueue(dataGramP);
                
            }
        }
        tbvThreadT::yield();
    } else {
        dataSize = 64; // bad hardcoded magic number of bytes
        cycles = (unsigned int) ceil(dataSize/(float)trafficSchedulerTvm.databusWidth);
        trafficSchedulerTvm.addCredit(cycles + trafficSchedulerTvm.invalidOverheadCycles, useDualLeakyBucket);
    }

    trafficSchedulerTvm.addSystemCredit(cycles);
    
}

/**************************************************************************/
void trafficSchedulerTaskT::connectDataProvider(dataProviderT *dataProvider){
  this->dataProvider = dataProvider;
}

//**************************************************************************
//          trafficSchedulerTvmT
//**************************************************************************


/**************************************************************************/
trafficSchedulerTvmT::trafficSchedulerTvmT() :
  tbvTvmT(),
  trafficMutex( "traffic scheduler mutex" ),
  stallMutex( "traffic scheduler stall mutex" ),
  trafficSchedulerTask(*this),
  rst_l (getFullInterfaceHdlNameP("rst_l") ),
  clk (getFullInterfaceHdlNameP("clk") ),
  databusWidth(8),
  validOverheadCycles(0),
  invalidOverheadCycles(0),
  credit(0),
  useDualLeakyBucket(TRUE),
  useRandomPeek(3,FALSE)
  { }


/**************************************************************************/
void trafficSchedulerTvmT::setLeakyBucketParam( unsigned int fid, float cr,
					     float lr, float d)
{
  if (useDualLeakyBucket) {
    tbvExceptionT::setUserExceptionTypeString("MAL_FUNCTION");
    tbvExceptionT::reportUserException("MAL_FUNCTION",
				       tbvExceptionT::ERROR,
				       "in trafficSchedulerTvmT::setLeakyBucketParam should be called with 7 arguments since dual leaky bucket is used");
  }
  else {
    leakyBuckets[fid].averageBucket.setParameters(cr, lr, d);
  }
}


/**************************************************************************/
void trafficSchedulerTvmT::setLeakyBucketParam( unsigned int fid, float pcr,
						float plr, float pd, float mcr, float mlr, float md)
{
  if (!useDualLeakyBucket) {
    tbvExceptionT::setUserExceptionTypeString("MAL_FUNCTION");
    tbvExceptionT::reportUserException("MAL_FUNCTION",
				       tbvExceptionT::ERROR,
				       "in trafficSchedulerTvmT::setLeakyBucketParam should be called with 4 arguments since dual leaky bucket is NOT used");
  }
  else {
    leakyBuckets[fid].averageBucket.setParameters(mcr,mlr,md);
    leakyBuckets[fid].peakBucket.setParameters(pcr,plr,pd);
  }
}


/**************************************************************************/
void trafficSchedulerTvmT::setTrafficParam( unsigned int fid, float pcr, float cdvt,
                                         float scr, float mbs)
{
  float peakCreditRate = 1;
  float peakLeakyRate = maxOutputRate/pcr;
  float averageCreditRate = 1;
  float averageLeakyRate = maxOutputRate/scr;
  float peakDepth = pcr*cdvt/databusWidth*peakCreditRate;
  float averageDepth = mbs/databusWidth*averageCreditRate;
  setLeakyBucketParam(fid, peakCreditRate, peakLeakyRate, peakDepth, 
		      averageCreditRate, averageLeakyRate, averageDepth);
}

/**************************************************************************/
void trafficSchedulerTvmT::setInterfaceParam(unsigned width, unsigned validOverhead, unsigned invalidOverhead)
{
  databusWidth = width;
  validOverheadCycles = validOverhead;
  invalidOverheadCycles = invalidOverhead;
}

/**************************************************************************/
void trafficSchedulerTvmT::setUseDualLeakyBucket(bool use)
{
  useDualLeakyBucket = use;
}


/**************************************************************************/
bool trafficSchedulerTvmT::getUseDualLeakyBucket()
{
  return useDualLeakyBucket;
}


/**************************************************************************/
void trafficSchedulerTvmT::setUseRandomPeek(bool use)
{
  useRandomPeek[PORT] = use;
  useRandomPeek[FID] = use;
}

/**************************************************************************/
void trafficSchedulerTvmT::setUseRandomPeekPort(bool use)
{
  useRandomPeek[PORT] = use;
}

/**************************************************************************/
void trafficSchedulerTvmT::setUseRandomPeekFid(bool use)
{
  useRandomPeek[FID] = use;
}


/**************************************************************************/
bool trafficSchedulerTvmT::getUseRandomPeek()
{
  return useRandomPeek[PORT] | useRandomPeek[FID];
}

/**************************************************************************/
bool trafficSchedulerTvmT::getUseRandomPeekPort()
{
  return useRandomPeek[PORT];
}

/**************************************************************************/
bool trafficSchedulerTvmT::getUseRandomPeekFid()
{
  return useRandomPeek[FID];
}

/**************************************************************************/
void trafficSchedulerTvmT::addCredit( unsigned int fid, unsigned int c)
{
  leakyBuckets[fid].addCredit(c, useDualLeakyBucket);
}


/**************************************************************************/
void trafficSchedulerTvmT::addCredit(unsigned int c, bool useDualLeakyBucket)
{
  map<unsigned int, dualLeakyBucketT>::iterator fidIter;
  for (fidIter = leakyBuckets.begin(); fidIter != leakyBuckets.end(); fidIter++) {
    (*fidIter).second.addCredit(c, useDualLeakyBucket);
  }
}


/**************************************************************************/
dualLeakyBucketT & trafficSchedulerTvmT::getDualLeakyBucket(unsigned int fid)
{
  return leakyBuckets[fid];
}


/**************************************************************************/
void trafficSchedulerTvmT::addSystemCredit( unsigned int c)
{
  credit += c;
}


/**************************************************************************/
void trafficSchedulerTvmT::setSystemCredit( unsigned int c)
{
  credit = c;
}


/**************************************************************************/
unsigned int trafficSchedulerTvmT::getSystemCredit()
{
  return credit;
}

/**************************************************************************/
void trafficSchedulerTvmT::setMaxOutputRate( float rate)
{
  maxOutputRate = rate;
}


/**************************************************************************/
float trafficSchedulerTvmT::getMaxOutputRate()
{
  return maxOutputRate;
}

/**************************************************************************/
void trafficSchedulerTvmT::create()
{
  new trafficSchedulerTvmT();
}

/***************************************************************************/
void trafficSchedulerTvmT::connectDataProvider(dataProviderT * dataProvider){
  trafficSchedulerTask.connectDataProvider(dataProvider);
}
/**************************************************************************/
