// ----------------------------------------------------------------------------
// (c) Copyright 2005, Springer.  All Rights Reserved. The code in this CD-ROM is
// distributed by Springer with ABSOLUTELY NO SUPPORT and NO WARRANTY from
// Springer. Use or reproduction of the information provided on this code for
// commercial gain is strictly prohibited. Explicit permission is given for the
// reproduction and use of this information in an instructional setting provided
// proper reference is given to the original source.  
// 
// Authors and Springer shall not be liable for damage in connection with, or
// arising out of, the furnishing, performance or use of the contents of the
// CD-ROM. 
// ----------------------------------------------------------------------------

#ifndef TRAFFIC_SCHEDULER_H 
#define TRAFFIC_SCHEDULER_H

#include <math.h>
#include <list.h>
#include <function.h>
#include "dataCom.h"
#include "dataPayload.h"
#include "flowMem.h"
#include "dataProvider.h"
#include "leakyBuckets.h"
#include "interfaceDriverTask.h"
#include "macros.h"
/// \defgroup trafficGenerator generator parts
/// traffic scheduler

///  This class is used to control the input(output of the generator) rate of a flow
///
///  \ingroup trafficGenerator


/// This task generates cells and calls the appropriate interface 
/// task to drive the cells into the chip
/// \ingroup trafficGenerator

class trafficSchedulerTvmT;

class trafficSchedulerTaskT : public backgroundTaskT {
public:
  
  trafficSchedulerTaskT( trafficSchedulerTvmT&  );
  ~trafficSchedulerTaskT() { }
  
  void 			body ( );
  void				addTask(interfaceDriverTaskT * taskP);
  list<interfaceDriverTaskT *> &		getTasks();
  
  //parent tvm, make it easy to access in task body
   trafficSchedulerTvmT & 	trafficSchedulerTvm;
  void connectDataProvider(dataProviderT *dataProvider);
protected:
  void sendCell(unsigned int fid, unsigned int port);
  
private:
  /// the tasks(interface tasks) to be spawmed when traffic is scheduled.
  list<interfaceDriverTaskT *> 		tasks;
  /// true if the simulation is over(No more traffic available from dataProvider)
  map<unsigned int, tbvBagT<unsigned int> > flowByPortBag;
  tbvBagT<unsigned int> portBag;
  vector<bool> flowOnPort;
  map<unsigned int, unsigned int> prevFlow;
  float portCredit, fidCredit;
  unsigned int port, fid;
  tbvMutexT bagMutex;
  dataProviderT *dataProvider;
};


/// Top level class for implementing generator

/// This is the TVM class for generator.  It uses the
/// \a trafficScedulerTaskT class for constructing and
/// sending cells over the appropriate interface.

/// \ingroup trafficGenerator

class trafficSchedulerTvmT : public tbvTvmT {
public:
  trafficSchedulerTvmT();
  ~trafficSchedulerTvmT() { }
  
  void 		setLeakyBucketParam(unsigned int fid, float cr, float lr, float d);
  void 		setLeakyBucketParam(unsigned int fid, float pcr, float plr, float pd,
				    float mcr, float mlr, float md);
  void 		setTrafficParam(unsigned int fid, float pcr, float cdvt, float scr,
				float mbs);
  void          setInterfaceParam(unsigned databusWidth, unsigned validOverheadCycles,
				  unsigned invalidOverheadCycles);
  
  void				setUseDualLeakyBucket(bool);
  bool				getUseDualLeakyBucket();
  void				setUseRandomPeek(bool);
  bool				getUseRandomPeek();
  void				setUseRandomPeekPort(bool);
  bool				getUseRandomPeekPort();
  void				setUseRandomPeekFid(bool);
  bool				getUseRandomPeekFid();
  dualLeakyBucketT & 		getDualLeakyBucket(unsigned int fid);
  void 				addCredit(unsigned int fid, unsigned int c);
  void 				addCredit(unsigned int c, bool useDualLeakyBucket);
  void				addSystemCredit(unsigned int c);
  void				setSystemCredit(unsigned int c);
  unsigned int			getSystemCredit();
  void				setMaxOutputRate(float rate);
  float				getMaxOutputRate();
  void 				setOutputInstanceBurstSize(tbvSmartUnsignedT * size);
  static void			create();
  void                          connectDataProvider(dataProviderT *dataProvider);
  /// used to serialize the datastream from the generator
  tbvMutexT 			trafficMutex;
  
  /// the traffic generation will stall until it can lock the stallMutex
  tbvMutexT 			stallMutex;
  trafficSchedulerTaskT		trafficSchedulerTask;
  
  tbvSignalHdlT   rst_l ;
  tbvSignalHdlT   clk ;
  
  unsigned                              databusWidth;
  unsigned                              validOverheadCycles;
  unsigned                              invalidOverheadCycles;
private:
  /// hash map used to store the leaky bucket for each flow id 
  map<unsigned int, dualLeakyBucketT>	leakyBuckets;
  
  /// total credit since the beginning of clock
  unsigned int 				credit;
  
  /// if true, use dual leakyBucket
  bool 					useDualLeakyBucket;
  
  /// if true, use random peek when looking for the input type from the datacomTypes or 
  /// the datacomRefs bags.
  vector<bool> 					useRandomPeek;
  enum randomPeekTypes { PORT, FID };
  
  float					maxOutputRate;
  
};  

#endif // class tbvTrafficGenT
