// ----------------------------------------------------------------------------
// (c) Copyright 2005, Springer.  All Rights Reserved. The code in this CD-ROM is
// distributed by Springer with ABSOLUTELY NO SUPPORT and NO WARRANTY from
// Springer. Use or reproduction of the information provided on this code for
// commercial gain is strictly prohibited. Explicit permission is given for the
// reproduction and use of this information in an instructional setting provided
// proper reference is given to the original source.
//
// Authors and Springer shall not be liable for damage in connection with, or
// arising out of, the furnishing, performance or use of the contents of the
// CD-ROM.
// ----------------------------------------------------------------------------

#include "pl3TxGen.h"
#include "TestBuilder.h"


// *****************************************************************
//                         pl3TxGen Task
// *****************************************************************

//
// Tvm pl3TGen Task pl3TxGen Constructor & Destructor
//
pl3TxGenTaskT::pl3TxGenTaskT ( pl3TxGenTvmT & tvm ) :
    tbvTaskTypeSafeT<tbvCellT> ( &tvm, "pl3TxGen" ),

    // TVM reference
    pl3TGen ( tvm )
{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

pl3TxGenTaskT::~pl3TxGenTaskT () {}

//
// Tvm pl3TGen Task pl3TxGen body method
//
void pl3TxGenTaskT::body ( tbvCellT *pl3TxGenArgP )
{
  tbvOut << pl3TGen.getInstanceNameP() << " is called @ ";
  tbvOut << tbvGetInt64Time(TBV_NS) << " ns\n";

  //
  // Do one transaction at a time
  //
  pl3TGen.pl3TxGenMutex.lock();

  vector<uint32> data = pl3TxGenArgP->getDataBy32();
  bool prty = 0;

  tbvSmartUnsignedT badPrtyRand;
  badPrtyRand.keepOnly(1,100);
  
  int beginTime = tbvGetInt64Time(TBV_NS);
  int loopTime = beginTime;

  while (pl3TxGenArgP->getValid() && (pl3TGen.port_stat[(pl3TxGenArgP->getInputPort().getUnsignedValue())] == 0)) {
    loopTime = tbvGetInt64Time(TBV_NS);
    tbvWaitCycle(pl3TGen.pp3tfclk, tbvThreadT::POSEDGE);	
    tbvWait(0.5, TBV_NS); // Hold time
    //  64 ports * 160 ns (Oc48) per port = 10240 = max port calendar
    if (loopTime - beginTime > 20000) 
      {
	tbvExceptionT::setUserExceptionTypeString("PHY INPUT STATUS 0 KEPT TOO LONG");
	tbvExceptionT::reportUserException("PHY INPUT STATUS 0 KEPT TOO LONG",
					   tbvExceptionT::ERROR,
					   "Waited 20000 ns for port_stat = 1 on port %d", pl3TxGenArgP->getInputPort().getUnsignedValue() );
	
	tbvExit();
      }
  }

  if (loopTime > beginTime) {
    tbvExceptionT::setUserExceptionTypeString("PHY CORE BACKPRESSURE");
    tbvExceptionT::reportUserException("PHY CORE BACKPRESSURE",
                                       tbvExceptionT::WARNING,
                                       "Backpressure on port %d for %d ns (%d to %d ns)", pl3TxGenArgP->getInputPort().getUnsignedValue(), (loopTime - beginTime), beginTime, loopTime );
  }

  tbvSignalT dataSig(31,0); // Used because easier to calculate parity

  //Start transaction recording
  if (pl3TxGenArgP->getValid())
    pl3TGen.pl3TxGenFiber.beginTransactionH("Valid Cell");
  else
    pl3TGen.pl3TxGenFiber.beginTransactionH("Invalid Cell");
  
  // If port has changed from last time and cell is valid, send new port
  dataSig = pl3TxGenArgP->getInputPort().getUnsignedValue();
  if ((dataSig.getValue() != pl3TGen.curr_port) && pl3TxGenArgP->getValid()) {
    pl3TGen.curr_port = dataSig.getValue();
    pl3TGen.pp3tenb_n = 1; 
    pl3TGen.pp3tsx = 1;
    randBit.randomize();
    dataSig = pl3TxGenArgP->getInputPort().getUnsignedValue() + ((randBit % 16777216) << 8); // Only lsb 8-bits should be considered
    pl3TGen.pp3tdat = dataSig.getValue();

    prty = dataSig.reductionXor();;
    badPrtyRand.randomize();
    if (badPrtyRand <= pl3TGen.badParity)
      prty = prty ^ 1;
    if (pl3TGen.prtySense)
      pl3TGen.pp3tprty = prty ^ 1;
    else
      pl3TGen.pp3tprty = prty;

    // Randomize don't-care signals
    randBit.randomize();
    pl3TGen.pp3tmod = randBit % 4;
    randBit.randomize();
    pl3TGen.pp3tsop = randBit % 2;
    randBit.randomize();
    pl3TGen.pp3teop = randBit % 2;
    randBit.randomize();
    pl3TGen.pp3terr = randBit % 2;
    
    tbvWaitCycle(pl3TGen.pp3tfclk, tbvThreadT::POSEDGE);
    tbvWait(0.5, TBV_NS); // Hold time
  }
  
  // Send data
  for (unsigned int i = 0; i < data.size(); i++) {

    pl3TGen.pp3tdat = data[i];
      
    // Calculate parity
    dataSig = data[i];
    if (pl3TxGenArgP->getValid()) {
      prty = dataSig.reductionXor();
      badPrtyRand.randomize();
      if (badPrtyRand <= pl3TGen.badParity)
	prty = prty ^ 1;
      if (pl3TGen.prtySense)
	pl3TGen.pp3tprty = prty ^ 1;
      else
	pl3TGen.pp3tprty = prty;
    }
    else { // parity ignored when tenb, tsx disabled so randomize it
      randBit.randomize();
      pl3TGen.pp3tprty = randBit % 2;
    }

    // Start of burst
    if (i == 0) {
      pl3TGen.pp3tenb_n = (1 - pl3TxGenArgP->getValid());
      pl3TGen.pp3tsop = pl3TxGenArgP->getSOP();
      if (data.size() >= 2) { // Not end of burst
	pl3TGen.pp3teop = 0;
	pl3TGen.pp3terr = 0;
	pl3TGen.pp3tmod = 0;
      }
    }
      
    if (i == 1)  pl3TGen.pp3tsop = 0;
      
    if (i == (data.size()-1)) {
      pl3TGen.pp3teop = pl3TxGenArgP->getEOP();
      pl3TGen.pp3tmod = (pl3TGen.badMode) ? ((4 - (pl3TxGenArgP->getSize()%4)) % 4) : ((pl3TxGenArgP->getEOP()) ? ((4 - (pl3TxGenArgP->getSize()%4)) % 4) : 0);
      // terr asserted only when teop is asserted
      if (pl3TxGenArgP->getEOP())
	pl3TGen.pp3terr = pl3TxGenArgP->getBad();
      else
	pl3TGen.pp3terr = 0;
    }
      
    // tsx is ignored when tenb_n is 0, so randomize it
    if (pl3TxGenArgP->getValid()) {
      randBit.randomize();
      pl3TGen.pp3tsx = randBit % 2;
    }
    else
      pl3TGen.pp3tsx = 0;
      
    tbvWaitCycle(pl3TGen.pp3tfclk, tbvThreadT::POSEDGE);
    tbvWait(0.5, TBV_NS); // Hold time
  }
    
  // In case another cell is not immediately generated after this one.
  pl3TGen.pp3tenb_n = 1;
  pl3TGen.pp3tsx = 0;
  if (pl3TxGenArgP->getValid() == 1)
    pl3TGen.port_stat[pl3TxGenArgP->getInputPort().getUnsignedValue()] = 0;
    
  //Record the values in the argument block
  pl3TGen.pl3TxGenFiber.recordAttribute(*pl3TxGenArgP, "Cell");
    
  //Finish Transaction Recording
  pl3TGen.pl3TxGenFiber.endTransaction();
  
  // release the semaphore
  pl3TGen.pl3TxGenMutex.unlock();

  tbvOut << pl3TGen.getInstanceNameP() << " done @ " << tbvGetInt64Time(TBV_NS) << " ns\n";

}


// *****************************************************************
//                         pl3TxQueue Task
// *****************************************************************

//
// Tvm pl3TGen Task pl3TxQueue Constructor & Destructor
//
pl3TxQueueTaskT::pl3TxQueueTaskT (pl3TxGenTvmT& pl3TGen, pl3TxGenTaskT& mainTxTask ) :
    tbvTaskTypeSafeT<tbvCellT> ( &pl3TGen, "pl3TxQueue" ),
    pl3TGen(pl3TGen),
    mainTxTask(mainTxTask)

{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

pl3TxQueueTaskT::~pl3TxQueueTaskT () {}

//
// Tvm pl3TGen Task pl3TxQueue body method
//
void pl3TxQueueTaskT::body ( tbvCellT *pl3TxGenArgP )
{
  //
  // Do one transaction at a time
  //
  pl3TGen.pl3TxQueueMutex.lock();
  pl3TGen.fullStatQueueMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //

  if (tbv4StateNotEqual(pl3TGen.rst_l, 1))
     tbvWait( pl3TGen.not_reset && pl3TGen.posedge_clk , tbvEventExprT::RWSYNC );

  // Push the current transfer request into the FIFO
  pl3TGen.fullStatQueue.push_back(pl3TxGenArgP);

  tbvCellT *queueArg = NULL;

  unsigned int i = 0;
  unsigned int main_task_run = 0;
  unsigned int flag_stat[64];

  for (unsigned int i = 0; i < 64; i++)
      flag_stat[i] = 0;

  // Go through all FIFO entries
  while (i < pl3TGen.fullStatQueue.size())
   if (pl3TGen.fullStatQueue.size() > 0)
    {
      queueArg = pl3TGen.fullStatQueue[i];
      flag_stat[queueArg->getInputPort().getUnsignedValue()] = flag_stat[queueArg->getInputPort().getUnsignedValue()] | (((pl3TGen.port_stat[(queueArg->getInputPort().getUnsignedValue()) % 64] == 0)) ? 1 : 0);


      // Send the current transfer request if the status (per port) allows or if the data is invalid
      if (((pl3TGen.port_stat[(queueArg->getInputPort().getUnsignedValue()) % 64] != 0) && (flag_stat[queueArg->getInputPort().getUnsignedValue()] == 0)) || (queueArg->getValid() == 0))
       {
         if (pl3TGen.fullStatQueue.size() > 0)
          {
            mainTxTask.run(queueArg);
            main_task_run = 1;
          }
         if (pl3TGen.fullStatQueue.size() > 0)
          {
            // Shift up all entirs in the FIFO and erase the last one (erase did not work)
            for (unsigned int j = i; j < pl3TGen.fullStatQueue.size() - 1; j++)
                {
                  pl3TGen.fullStatQueue[j] = pl3TGen.fullStatQueue[j+1];
                }
            pl3TGen.fullStatQueue.pop_back();
            i = i - 1;
          }
       }
      i = i + 1;
    }

// Send invalid cell if nothing can be popped from the FIFO
tbvCellT invalid_cell;
vector<uint8> pkt(64);
tbvControlDataT control;
control.fid.input_port = 0;
control.valid = 0;
control.sop = 0;
control.eop = 0;

for (int j = 0; j < 64; j++)
     pkt[j] = j;
invalid_cell.setPayload(pkt);
invalid_cell.setControlData(control);

if (main_task_run == 0)
 {
   mainTxTask.run(&invalid_cell);
   //tbvOut << "INVALID CELL" << endl;
 }


  pl3TGen.fullStatQueueMutex.unlock();
  pl3TGen.pl3TxQueueMutex.unlock();

}


// *****************************************************************
//                         pl3TxFull Task
// *****************************************************************

//
// Tvm pl3TGen Task pl3TxFull Constructor & Destructor
//
pl3TxFullTaskT::pl3TxFullTaskT (pl3TxGenTvmT& pl3TGen, pl3TxGenTaskT& mainTxTask ) :
    tbvTaskT ( &pl3TGen, "pl3TxFull" ),
    pl3TGen(pl3TGen),
    mainTxTask(mainTxTask)

{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

pl3TxFullTaskT::~pl3TxFullTaskT () {}

//
// Tvm pl3TGen Task pl3TxFull body method
//
void pl3TxFullTaskT::body (tbvSmartDataT *pl3TxGenArgP)
{
  //
  // Do one transaction at a time
  //
  pl3TGen.pl3TxFullMutex.lock();
  pl3TGen.fullStatQueueMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //

  if (tbv4StateNotEqual(pl3TGen.rst_l, 1))
     tbvWait( pl3TGen.not_reset && pl3TGen.posedge_clk , tbvEventExprT::RWSYNC );

tbvCellT *queueArg = NULL;

unsigned int i = 0;
unsigned int stat_flag[64];

for (unsigned int i = 0; i < 64; i++)
    stat_flag[i] = 0;

// Go through all FIFO entries
while (i < pl3TGen.fullStatQueue.size())
 if (pl3TGen.fullStatQueue.size() > 0)
  {
    queueArg = pl3TGen.fullStatQueue[i];
    stat_flag[queueArg->getInputPort().getUnsignedValue()] = stat_flag[queueArg->getInputPort().getUnsignedValue()] | (((pl3TGen.port_stat[(queueArg->getInputPort().getUnsignedValue()) % 64] == 0)) ? 1 : 0);


     // Send the current transfer request if the status (per port) allows or if the data is invalid
    if (((pl3TGen.port_stat[(queueArg->getInputPort().getUnsignedValue()) % 64] != 0) && (stat_flag[queueArg->getInputPort().getUnsignedValue()] == 0)) || (queueArg->getValid() == 0))
     {
       if (pl3TGen.fullStatQueue.size() > 0)
        {
          mainTxTask.run(queueArg);
        }
       if (pl3TGen.fullStatQueue.size() > 0)
        {
          // Shift up all entirs in the FIFO and erase the last one (erase did not work)
          for (unsigned int j = i; j < pl3TGen.fullStatQueue.size() - 1; j++)
              {
                pl3TGen.fullStatQueue[j] = pl3TGen.fullStatQueue[j+1];
              }
          pl3TGen.fullStatQueue.pop_back();
          i = i - 1;
        }
     }
    i = i + 1;
  }


  pl3TGen.fullStatQueueMutex.unlock();
  pl3TGen.pl3TxFullMutex.unlock();

}


// *****************************************************************
//                         pl3TxStat Task
// *****************************************************************

//
// Tvm pl3TGen Task pl3TxStat Constructor & Destructor
//
pl3TxStatTaskT::pl3TxStatTaskT ( pl3TxGenTvmT & tvm ) :
    tbvTaskTypeSafeT<tbvCellT> ( &tvm, "pl3TxStat" ),

    // TVM reference
    pl3TGen ( tvm )
{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

pl3TxStatTaskT::~pl3TxStatTaskT () {}

//
// Tvm pl3TGen Task pl3TxStat body method
//
void pl3TxStatTaskT::body ( tbvCellT *pl3TxGenArgP )
{
  //
  // Do one transaction at a time
  //

  pl3TGen.pl3TxStatMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //

  if (tbv4StateNotEqual(pl3TGen.rst_l, 1))
     tbvWait( pl3TGen.not_reset && pl3TGen.posedge_clk , tbvEventExprT::RWSYNC );

  unsigned int old_port_stat;
  unsigned int port_count = pl3TGen.port_cnt;
  //Collect the list of ports
  for(list<int>::iterator i = pl3TGen.port_list.begin(); i != pl3TGen.port_list.end(); i++ )
      pl3TGen.port_index.push_back(*i);

  unsigned int j = 0;
  for(list<int>::iterator i = pl3TGen.port_list.begin(); i != pl3TGen.port_list.end(); i++ )
    {
      // Get the status of the ports two clk cycles after the polling of the addresses
      unsigned int adr_cnt =  *i;
      unsigned int port = port_count + j - 3;
      pl3TGen.pp3tadr = adr_cnt % 64;
      tbvWait(0.5, TBV_NS);
      old_port_stat = pl3TGen.port_stat[pl3TGen.port_index[port % port_count]];
      pl3TGen.port_stat[pl3TGen.port_index[port % port_count]]  = ((pl3TGen.pp3ptpa.getValue())) % 2;
      //tbvOut << dec << "@ " << tbvGetInt64Time(TBV_NS)  << "Port_Status " << pl3TGen.port_stat[pl3TGen.port_index[port % port_count]] << "  ADDR " << (port % port_count) << " Port_index " << pl3TGen.port_index[port % port_count] << " Port_adr_cnt " << port << " Old_Port_Status " << old_port_stat << " PORT_ADDR : " << adr_cnt << endl;
      //If the status (per port) changes from 0 to 1 and the queue is not empty -> spawn the pl3TxFullTask one more time
      if (((old_port_stat == 0) && (pl3TGen.port_stat[pl3TGen.port_index[port % port_count]] == 1)) && (pl3TGen.fullStatQueue.size() > 0))
       {
         pl3TGen.pl3TxFull.spawn();
         //tbvOut << "SPAWN at : " << tbvGetInt64Time(TBV_NS) << " ns"  << endl;
       }

      j = j + 1;
      tbvWaitCycle(pl3TGen.pp3tfclk, tbvThreadT::POSEDGE);

    }

  pl3TGen.port_index.clear();

  // release the semaphore
  pl3TGen.pl3TxStatMutex.unlock();

}


// *****************************************************************
// a task runStat
// *****************************************************************

//
// Tvm pl3TGen Task runStat Constructor & Destructor
//
runStatTaskT::runStatTaskT ( pl3TxGenTvmT & tvm ) :
    tbvTaskT ( &tvm, "runStatTask" ),


    // TVM reference
    pl3TGen ( tvm )
{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

runStatTaskT::~runStatTaskT () {}
void runStatTaskT::body ( tbvSmartDataT *pl3TxGenArgP )
{
  //for (unsigned int i = 0; i < 64; i++)
    //pl3TGen.port_stat[i] = 0;

  while(pl3TGen.running && !pl3TGen.port_list.empty())
   {
    pl3TGen.pl3TxStat.run();
    tbvThreadT::yield();
   }
}


//******************************************
// stop function
//******************************************
void pl3TxGenTvmT::stop()
 {
   while (fullStatQueue.size() > 0)
     {
       tbvWaitCycle(pp3tfclk, tbvThreadT::POSEDGE);
     }
   running = false;
 }



// *****************************************************************
//                      pl3TGen TVM
// *****************************************************************

//
// TVM pl3TGen Constructor & Destructor
//
pl3TxGenTvmT::pl3TxGenTvmT ( ) :
  tbvTvmT (),

  // Initialize the TVM Signals
        pp3tfclk (getFullInterfaceHdlNameP("pp3tfclk") ),
        rst_l (getFullInterfaceHdlNameP("rst_l") ),
        pp3tenb_n (getFullInterfaceHdlNameP("pp3tenb_n_o") ),
        pp3tsx (getFullInterfaceHdlNameP("pp3tsx_o") ),
        pp3tadr (getFullInterfaceHdlNameP("pp3tadr_o") ),
        pp3tdat (getFullInterfaceHdlNameP("pp3tdat_o") ),
        pp3tsop (getFullInterfaceHdlNameP("pp3tsop_o") ),
        pp3teop (getFullInterfaceHdlNameP("pp3teop_o") ),
        pp3tmod (getFullInterfaceHdlNameP("pp3tmod_o") ),
        pp3tprty (getFullInterfaceHdlNameP("pp3tprty_o") ),
        pp3terr (getFullInterfaceHdlNameP("pp3terr_o") ),
        pp3ptpa (getFullInterfaceHdlNameP("pp3ptpa") ),
        pp3stpa (getFullInterfaceHdlNameP("pp3stpa") ),


  // Initialize the Mutex
  pl3TxGenMutex( "pl3TxGenMutex" ),
  pl3TxQueueMutex( "pl3TxQueueMutex" ),
  pl3TxFullMutex( "pl3TxFullMutex" ),
  pl3TxStatMutex( "pl3TxStatMutex" ),
  fullStatQueueMutex( "fullStatQueueMutex" ),

  pl3TxGenFiber( this, "pl3TxGenFiber"),
  pl3TxQueueFiber( this, "pl3TxQueueFiber"),
  pl3TxFullFiber( this, "pl3TxFullFiber"),

  // Create the task objects
  pl3TxGen(* this),
  pl3TxQueue(* this, pl3TxGen),
  pl3TxFull(* this, pl3TxGen),
  pl3TxStat(* this),
  runStat(* this),
  running(false),
  port_cnt(64),
  prtySense(0),
  badParity(0),
  badMode(0),
  curr_port(0xFF), // unused
  statFull(0),

  // Create the Event Expressions
  posedge_clk ( pp3tfclk, tbvThreadT::POSEDGE ),
  not_reset ( rst_l , 1 )

{
 //add any construtor code needed for tvm pl3TGen
 pl3TxGenFiber.setRecording(TRUE);
}

//destructor for tvm pl3TGen
pl3TxGenTvmT::~pl3TxGenTvmT()
{
  //add any clean up code needed for tvm pl3TGen

}

//
// Method called from $tbv_tvm_connect, to create the TVM object.
//
void pl3TxGenTvmT::create ( )
{
  new pl3TxGenTvmT ( );
};


