// ----------------------------------------------------------------------------
// (c) Copyright 2005, Springer.  All Rights Reserved. The code in this CD-ROM is
// distributed by Springer with ABSOLUTELY NO SUPPORT and NO WARRANTY from
// Springer. Use or reproduction of the information provided on this code for
// commercial gain is strictly prohibited. Explicit permission is given for the
// reproduction and use of this information in an instructional setting provided
// proper reference is given to the original source.
//
// Authors and Springer shall not be liable for damage in connection with, or
// arising out of, the furnishing, performance or use of the contents of the
// CD-ROM.
// ----------------------------------------------------------------------------

#include "TestBuilder.h"
#include "spi4IfGen.h"


// ****************************************
// a task spi4TData  
// ****************************************
spi4TDataTaskT::spi4TDataTaskT(spi4IfTvmT& spi4IfTvm)
  : tbvTaskTypeSafeT<tbvCellT>(&spi4IfTvm, "spi4TDataTask"),
    spi4IfTvm(spi4IfTvm)

{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
 }


spi4TDataTaskT::~spi4TDataTaskT() {}

  //
  // TVM spi4IfTvm : body method for task spi4TData
  //

void spi4TDataTaskT::body(tbvCellT *TrGenArgsP)
{

  //
  // Do one transaction at a time
  //

  spi4IfTvm.spi4TDataMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //

  if (tbv4StateNotEqual(spi4IfTvm.reset, 1))
     tbvWait( spi4IfTvm.not_reset && spi4IfTvm.anyedge_clk , tbvEventExprT::RWSYNC );

  //For random bad parity generation
  tbvSmartUnsignedT badPrtyRand;
  badPrtyRand.keepOnly(1,100);
  badPrtyRand.randomize();


  //For Pattern task - calculate the number of clock cycles the current burst will take
  if ((TrGenArgsP->getSize())%2 == 0)
     spi4IfTvm.size4pattern = (TrGenArgsP->getSize())/2;
  else
     spi4IfTvm.size4pattern = (TrGenArgsP->getSize())/2 + 1;

  spi4IfTvm.size4pattern = spi4IfTvm.size4pattern + 1;  // one clk cycle for cntr word

  // If the number of clocks since last training pattern + the clocks the current burst will need exceeds the programmed max threshold ->run a training pattern
  if ((spi4IfTvm.size4pattern + spi4IfTvm.training_count) > spi4IfTvm.data_max_t)
   {
     spi4IfTvm.training_count = 0;
     trPattern_data();
   }  


  // Start transaction recording
  spi4IfTvm.spi4IfFiber.beginTransactionH("spi4IfFiber");

  vector<uint16> data;

  // Take previous data - needed for calculation of DIP4 later on  
  data = spi4IfTvm.old_data;
  // Update the previous data with the current one  
  spi4IfTvm.old_data = TrGenArgsP->getDataBy16();

  // Get the current burst size in terms of 16-byte blocks
  if ((TrGenArgsP->getSize())%16 == 0)
     spi4IfTvm.block_16 = (TrGenArgsP->getSize())/16;
  else
     spi4IfTvm.block_16 = (TrGenArgsP->getSize())/16 + 1;

  // Based on the status of the current port select the appropriate threshold :
  // 0 - MaxBurst1 starving; 1 - MaxBusrt2 hungry; 2 - satisfied;
  if (spi4IfTvm.status[TrGenArgsP->getInputPort()] == 0)
     spi4IfTvm.thresh[TrGenArgsP->getInputPort()] = spi4IfTvm.MaxBurst1[TrGenArgsP->getInputPort()];
  else if (spi4IfTvm.status[TrGenArgsP->getInputPort()] == 1)
    spi4IfTvm.thresh[TrGenArgsP->getInputPort()] = spi4IfTvm.MaxBurst2[TrGenArgsP->getInputPort()];
  else {
    tbvExceptionT::setUserExceptionTypeString("SPI4 CORE BACKPRESSURE");
    tbvExceptionT::reportUserException("SPI4 CORE BACKPRESSURE",
				       tbvExceptionT::WARNING,
				       "Backpressure on port %d @ %d ns",(TrGenArgsP->getInputPort()).getUnsignedValue(), (int) tbvGetInt64Time(TBV_NS));
    spi4IfTvm.thresh[TrGenArgsP->getInputPort()] = 0;
  }

// Odd Parity Calculation

  unsigned int data_cntr;
  unsigned int data_cntr_while;
  unsigned int dip16_while;
 
  data_cntr = 0;
  data_cntr_while = 0;

// control word for both valid and inalid data
  data_cntr = ((((spi4IfTvm.block16_cnt[TrGenArgsP->getInputPort()] + spi4IfTvm.block_16) > spi4IfTvm.thresh[TrGenArgsP->getInputPort()]) ?  0 : (TrGenArgsP->getValid())) << 15) | (spi4IfTvm.curr_eop << 14) | (spi4IfTvm.odd_byte << 13) | (((((TrGenArgsP->getValid() == 0) || (spi4IfTvm.block16_cnt[TrGenArgsP->getInputPort()] + spi4IfTvm.block_16) > spi4IfTvm.thresh[TrGenArgsP->getInputPort()])) ? 0 : (TrGenArgsP->getSOP())) << 12) | (((((TrGenArgsP->getValid() == 0) || (spi4IfTvm.block16_cnt[TrGenArgsP->getInputPort()] + spi4IfTvm.block_16) > spi4IfTvm.thresh[TrGenArgsP->getInputPort()]))? 0 : (TrGenArgsP->getInputPort())) << 4) | 0xf; 

  data_cntr_while = (TrGenArgsP->getValid() << 15) | (0 << 14) | (0 << 13) | (((TrGenArgsP->getValid() == 0) ? 0 : (TrGenArgsP->getSOP())) << 12) | (((TrGenArgsP->getValid() == 0) ? 0 : (TrGenArgsP->getInputPort())) << 4) | 0xf; 

  dip16_while = data_cntr_while;

  spi4IfTvm.dip4 = spi4IfTvm.dip4_calc(0, data_cntr, data);

  unsigned int  dip4_while = 
 	((((dip16_while & (0x8000 >> 0)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 4)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 8)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 12)) ? 1 : 0)) << 3) |
 	((((dip16_while & (0x8000 >> 1)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 5)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 9)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 13)) ? 1 : 0)) << 2) |
 	((((dip16_while & (0x8000 >> 2)) ? 1 : 0) ^ ((dip16_while & (0x8000 >>  6)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 10)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 14)) ? 1 : 0)) << 1) |
 	 (((dip16_while & (0x8000 >> 3)) ? 1 : 0) ^ ((dip16_while & (0x8000 >>  7)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 11)) ? 1 : 0) ^ ((dip16_while & (0x8000 >> 15)) ? 1 : 0)); 

  unsigned int was_in_while = 0;

// While we have valid, but the current transfer exceeds the threshold we have to send invalid data with a control word representing
// valid or invalid data before this point.

  while (((TrGenArgsP->getValid() == 1) && ((spi4IfTvm.block16_cnt[TrGenArgsP->getInputPort()] + spi4IfTvm.block_16) > spi4IfTvm.thresh[TrGenArgsP->getInputPort()])))
   {
     if (spi4IfTvm.block_16 > spi4IfTvm.MaxBurst1[TrGenArgsP->getInputPort()]) {
       tbvExceptionT::setUserExceptionTypeString("SPI4 MAX BURST VIOLATION");
       tbvExceptionT::reportUserException("SPI4 MAX BURST VIOLATION",
					  tbvExceptionT::ERROR,
					  "For port %d, Burst size = %d > MaxBurst1 = %d", 
					  (unsigned) TrGenArgsP->getInputPort(), spi4IfTvm.block_16, 
					  spi4IfTvm.MaxBurst1[TrGenArgsP->getInputPort()]);
       tbvExit();
     }
 // define/send control word before the invalid data
     if (spi4IfTvm.last_data_valid)
      {
        spi4IfTvm.tctl = 1;
        spi4IfTvm.tdat(15,15) = 0;
	if (spi4IfTvm.curr_eop)
	  if (spi4IfTvm.last_data_bad) {
	    spi4IfTvm.tdat(14,13) = 1; // EOP Abort
	    if (spi4IfTvm.odd_byte != 1)
	      spi4IfTvm.dip4 ^= 0x6;
	    else 
	      spi4IfTvm.dip4 ^= 0x4;
	  }
	  else {
	    spi4IfTvm.tdat(14,14) = 1; // EOP Normal termination
	    spi4IfTvm.tdat(13,13) = spi4IfTvm.odd_byte; // 0 = 2 bytes valid, 1 = 1 byte valid
	  }
	else
	  spi4IfTvm.tdat(14,13) = 0; // Not EOP
	    
        spi4IfTvm.curr_eop = 0;
        spi4IfTvm.odd_byte = 0;
        spi4IfTvm.tdat(12,12) = 0;
        spi4IfTvm.tdat(11,4) = 0;
        if (badPrtyRand < spi4IfTvm.badParity) {
	  tbvOut << spi4IfTvm.getInstanceNameP() << "generating bad parity cell\n";
           spi4IfTvm.tdat(3,0) = spi4IfTvm.dip4 ^ 1;
	}
        else
           spi4IfTvm.tdat(3,0) = spi4IfTvm.dip4;
 
        tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

      }

// Sending invalid data
     spi4IfTvm.tdat = 0x000f;
     tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);
     spi4IfTvm.last_data_valid = 0;
     was_in_while = 1;

// Keep updating the status while waiting and sending invalid data
     if (spi4IfTvm.status[TrGenArgsP->getInputPort()] == 0)
        spi4IfTvm.thresh[TrGenArgsP->getInputPort()] = spi4IfTvm.MaxBurst1[TrGenArgsP->getInputPort()];
     else
       if (spi4IfTvm.status[TrGenArgsP->getInputPort()] == 1)
          spi4IfTvm.thresh[TrGenArgsP->getInputPort()] = spi4IfTvm.MaxBurst2[TrGenArgsP->getInputPort()];
       else 
          spi4IfTvm.thresh[TrGenArgsP->getInputPort()] = 0;
   }

// Here we have enough credit to send the current transfer
// Sending control word and valid data
  if (TrGenArgsP->getValid() == 1)
   {
     spi4IfTvm.tctl = 1;

     // Send the controls
     spi4IfTvm.tdat(15,15) = 1;
     if (spi4IfTvm.curr_eop)
       if (spi4IfTvm.last_data_bad) {
	 spi4IfTvm.tdat(14,13) = 1; // EOP Abort
	 if (spi4IfTvm.odd_byte != 1)
	   spi4IfTvm.dip4 ^= 0x6;
	 else 
	   spi4IfTvm.dip4 ^= 0x4;
       }
       else {
	 spi4IfTvm.tdat(14,14) = 1; // EOP Normal termination
	 spi4IfTvm.tdat(13,13) = spi4IfTvm.odd_byte; // 0 = 2 bytes valid, 1 = 1 byte valid
       }
     else
       spi4IfTvm.tdat(14,13) = 0; // Not EOP
	    
     spi4IfTvm.curr_eop = TrGenArgsP->getEOP();
     spi4IfTvm.odd_byte = (TrGenArgsP->getSize())%2;
     spi4IfTvm.tdat(12,12) = TrGenArgsP->getSOP();
     spi4IfTvm.tdat(11,4) = TrGenArgsP->getInputPort();
     spi4IfTvm.fl[TrGenArgsP->getInputPort()] = 0;

     if(spi4IfTvm.invalid) {
       if (badPrtyRand < spi4IfTvm.badParity) {
	 tbvOut << spi4IfTvm.getInstanceNameP() << "generating bad parity cell\n";
         spi4IfTvm.tdat(3,0) = dip4_while ^ 1; //0xF;
       }
       else
         spi4IfTvm.tdat(3,0) = dip4_while;
     }
     else{
       if (badPrtyRand < spi4IfTvm.badParity) {
	  tbvOut << spi4IfTvm.getInstanceNameP() << "generating bad parity cell\n";
         spi4IfTvm.tdat(3,0) = (was_in_while) ? (dip4_while ^ 1) : (spi4IfTvm.dip4 ^ 1);
       }
       else
         spi4IfTvm.tdat(3,0) = (was_in_while) ? (dip4_while) : (spi4IfTvm.dip4);
     }

     tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);
     spi4IfTvm.tctl = 0;

     data = spi4IfTvm.old_data;

     int beginTime;

     // Send the data
     for (unsigned int i = 0; i < data.size(); i++)
       {
         if ((i == data.size()-1) && (spi4IfTvm.odd_byte == 1))
            spi4IfTvm.tdat = data[i]&0xff00;
         else
            spi4IfTvm.tdat = data[i];

         tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

         beginTime = tbvGetInt64Time(TBV_NS);
       }

     spi4IfTvm.block16_cnt[TrGenArgsP->getInputPort()] = (spi4IfTvm.block16_cnt[TrGenArgsP->getInputPort()] + spi4IfTvm.block_16);

     spi4IfTvm.tctl = 1;
     spi4IfTvm.invalid = 0;
     spi4IfTvm.last_data_valid = 1;

   }

  else

// Sending invalid data
   {
     spi4IfTvm.tctl = 1;
     spi4IfTvm.tdat(15,15) = 0;
     if (spi4IfTvm.curr_eop)
       if (spi4IfTvm.last_data_bad) {
	 spi4IfTvm.tdat(14,13) = 1; // EOP Abort
	 if (spi4IfTvm.odd_byte != 1)
	   spi4IfTvm.dip4 ^= 0x6;
	 else 
	   spi4IfTvm.dip4 ^= 0x4;
       }
       else {
	 spi4IfTvm.tdat(14,14) = 1; // EOP Normal termination
	 spi4IfTvm.tdat(13,13) = spi4IfTvm.odd_byte; // 0 = 2 bytes valid, 1 = 1 byte valid
       }
     else
       spi4IfTvm.tdat(14,13) = 0; // Not EOP
	    
     spi4IfTvm.curr_eop = 0;
     spi4IfTvm.odd_byte = 0;
     spi4IfTvm.tdat(12,12) = 0;
     spi4IfTvm.tdat(11,4) = 0;
    // spi4IfTvm.tdat(3,0) = 0;

     if (spi4IfTvm.last_data_valid == 0)
        spi4IfTvm.tdat(3,0) = 0xf;
     else
      {
	if (badPrtyRand < spi4IfTvm.badParity) {
	  tbvOut << spi4IfTvm.getInstanceNameP() << "generating bad parity cell\n";	  
          spi4IfTvm.tdat(3,0) = spi4IfTvm.dip4 ^ 1;  
	}
       else
          spi4IfTvm.tdat(3,0) = spi4IfTvm.dip4;
      }

     tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);


     data = TrGenArgsP->getDataBy16();

     // Send the data
     for (unsigned int i = 0; i < data.size()+1; i++)
       {
         spi4IfTvm.tdat = 0x000f;  //data[i];   

         tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);
       }

     vector<uint16> invalid(data.size(),0x000f);
     spi4IfTvm.old_data = invalid;
     spi4IfTvm.invalid = 1;
     spi4IfTvm.last_data_valid = 0;

   }


  spi4IfTvm.last_data_bad = TrGenArgsP->getBad();
  // Record the values in the argument block onto the fiber
  spi4IfTvm.spi4IfFiber.recordAttribute(*TrGenArgsP, "spi4IfFiber");

  // Finish Transaction Recording
  spi4IfTvm.spi4IfFiber.endTransaction();


  spi4IfTvm.spi4TDataMutex.unlock();

}

// ****************************************
// trPattern_data function
// ****************************************
void spi4TDataTaskT::trPattern_data()
{
  //spi4IfTvm.spi4PatternMutex.lock();
  //spi4IfTvm.spi4TDataMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //
  if (tbv4StateNotEqual(spi4IfTvm.reset, 1))
     tbvWait( spi4IfTvm.not_reset && spi4IfTvm.anyedge_clk , tbvEventExprT::RWSYNC );


  // Start transaction recording
  spi4IfTvm.spi4IfFiber.beginTransactionH("training pattern");

  spi4IfTvm.flag_data = 1;

//  vector<uint16> data = TrGenArgsP->getDataBy16();
  vector<uint16> data;

  data = spi4IfTvm.old_data;
  //spi4IfTvm.old_data = TrGenArgsP->getDataBy16();


// Odd Parity Calculation - calculated similarly to Odd Parity Calculation in spi4TDataTask

  unsigned int data_cntr;

  data_cntr = 0;

  data_cntr = (0 << 15) | (spi4IfTvm.curr_eop << 14) | (spi4IfTvm.odd_byte << 13) | (0 << 12) | (0 << 4) | 0xf;

  spi4IfTvm.dip4 = spi4IfTvm.dip4_calc(spi4IfTvm.invalid, data_cntr, data);


// Send the control word before the training pattern

   spi4IfTvm.tctl = 1;
   spi4IfTvm.tdat(15,15) = 0;
   spi4IfTvm.tdat(14,14) = (spi4IfTvm.last_data_valid) ? spi4IfTvm.curr_eop : 0;
   spi4IfTvm.tdat(13,13) = (spi4IfTvm.last_data_valid) ? spi4IfTvm.odd_byte : 0;
   spi4IfTvm.curr_eop = 0;
   spi4IfTvm.odd_byte = 0;
   spi4IfTvm.tdat(12,12) = 0;
   spi4IfTvm.tdat(11,4) = 0;
   spi4IfTvm.tdat(3,0) = spi4IfTvm.dip4; 

   tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

   // Send the pattern
  for (unsigned int m2 = 0; m2 < spi4IfTvm.training_m; m2++)
   for (unsigned int i = 0; i < 10; i++)
    {
       spi4IfTvm.tctl = 1;
       spi4IfTvm.tdat = 0x0fff;

       for (unsigned int k = 0; k < 10; k++)
           tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

       spi4IfTvm.tctl = 0;
       spi4IfTvm.tdat = 0xf000;

       for (unsigned int j = 0; j < 10; j++)
           tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);
    }

   spi4IfTvm.training_count = 0;
   spi4IfTvm.tctl = 1;
   spi4IfTvm.tdat = 0x000f;
   spi4IfTvm.last_data_valid = 0;
   spi4IfTvm.invalid = 1;

   spi4IfTvm.flag_data = 0;

  // Record the values in the argument block onto the fiber
  tbvCellT invalid_pattern(1);
  spi4IfTvm.spi4IfFiber.recordAttribute(invalid_pattern, "training pattern");

  // Finish Transaction Recording
  spi4IfTvm.spi4IfFiber.endTransaction();


  //spi4IfTvm.spi4PatternMutex.unlock();
  //spi4IfTvm.spi4TDataMutex.unlock();

}


// ****************************************
// a task fullStat
// ****************************************
fullStatTaskT::fullStatTaskT(spi4IfTvmT& spi4IfTvm, spi4TDataTaskT& mainTask)
  : tbvTaskTypeSafeT<tbvCellT>(&spi4IfTvm, "fullStatTask"),
    spi4IfTvm(spi4IfTvm),
    mainTask(mainTask)

{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  //setSynchronization(FALSE, FALSE);
 }


fullStatTaskT::~fullStatTaskT() {}

  //
  // TVM spi4IfTvm : body method for task fullStat
  //

void fullStatTaskT::body(tbvCellT *TrGenArgsP)
{

  //
  // Do one transaction at a time
  //

  spi4IfTvm.fullStatMutex.lock();
  spi4IfTvm.portQueueMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //

  if (tbv4StateNotEqual(spi4IfTvm.reset, 1))
     tbvWait( spi4IfTvm.not_reset && spi4IfTvm.anyedge_clk , tbvEventExprT::RWSYNC );

// Push the current transfer request into the FIFO
spi4IfTvm.portQueue.push_back(TrGenArgsP);

tbvCellT *pipedArg = NULL;

unsigned int i = 0;
unsigned int main_task_run = 0;
unsigned int flag_stat[64];

for (unsigned int i = 0; i < 64; i++)
    flag_stat[i] = 0;

// Go through all FIFO entries
while (i < spi4IfTvm.portQueue.size())
 if (spi4IfTvm.portQueue.size() > 0) 
  {
    pipedArg = spi4IfTvm.portQueue[i];

// Calculate the number of 16-byte chunks in the current transfer
    if ((pipedArg->getSize())%16 == 0)
       spi4IfTvm.block_16 = (pipedArg->getSize())/16;
    else
       spi4IfTvm.block_16 = (pipedArg->getSize())/16 + 1;

// Select the correct threshold based on the status for the current port
    if (spi4IfTvm.status[pipedArg->getInputPort()] == 0)
       spi4IfTvm.thresh[pipedArg->getInputPort()] = spi4IfTvm.MaxBurst1[pipedArg->getInputPort()];
    else
      if (spi4IfTvm.status[pipedArg->getInputPort()] == 1)
         spi4IfTvm.thresh[pipedArg->getInputPort()] = spi4IfTvm.MaxBurst2[pipedArg->getInputPort()];
      else
         spi4IfTvm.thresh[pipedArg->getInputPort()] = 0;

    flag_stat[pipedArg->getInputPort().getUnsignedValue()] = flag_stat[pipedArg->getInputPort().getUnsignedValue()] | (((spi4IfTvm.block_16 + spi4IfTvm.block16_cnt[pipedArg->getInputPort()]) < spi4IfTvm.thresh[pipedArg->getInputPort()]) ? 0 : 1);

// Send the current transfer request if the threshold condition is met or if the data is invalid (no need to check threshold)
    if ((((spi4IfTvm.block_16 + spi4IfTvm.block16_cnt[pipedArg->getInputPort()]) < spi4IfTvm.thresh[pipedArg->getInputPort()]) && (flag_stat[pipedArg->getInputPort().getUnsignedValue()] == 0)) || (pipedArg->getValid() == 0))
     {
       if (spi4IfTvm.portQueue.size() > 0)
        {
          mainTask.run(pipedArg);
          main_task_run = 1;
        }
       if (spi4IfTvm.portQueue.size() > 0)
        {
// Shift up all entirs in the FIFO and erase the last one (erase did not work)
           for (unsigned int j = i; j < spi4IfTvm.portQueue.size() - 1; j++)
             {
               spi4IfTvm.portQueue[j] = spi4IfTvm.portQueue[j+1];  
             }
           spi4IfTvm.portQueue.pop_back();
           i = i - 1;
        }
     }
    i = i + 1;
  }
 // send invalid cell if nothing can be popped from the FIFO  
tbvCellT invalid_cell;
vector<uint8> pkt(64);
tbvControlDataT control;
control.fid.input_port = 0;
control.valid = 0;
control.sop = 0;
control.eop = 0;

for (int j = 0; j < 64; j++)
     pkt[j] = j;
invalid_cell.setPayload(pkt);
invalid_cell.setControlData(control);

if (main_task_run == 0)
 {
   mainTask.run(&invalid_cell);
 }

  spi4IfTvm.portQueueMutex.unlock();
  spi4IfTvm.fullStatMutex.unlock();

}


// *****************************************************************
// a task queueStat
// *****************************************************************
queueStatTaskT::queueStatTaskT (spi4IfTvmT& spi4IfTvm, spi4TDataTaskT& mainTask) 
  : tbvTaskT ( &spi4IfTvm, "queueStatTask" ),
    spi4IfTvm ( spi4IfTvm ),
    mainTask(mainTask)

{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

queueStatTaskT::~queueStatTaskT () {}

void queueStatTaskT::body ( tbvSmartDataT *TrGenArgsP )
{

  spi4IfTvm.queueStatMutex.lock();
  spi4IfTvm.portQueueMutex.lock();


  if (tbv4StateNotEqual(spi4IfTvm.reset, 1))
     tbvWait( spi4IfTvm.not_reset && spi4IfTvm.anyedge_clk , tbvEventExprT::RWSYNC );

 // send invalid cell if nothing can be popped from the FIFO
tbvCellT last_invalid_cell;
vector<uint8> pkt(64);
tbvControlDataT control;
control.fid.input_port = 0;
control.valid = 0;
control.sop = 0;
control.eop = 0;

for (int j = 0; j < 64; j++)
     pkt[j] = j;
last_invalid_cell.setPayload(pkt);
last_invalid_cell.setControlData(control);


tbvCellT *pipedArg = NULL;

unsigned int i = 0;
unsigned int sent_successfully = 0;
unsigned int stat_flag[64];

for (unsigned int i = 0; i < 64; i++)
    stat_flag[i] = 0;

// Go through all FIFO entries
while (i < spi4IfTvm.portQueue.size())
 if (spi4IfTvm.portQueue.size() > 0)
  {
    pipedArg = spi4IfTvm.portQueue[i];

// Calculate the number of 16-byte chunks in the current transfer
    if ((pipedArg->getSize())%16 == 0)
       spi4IfTvm.block_16 = (pipedArg->getSize())/16;
    else
       spi4IfTvm.block_16 = (pipedArg->getSize())/16 + 1;


// Select the correct threshold based on the status for the current port
    if (spi4IfTvm.status[pipedArg->getInputPort()] == 0)
       spi4IfTvm.thresh[pipedArg->getInputPort()] = spi4IfTvm.MaxBurst1[pipedArg->getInputPort()];
    else
      if (spi4IfTvm.status[pipedArg->getInputPort()] == 1)
         spi4IfTvm.thresh[pipedArg->getInputPort()] = spi4IfTvm.MaxBurst2[pipedArg->getInputPort()];
      else
         spi4IfTvm.thresh[pipedArg->getInputPort()] = 0;


    stat_flag[pipedArg->getInputPort().getUnsignedValue()] = stat_flag[pipedArg->getInputPort().getUnsignedValue()] | (((spi4IfTvm.block_16 + spi4IfTvm.block16_cnt[pipedArg->getInputPort()]) < spi4IfTvm.thresh[pipedArg->getInputPort()]) ? 0 : 1);

// Send the current transfer request if the threshold condition is met or if the data is invalid (no need to check threshold)
    if ((((spi4IfTvm.block_16 + spi4IfTvm.block16_cnt[pipedArg->getInputPort()]) < spi4IfTvm.thresh[pipedArg->getInputPort()]) && (stat_flag[pipedArg->getInputPort().getUnsignedValue()] == 0)) || (pipedArg->getValid() == 0))
     {
       if (spi4IfTvm.portQueue.size() > 0)
        {
          mainTask.run(pipedArg);
          sent_successfully = 1;
        }
       if (spi4IfTvm.portQueue.size() > 0)
        {
// Shift up all entirs in the FIFO and erase the last one (erase did not work)
          for (unsigned int j = i; j < spi4IfTvm.portQueue.size() - 1; j++)
            {
              spi4IfTvm.portQueue[j] = spi4IfTvm.portQueue[j+1];
            }
          spi4IfTvm.portQueue.pop_back();
          i = i - 1;
        }
     }
    i = i + 1;
  }

  //Send Last Invalid cell to match the last parity
  if(sent_successfully == 1)
    {
       mainTask.run(&last_invalid_cell);
       tbvOut << "LAST_INVALID CELL" << endl;
    }


  spi4IfTvm.portQueueMutex.unlock();
  spi4IfTvm.queueStatMutex.unlock();

}


// ****************************************
// a task spi4Pattern
// ****************************************
spi4PatternTaskT::spi4PatternTaskT(spi4IfTvmT& spi4IfTvm)
  : tbvTaskTypeSafeT<tbvCellT>(&spi4IfTvm, "spi4PatternTask"),
    spi4IfTvm(spi4IfTvm)

{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
 }


spi4PatternTaskT::~spi4PatternTaskT() {}

  //
  // TVM spi4IfTvm : body method for task spi4Pattern
  //

void spi4PatternTaskT::body(tbvCellT *TrGenArgsP)
{

while (spi4IfTvm.running)
{
// Send training pattern if just starting simulation or if Status = 3 for
// too long or training ocunter has reached its max value
 if ((spi4IfTvm.deskewed) || (spi4IfTvm.training_count == spi4IfTvm.data_max_t))
 {
    if (spi4IfTvm.deskewed)
     {
       trPattern_deskewed();
       spi4IfTvm.training_count = 0;
     }
    if (spi4IfTvm.training_count == spi4IfTvm.data_max_t)
     {
       trPattern_cycles();
       spi4IfTvm.training_count = 0;
     }
 }
else
 {
  tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);
 }
}
}


// ****************************************
// trPattern_deskewed function
// ****************************************
void spi4PatternTaskT::trPattern_deskewed()
{
  //spi4IfTvm.spi4PatternMutex.lock();
  spi4IfTvm.spi4TDataMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //
  if (tbv4StateNotEqual(spi4IfTvm.reset, 1))
     tbvWait( spi4IfTvm.not_reset && spi4IfTvm.anyedge_clk , tbvEventExprT::RWSYNC );


  // Start transaction recording
  spi4IfTvm.spi4IfFiber.beginTransactionH("training pattern");

  spi4IfTvm.flag_deskewed = 1;

//  vector<uint16> data = TrGenArgsP->getDataBy16();
  vector<uint16> data;

  data = spi4IfTvm.old_data;
  //spi4IfTvm.old_data = TrGenArgsP->getDataBy16();


// Odd Parity Calculation - calculated similarly to Odd Parity Calculation in spi4TDataTask

  unsigned int data_cntr;

  data_cntr = 0;

  data_cntr = (0 << 15) | (spi4IfTvm.curr_eop << 14) | (spi4IfTvm.odd_byte << 13) | (0 << 12) | (0 << 4) | 0xf;

  spi4IfTvm.dip4 = spi4IfTvm.dip4_calc(spi4IfTvm.invalid, data_cntr, data);

// Send the control word before the training pattern
 
   spi4IfTvm.tctl = 1;
   spi4IfTvm.tdat(15,15) = 0;
   spi4IfTvm.tdat(14,14) = (spi4IfTvm.last_data_valid) ? spi4IfTvm.curr_eop : 0;
   spi4IfTvm.tdat(13,13) = (spi4IfTvm.last_data_valid) ? spi4IfTvm.odd_byte : 0;
   spi4IfTvm.curr_eop = 0;
   spi4IfTvm.odd_byte = 0;
   spi4IfTvm.tdat(12,12) = 0;
   spi4IfTvm.tdat(11,4) = 0;
   spi4IfTvm.tdat(3,0) = spi4IfTvm.dip4;

   //tbvOut << " DIP4_PATTERN at : " << spi4IfTvm.tdat(3,0) << tbvGetInt64Time(TBV_NS) << " ns" << endl;

   tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);


   // Send the pattern
   for (unsigned int i = 0; i < 10; i++)
    {
     spi4IfTvm.tctl = 1;
     spi4IfTvm.tdat = 0x0fff;

     for (unsigned int k = 0; k < 10; k++)
         tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

     spi4IfTvm.tctl = 0;
     spi4IfTvm.tdat = 0xf000;

     for (unsigned int j = 0; j < 10; j++)
         tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);
    }

   spi4IfTvm.tctl = 1;
   spi4IfTvm.tdat = 0x000f;
   spi4IfTvm.last_data_valid = 0;
   spi4IfTvm.invalid = 1;

   spi4IfTvm.flag_deskewed = 0;

  // Record the values in the argument block onto the fiber
  tbvCellT invalid_pattern(1);
  spi4IfTvm.spi4IfFiber.recordAttribute(invalid_pattern, "training pattern");

  // Finish Transaction Recording
  spi4IfTvm.spi4IfFiber.endTransaction();


  //spi4IfTvm.spi4PatternMutex.unlock();
  spi4IfTvm.spi4TDataMutex.unlock();

}



// ****************************************
// trPattern_cycles function
// ****************************************
void spi4PatternTaskT::trPattern_cycles()
{
  //spi4IfTvm.spi4PatternMutex.lock();
  spi4IfTvm.spi4TDataMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //
  if (tbv4StateNotEqual(spi4IfTvm.reset, 1))
     tbvWait( spi4IfTvm.not_reset && spi4IfTvm.anyedge_clk , tbvEventExprT::RWSYNC );


  // Start transaction recording
  spi4IfTvm.spi4IfFiber.beginTransactionH("training pattern");

  spi4IfTvm.flag_cycles = 1;

  vector<uint16> data;

  data = spi4IfTvm.old_data;

// Odd Parity Calculation - calculated similarly to Odd Parity Calculation in spi4TDataTask

  unsigned int data_cntr;

  data_cntr = 0;

  data_cntr = (0 << 15) | (spi4IfTvm.curr_eop << 14) | (spi4IfTvm.odd_byte << 13) | (0 << 12) | (0 << 4) | 0xf;

  spi4IfTvm.dip4 = spi4IfTvm.dip4_calc(spi4IfTvm.invalid, data_cntr, data);

// Send the control word before the training pattern
 
   spi4IfTvm.tctl = 1;
   spi4IfTvm.tdat(15,15) = 0;
   spi4IfTvm.tdat(14,14) = (spi4IfTvm.last_data_valid) ? spi4IfTvm.curr_eop : 0;
   spi4IfTvm.tdat(13,13) = (spi4IfTvm.last_data_valid) ? spi4IfTvm.odd_byte : 0;
   spi4IfTvm.curr_eop = 0;
   spi4IfTvm.odd_byte = 0;
   spi4IfTvm.tdat(12,12) = 0;
   spi4IfTvm.tdat(11,4) = 0;
   spi4IfTvm.tdat(3,0) = spi4IfTvm.dip4;

   //tbvOut << " DIP4_PATTERN at : " << spi4IfTvm.tdat(3,0) << tbvGetInt64Time(TBV_NS) << " ns" << endl;

   tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

   // Send the pattern
  for (unsigned int m2 = 0; m2 < spi4IfTvm.training_m; m2++)
   for (unsigned int i = 0; i < 10; i++)
    {
       spi4IfTvm.tctl = 1;
       spi4IfTvm.tdat = 0x0fff;

       for (unsigned int k = 0; k < 10; k++)
           tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

       spi4IfTvm.tctl = 0;
       spi4IfTvm.tdat = 0xf000;

       for (unsigned int j = 0; j < 10; j++)
           tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);
    }

   spi4IfTvm.tctl = 1;
   spi4IfTvm.tdat = 0x000f;
   spi4IfTvm.last_data_valid = 0;
   spi4IfTvm.invalid = 1;

   spi4IfTvm.flag_cycles = 0;

  // Record the values in the argument block onto the fiber
  tbvCellT invalid_pattern(1);
  spi4IfTvm.spi4IfFiber.recordAttribute(invalid_pattern, "training pattern");

  // Finish Transaction Recording
  spi4IfTvm.spi4IfFiber.endTransaction();


  //spi4IfTvm.spi4PatternMutex.unlock();
  spi4IfTvm.spi4TDataMutex.unlock();

}


// ****************************************
// a task spi4TStat
// ****************************************
spi4TStatTaskT::spi4TStatTaskT(spi4IfTvmT& spi4IfTvm)
  : tbvTaskTypeSafeT<tbvCellT>(&spi4IfTvm, "spi4TStatTask"),
    spi4IfTvm(spi4IfTvm)

{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
 }


spi4TStatTaskT::~spi4TStatTaskT() {}

  //
  // TVM spi4IfTvm : body method for task spi4TStat
  //

void spi4TStatTaskT::body(tbvCellT *TrGenArgsP)
{

  //
  // Do one transaction at a time
  //

  spi4IfTvm.spi4TStatMutex.lock();

  //
  // wait for Reset to end (if it is asserted)
  //

  if (tbv4StateNotEqual(spi4IfTvm.reset, 1))
     tbvWait( spi4IfTvm.not_reset && spi4IfTvm.posedge_tsclk , tbvEventExprT::RWSYNC );


  // Start transaction recording
//      spi4IfTvm.spi4TStatFiber.beginTransactionH("spi4TStatFiber");

  unsigned int temp_status[256];
  unsigned int x0, x1;
  unsigned int dip2_index;
  unsigned int x1_xor;
  unsigned int x0_xor;
  unsigned int stat3Start;

// Get the value of each of the two Status Bus bits
  x1 = ((spi4IfTvm.tstat(1,1)) !=0)? 1 : 0;
  x0 = ((spi4IfTvm.tstat(0,0)) !=0)? 1 : 0;

// Wait for the status to become 3 
  while (spi4IfTvm.running && ((x1 == 0) || (x0 == 0)) )
  {
    tbvWaitCycle(spi4IfTvm.tsclk, tbvThreadT::POSEDGE);
    x1 = ((spi4IfTvm.tstat(1,1)) !=0)? 1 : 0;
    x0 = ((spi4IfTvm.tstat(0,0)) !=0)? 1 : 0;
  }

// Remember the time when status 3 began -> to be used for starting training pattern through the flag spi4IfTvm.deskewed

  stat3Start = tbvGetInt64Time(TBV_NS);

// Wait while status remains 3
  while (spi4IfTvm.running && (x1 != 0) && (x0 != 0))
  {
    tbvWaitCycle(spi4IfTvm.tsclk, tbvThreadT::POSEDGE);
    x1 = ((spi4IfTvm.tstat(1,1)) !=0)? 1 : 0;
    x0 = ((spi4IfTvm.tstat(0,0)) !=0)? 1 : 0;
    unsigned int stat3End = tbvGetInt64Time(TBV_NS);
    if ((stat3End - stat3Start > (spi4IfTvm.cal_len * spi4IfTvm.cal_m * 1)) && ((x1 != 0) && (x0 != 0)))
     {
       stat3Start = tbvGetInt64Time(TBV_NS);
       //tbvOut << " Pattern_task_run at : " << tbvGetInt64Time(TBV_NS) << " ns" << endl;
       //spi4IfTvm.spi4Pattern.spawn();
       spi4IfTvm.deskewed = 1;
     }
  }
  
  spi4IfTvm.deskewed = 0;


//Calculate DIP2 on the status received
  x1_xor = 0;
  x0_xor = 0;
  dip2_index = 1;

  for (unsigned int k = 0; k < spi4IfTvm.cal_m; k++)
    for (unsigned int i = 0; i < spi4IfTvm.cal_len; i++)
      {
	x1 = ((spi4IfTvm.tstat(1,1)) !=0)? 1 : 0;
	x0 = ((spi4IfTvm.tstat(0,0)) !=0)? 1 : 0;
        //tbvOut << " x1_and_x0 are  : " << x1 << "  " << x0 << endl;

        x1_xor = x1_xor ^ ((dip2_index == 1) ? x1 : x0);
        x0_xor = x0_xor ^ ((dip2_index == 1) ? x0 : x1);
        dip2_index = 1 - dip2_index;

        temp_status[spi4IfTvm.port_cal[i]] = x1+x1+x0; 
        //tbvOut << " TEMP_STAT is  : " << temp_status[spi4IfTvm.port_cal[i]] <<" at : " << tbvGetInt64Time(TBV_NS) << " ns"    << endl;
        spi4IfTvm.fl[spi4IfTvm.port_cal[i]] = 1;
        spi4IfTvm.block16_cnt[spi4IfTvm.port_cal[i]] = 0;
        tbvWaitCycle(spi4IfTvm.tsclk, tbvThreadT::POSEDGE);
      }

  x1 = ((spi4IfTvm.tstat(1,1)) !=0)? 1 : 0;
  x0 = ((spi4IfTvm.tstat(0,0)) !=0)? 1 : 0;
  spi4IfTvm.dip2 = x1+x1+x0;  // 0;

  x1_xor = x1_xor ^ 1; 
  x0_xor = x0_xor ^ 1; 


  unsigned int dip2_calc;
  unsigned int spawn_flag;

  dip2_calc = (dip2_index != 0) ? (x1_xor + x1_xor + x0_xor) : (x0_xor + x0_xor + x1_xor);

  //tbvOut << "DIP2 is : "<< dip2_calc << "  spi4IfTvmDIP2 is : "<< spi4IfTvm.dip2 << " at : " << tbvGetInt64Time(TBV_NS) << " ns"  << endl;

  spawn_flag = 0;
// Only if calculated DIP2 matches the received DIP2, status would be updated
  if (dip2_calc == spi4IfTvm.dip2)
     for (unsigned int d = 0; d < spi4IfTvm.cal_len; d++) 
      {
// Check if a favourable status change has happened (2->0, 2->1 or 1->0)
        if (spi4IfTvm.status[spi4IfTvm.port_cal[d]] > temp_status[spi4IfTvm.port_cal[d]])
           spawn_flag = 1;
        spi4IfTvm.status[spi4IfTvm.port_cal[d]] = temp_status[spi4IfTvm.port_cal[d]]; 
        //tbvOut << "STATUS_stat is : "<< spi4IfTvm.status[spi4IfTvm.port_cal[d]] << "  UPDATED_TEMP is:" << temp_status[spi4IfTvm.port_cal[d]] << " at : " << tbvGetInt64Time(TBV_NS) << " ns" << endl;
           
      }

// If a favourable status update has happened and the Non-Blocking transfer request FIFO is not empty - try to send the pending data by spawning queueStat task
   if ((spawn_flag == 1) && (spi4IfTvm.portQueue.size() > 0))
    {
      spi4IfTvm.queueStat.spawn();
      //tbvOut << "SPAWN at : " << tbvGetInt64Time(TBV_NS) << " ns"  << endl;
    }


  // Record the values in the argument block onto the fiber
//      spi4IfTvm.spi4TStatFiber.recordAttribute(*TrGenArgsP, "spi4TStatFiber");

  // Finish Transaction Recording
//      spi4IfTvm.spi4TStatFiber.endTransaction();


  spi4IfTvm.spi4TStatMutex.unlock();

}


// *****************************************************************
// a task runTStat 
// *****************************************************************

//
// Tvm spi4IfTvmT Task runTStat Constructor & Destructor
//
runTStatTaskT::runTStatTaskT ( spi4IfTvmT & spi4IfTvm ) :
    tbvTaskT ( &spi4IfTvm, "runTStatTask" ),

    // TVM reference
    spi4IfTvm ( spi4IfTvm )
{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

runTStatTaskT::~runTStatTaskT () {}
void runTStatTaskT::body ( tbvSmartDataT *TrGenArgsP )
{

while(spi4IfTvm.running)
{
tbvWaitCycle(spi4IfTvm.tsclk, tbvThreadT::POSEDGE);
spi4IfTvm.spi4TStat.run();
}

}


// *****************************************************************
// a task trainingCnt
// *****************************************************************

//
// Tvm spi4IfTvmT Task trainingCnt Constructor & Destructor
//
trainingCntTaskT::trainingCntTaskT ( spi4IfTvmT & spi4IfTvm ) :
    tbvTaskT ( &spi4IfTvm, "trainingCntTask" ),

    // TVM reference
    spi4IfTvm ( spi4IfTvm )
{
  // Disable the mutex built into the fiber and turn off automatic
  // transaction recording
  setSynchronization(FALSE, FALSE);
}

trainingCntTaskT::~trainingCntTaskT () {}
void trainingCntTaskT::body ( tbvSmartDataT *TrGenArgsP )
{
   spi4IfTvm.training_count = 0;
   while (spi4IfTvm.running)
     {
       tbvWaitCycle(spi4IfTvm.tdclk, tbvThreadT::ANYEDGE);

// This is a free running counter, which is being reset, only when a 
// training pattern starts. It is clipped at data_max_t - max number of clock cycles allowed between two training patterns.

       if ((spi4IfTvm.flag_data == 1) || (spi4IfTvm.flag_deskewed == 1) || (spi4IfTvm.flag_cycles == 1))
          spi4IfTvm.training_count = 0;
       else
         if (spi4IfTvm.training_count != spi4IfTvm.data_max_t)
            spi4IfTvm.training_count += 1;
     }
}

//******************************************
// dip4_calc function
//******************************************
unsigned int spi4IfTvmT::dip4_calc(unsigned int inavlid, unsigned int data_cntr, vector<uint16> & data) 
{

  unsigned int lines;
  lines = data.size();
  unsigned int jj;
  unsigned int t;
  unsigned int dip16;
  unsigned int dip16_1;
  unsigned int dip16_2;
  unsigned int dip16_3;

  dip16 = 0;

  if (invalid)
     dip16 = data_cntr;
  else
  {
  for (unsigned int k = 0; k < 16; k++)
    {
      jj = k;
      t = 0;
      for (unsigned int m = 0; m < (lines); m++)
        {
          t = t ^ ((data[m] & (0x8000 >> (jj % 16))) ? 1 : 0);
          jj = jj + 1;
        }

      dip16_1 = (((t != 0) ? 0x8000 : 0) >> ((jj)%16));
      dip16_2 = (0x8000 >> ((jj)%16)) & data_cntr;
      dip16_3 = dip16_1 ^ dip16_2;

      dip16 = dip16 | dip16_3;

    }
   }

 return

        ((((dip16 & (0x8000 >> 0)) ? 1 : 0) ^ ((dip16 & (0x8000 >> 4)) ? 1 : 0) ^ ((dip16 & (0x8000 >> 8)) ?
1 : 0) ^ ((dip16 & (0x8000 >> 12)) ? 1 : 0)) << 3) |
        ((((dip16 & (0x8000 >> 1)) ? 1 : 0) ^ ((dip16 & (0x8000 >> 5)) ? 1 : 0) ^ ((dip16 & (0x8000 >> 9)) ?
1 : 0) ^ ((dip16 & (0x8000 >> 13)) ? 1 : 0)) << 2) |
        ((((dip16 & (0x8000 >> 2)) ? 1 : 0) ^ ((dip16 & (0x8000 >>  6)) ? 1 : 0) ^ ((dip16 & (0x8000 >> 10))
? 1 : 0) ^((dip16 & (0x8000 >> 14)) ? 1 : 0)) << 1) |
         (((dip16 & (0x8000 >> 3)) ? 1 : 0) ^ ((dip16 & (0x8000 >>  7)) ? 1 : 0) ^ ((dip16 & (0x8000 >> 11))
? 1 : 0) ^((dip16 & (0x8000 >> 15)) ? 1 : 0));

}


//******************************************
// stop function
//******************************************
void spi4IfTvmT::stop()
 {
   while (portQueue.size() > 0)
     {
       tbvWaitCycle(tsclk, tbvThreadT::POSEDGE);
     }
   running = false;
   //tbvOut << " STOP_function at :   "<< tbvGetInt64Time(TBV_NS) << " ns" << endl;
 }



// ****************************************
// spi4IfTvmT methods
// ****************************************
void spi4IfTvmT::create()
{
  //
  // the base constructor automatically record this TVM instance
  // and will delete it at the end of the simulation.
  //
  new spi4IfTvmT();
}

spi4IfTvmT::spi4IfTvmT() : tbvTvmT(),
    tdclk(getFullInterfaceHdlNameP("tdclk")),
    tsclk(getFullInterfaceHdlNameP("tsclk")),
    reset(getFullInterfaceHdlNameP("reset")),
    tdat(getFullInterfaceHdlNameP("tdat_o")),
    tctl(getFullInterfaceHdlNameP("tctl_o")),
    tstat(getFullInterfaceHdlNameP("tstat")),

    // Initialize the Mutex
    spi4TDataMutex( "spi4TDataMutex" ),
    fullStatMutex( "fullStatMutex" ),
    queueStatMutex( "queueStatMutex" ),
    spi4TStatMutex( "spi4TStatMutex" ),
    portQueueMutex( "portQueueMutex" ),

    spi4IfFiber(this, "spi4IfFiber" ),
    fullStatFiber(this, "fullStatFiber" ),
    queueStatFiber(this, "queueStatFiber" ),

    spi4TData( * this),
    fullStat( * this, spi4TData),
    queueStat( * this, spi4TData),
    spi4Pattern( * this),
    spi4TStat( * this),
    runTStat( * this),
    trainingCnt( * this),
    running(false),
    cal_len(20),
    cal_m(5),
    old_data(),
    //status(),
    MaxBurst1(256,0),
    MaxBurst2(256,0),
    invalid(1),
    last_data_valid(0),
    last_data_bad(0),
    deskewed(1),
    flag_data(0),
    flag_deskewed(0),
    flag_cycles(0),
    data_max_t(65534),
    training_m(1),
    badParity(0),
    fullPort(0),

    // Create the Event Expressions
    anyedge_clk ( tdclk, tbvThreadT::ANYEDGE ),
    posedge_tsclk ( tsclk, tbvThreadT::POSEDGE ),
    not_reset ( reset , 1 )

{ }

spi4IfTvmT::~spi4IfTvmT() {}



