// ----------------------------------------------------------------------------
// (c) Copyright 2005, Springer.  All Rights Reserved. The code in this CD-ROM is
// distributed by Springer with ABSOLUTELY NO SUPPORT and NO WARRANTY from
// Springer. Use or reproduction of the information provided on this code for
// commercial gain is strictly prohibited. Explicit permission is given for the
// reproduction and use of this information in an instructional setting provided
// proper reference is given to the original source.
//
// Authors and Springer shall not be liable for damage in connection with, or
// arising out of, the furnishing, performance or use of the contents of the
// CD-ROM.
// ----------------------------------------------------------------------------

#ifndef INCLUDED_SPI4IF_TVM
#define INCLUDED_SPI4IF_TVM

#include "TestBuilder.h"
#include "tbvCell.h"
//#include <stl.h>
//#include <queue>


class spi4IfTvmT;


// ****************************************
// a task spi4TData   
// ****************************************

/// The main task which generates the controls and the data

class spi4TDataTaskT : public tbvTaskTypeSafeT<tbvCellT>
{
  public:
    //parent tvm, make it easy to access in task body
    spi4IfTvmT& spi4IfTvm;

    //constructor, destructor
    spi4TDataTaskT(spi4IfTvmT& );
    virtual ~spi4TDataTaskT();

    //body function for task functionality
    virtual void body(tbvCellT *TrGenArgsP );

    void trPattern_data();
};


// ****************************************
// a task fullStat   
// ****************************************

/// The main task which generates the controls and the data if non-blocking logic

/// This task should be spawned if non-blocking logic is defined -
/// If the status of the current port doesn't allow the data to be sent,
/// the task puts the transfer data into a queue and then goes to the next port -
/// If the queue is not empty and the status now is OK - the blocking task \a spi4TDataTaskT is called

class fullStatTaskT : public tbvTaskTypeSafeT<tbvCellT>
{
  private:
    spi4IfTvmT& spi4IfTvm;
    spi4TDataTaskT& mainTask;
    //vector<queue<tbvCellT *> > portQueue;

  public:
    fullStatTaskT(spi4IfTvmT& spi4IfTvm, spi4TDataTaskT & mainTask);
    virtual ~fullStatTaskT();

    //body function for task functionality
    virtual void body(tbvCellT *TrGenArgsP );
};


// ****************************************
// a task queueStat   
// ****************************************

/// This task is used if non-blocking logic is defined

/// This task should be spawned from \a spi4TStatTaskT if the generations are done, but
/// the queue is not empty and a favorable status update has happened
/// It calls the blocking task \a spi4TDataTaskT

class queueStatTaskT : public tbvTaskT
{
  private:
    spi4IfTvmT& spi4IfTvm;
    spi4TDataTaskT& mainTask;

  public:
    queueStatTaskT(spi4IfTvmT& spi4IfTvm, spi4TDataTaskT & mainTask);
    virtual ~queueStatTaskT();

    //body function for task functionality
    virtual void body(tbvSmartDataT * );
};


// ****************************************
// a task spi4Pattern   
// ****************************************

/// This task sends training pattern

/// runs freely (while running is TRUE) checking for any one of 3 possible flags that 
/// would require a training patterns to be sent. There are 3 possible reasons (flags): 
/// - Very beginning or Status remained idle (3) for too long
/// - There is no traffic and the time that has passed since last training pattern exceeds max allowed
/// - There is traffic and the time since last training patterns has not exceeded yet the max, 
/// but after the end of the current transfer the max time will be exceeded, 
/// so a training pattern would be needed before the beginning of the current transfer

class spi4PatternTaskT : public tbvTaskTypeSafeT<tbvCellT>
{
  public:
    //parent tvm, make it easy to access in task body
    spi4IfTvmT& spi4IfTvm;

    //constructor, destructor
    spi4PatternTaskT(spi4IfTvmT& );
    virtual ~spi4PatternTaskT();

    //body function for task functionality
    virtual void body(tbvCellT *TrGenArgsP );

    void trPattern_deskewed();
    void trPattern_cycles();

};


// ****************************************
// a task spi4TStat   
// ****************************************

/// This task is used for monitoring the status per port

/// The status task should be spawned by \a runTStatTaskT

class spi4TStatTaskT : public tbvTaskTypeSafeT<tbvCellT>
{
  public:
    //parent tvm, make it easy to access in task body
    spi4IfTvmT& spi4IfTvm;

    //constructor, destructor
    spi4TStatTaskT(spi4IfTvmT& );
    virtual ~spi4TStatTaskT();

    //body function for task functionality
    virtual void body(tbvCellT *TrGenArgsP );
};


// ****************************************
// a task runTStat   
// ****************************************

/// This task is used for running the status task \a spi4TStatTaskT

class runTStatTaskT : public tbvTaskT
{
  public:
    //parent tvm, make it easy to access in task body
    spi4IfTvmT& spi4IfTvm;

    //constructor, destructor
    runTStatTaskT(spi4IfTvmT& );
    virtual ~runTStatTaskT();

    //body function for task functionality
    virtual void body(tbvSmartDataT * );
};


// ****************************************
// a task trainingCnt
// ****************************************

/// runs freely (while running is TRUE) getting reset when a training pattern starts

/// If this count reaches a certain threshold a flag is being set to trigger a training pattern

class trainingCntTaskT : public tbvTaskT
{
  public:
    //parent tvm, make it easy to access in task body
    spi4IfTvmT& spi4IfTvm;

    //constructor, destructor
    trainingCntTaskT(spi4IfTvmT& );
    virtual ~trainingCntTaskT();

    //body function for task functionality
    virtual void body(tbvSmartDataT * );
};


// ****************************************
// a TVM spi4IfTvm
// ****************************************

/// The main class for implementing the spi4 generator

class spi4IfTvmT : public tbvTvmT
{
public:
  unsigned int status[256];
  unsigned int port_cal[256];
  unsigned int block16_cnt[256];
  unsigned int fl[256];
  unsigned int dip2;
  unsigned int curr_eop;
  unsigned int odd_byte;
  unsigned int dip4;
  unsigned int block_16;
  unsigned int thresh[256];

  //
  // These signals are connected to Verilog signals
  // through the test builder registry
  //
  tbvSignalHdlT tdclk;
  tbvSignalHdlT tsclk;
  tbvSignalHdlT reset;
  tbvSignalHdlT tdat;
  tbvSignalHdlT tctl;
  tbvSignalHdlT tstat;

  // The Mutex
  tbvMutexT spi4TDataMutex;
  tbvMutexT fullStatMutex;
  tbvMutexT queueStatMutex;
  tbvMutexT spi4TStatMutex;
  tbvMutexT portQueueMutex;

  tbvFiberT spi4IfFiber;
  tbvFiberT fullStatFiber;
  tbvFiberT queueStatFiber;

  //
  // basic tasks
  //
  spi4TDataTaskT spi4TData;
  fullStatTaskT fullStat;
  queueStatTaskT queueStat;
  spi4PatternTaskT spi4Pattern;
  spi4TStatTaskT spi4TStat;
  runTStatTaskT runTStat;
  trainingCntTaskT trainingCnt;
  bool running;
  void start(){running = true;}
  void stop();

  unsigned int cal_len;
  unsigned int cal_m;

  void setCal_len(unsigned int k){cal_len = k;}
  void setCal_m(unsigned int k){cal_m = k;}
  vector <uint16> old_data;
  vector <uint16> MaxBurst1;
  vector <uint16> MaxBurst2;
  unsigned int invalid;
  unsigned int last_data_valid;
  unsigned int last_data_bad;
  unsigned int deskewed;

  unsigned int flag_data;
  unsigned int flag_deskewed;
  unsigned int flag_cycles;
  unsigned int size4pattern;
  unsigned int training_count;
  unsigned int data_max_t;
  unsigned int training_m;
  void setData_Max_T(unsigned int k){data_max_t = k;}
  void setTraining_M(unsigned int k){training_m = k;}

  unsigned int badParity;
  void setBadParity(unsigned int k){badParity = k;}

  unsigned int fullPort;
  void setFullPort(unsigned int k){fullPort = k;}

  vector<tbvCellT *> portQueue;

/// Calculates Odd Parity
  unsigned int dip4_calc(unsigned int, unsigned int, vector <uint16> &);


  // Event Expressions
  tbvEventExprT anyedge_clk;
  tbvEventExprT posedge_tsclk;
  tbvEventExprT not_reset;

  //
  // This function is called from $tbv_tvm_connect, to instantiate this TVM.
  //
  static void create();

  //
  // Constructor / Destructor
  //
  spi4IfTvmT();
  virtual ~spi4IfTvmT();
};


#endif


