// ----------------------------------------------------------------------------
// (c) Copyright 2005, Springer.  All Rights Reserved. The code in this CD-ROM is
// distributed by Springer with ABSOLUTELY NO SUPPORT and NO WARRANTY from
// Springer. Use or reproduction of the information provided on this code for
// commercial gain is strictly prohibited. Explicit permission is given for the
// reproduction and use of this information in an instructional setting provided
// proper reference is given to the original source.
//
// Authors and Springer shall not be liable for damage in connection with, or
// arising out of, the furnishing, performance or use of the contents of the
// CD-ROM.
// ----------------------------------------------------------------------------

#include "tbvATMCell.h"
#define HEC_GENERATOR   0x107               /* x^8 + x^2 +  x  + 1  */
#define COSET_LEADER    0x055               /* x^6 + x^4 + x^2 + 1  */


/**************************************************************************/
// Needs to be called with 48(payload size). Header needs to be set to 4.
tbvATMCellT::tbvATMCellT(bool nni = true,
			 const tbvControlDataT * controlP = NULL,
    		 	 const char * nameP = NULL) :
	        tbvCellT(48, controlP, nameP),
		_vpi(this,"vpi",true),
		_vci(this,"vci",true),
		_pti(this,"pti",true), 
                _atm_header_clp(this,"_atm_header_clp",true),
                _ishec(false) 
{ 
  _nni = nni;
  setup(); 
}


/**************************************************************************/
// Header needs to set to 4.
tbvATMCellT::tbvATMCellT(
			 const char * header,
			 const char * payload,
			 bool nni = true,
			 const tbvControlDataT * controlP = NULL, 
    		   	 const char * nameP = NULL) :
	        tbvCellT(payload,48,controlP, nameP), 
		_vpi(this,"vpi",true),
		_vci(this,"vci",true),
		_pti(this,"pti",true),  
		_atm_header_clp(this,"_atm_header_clp",true),
                _ishec(false)

{ 
  _nni = nni;
  setHeader(header);
  setup();
  
}


/**************************************************************************/
tbvATMCellT::tbvATMCellT(
			 const vector<uint8> & header,
			 const vector<uint8> & payload, 
			 bool nni = true,
		 	 const tbvControlDataT * controlP = NULL,
    		 	 const char * nameP = NULL) :
	        tbvCellT(payload, controlP, nameP),
                _vpi(this,"vpi",true),
		_vci(this,"vci",true),
		_pti(this,"pti",true),  
		_atm_header_clp(this,"_atm_header_clp",true),
                _ishec(false)

{
  _nni = nni;
  setHeader(header);
  setup();
} 


/**************************************************************************/
tbvATMCellT::tbvATMCellT(const tbvATMCellT & rhs, 
    		 const char * nameP = NULL) :
	         tbvCellT(0,NULL,nameP),
		 _vpi(this,"vpi",true),
		 _vci(this,"vci",true),
		 _pti(this,"pti",true),  
                 _atm_header_clp(this,"_atm_header_clp",true),
                _ishec(false)

 {
 operator=(rhs);
 setup(); } 

/**************************************************************************/
tbvATMCellT::tbvATMCellT(const tbvCellT & rhs, 
    		 const char * nameP = NULL) :
	         tbvCellT(rhs, nameP)
{ operator=(rhs);
  setup();
} 

/**************************************************************************/
tbvATMCellT::tbvATMCellT(const tbvSmartDataT * parentP, const char * nameP = NULL,
		 bool derived = false) :
	       tbvCellT(parentP, nameP, derived) { setup(); }


/**************************************************************************/
tbvATMCellT & tbvATMCellT::operator=(const tbvATMCellT & rhs)
{
  tbvCellT::operator=(rhs);
    _ishec=rhs.isHec();
   _nni = rhs.getNNI(); 
  return *this;
}


/**************************************************************************/
tbvATMCellT * tbvATMCellT::duplicate(const char * nameP = NULL) const
{
  return new tbvATMCellT(*this, nameP);
}


/**************************************************************************/
tbvATMCellT * tbvATMCellT::duplicate(const tbvSmartDataT * parentP,
                                     const char * nameP = NULL,
                                     bool derived = false ) const
{
  tbvATMCellT * a = new tbvATMCellT(parentP, nameP, derived);
  *a = *this;
  return a;
}


  /**************************************************************************/
  tbvATMCellT::~tbvATMCellT() { wrapup(); }

/**************************************************************************/
void tbvATMCellT::wrapup()
{
}

/**************************************************************************/
void tbvATMCellT::setup() 
{ 
  if (!_nni) {
    _gfc.keepOnly(0,15); 
    _vpi.keepOnly(0,255);}
  else{
    _gfc = 0;
    _vpi.keepOnly(0,4095);}

  _vci.keepOnly(0,65535);
  _pti.keepOnly(0,7);
  _atm_header_clp.keepOnly(0,1);

  if (_payload.size() != 48)
  {
     tbvExceptionT::setUserExceptionTypeString("PAYLOAD_SIZE_MISMATCH");
     tbvExceptionT::reportUserException("PAYLOAD_SIZE_MISMATCH",
                                        tbvExceptionT::ERROR,
                                        "Atm cell payload size should be 48 bytes not %d", _payload.size());
  }
  if(_ishec) {
      setHeaderSize(5);
   }
  else
      setHeaderSize(4);
  setSOP(1);
  setEOP(1);
}


/**************************************************************************/
void tbvATMCellT::setGFC(const unsigned int gfc) 
{
  _gfc = gfc;
  if (_nni || !_gfc.satisfyConstraints())
  {
    tbvExceptionT::setUserExceptionTypeString("CONSTRAINT_VIOLATION");
    tbvExceptionT::reportUserException("CONSTRAINT_VIOLATION",
                                       tbvExceptionT::ERROR,
                                       "GFC field constraint violated");
   } 
} 


/**************************************************************************/
void tbvATMCellT::setVPI(const unsigned int vpi) 
{
  _vpi = vpi;
  if (!_vpi.satisfyConstraints())
  {
    tbvExceptionT::setUserExceptionTypeString("CONSTRAINT_VIOLATION");
    tbvExceptionT::reportUserException("CONSTRAINT_VIOLATION",
                                       tbvExceptionT::ERROR,
                                       "VPI field constraint violated");
   } 
} 


/**************************************************************************/
void tbvATMCellT::setVCI(const unsigned int vci) 
{
  _vci = vci;
  if (!_vci.satisfyConstraints())
  {
    tbvExceptionT::setUserExceptionTypeString("CONSTRAINT_VIOLATION");
    tbvExceptionT::reportUserException("CONSTRAINT_VIOLATION",
                                       tbvExceptionT::ERROR,
                                       "VCI field constraint violated");
   } 
} 


/**************************************************************************/
void tbvATMCellT::setPTI(const unsigned int pti) 
{
  _pti = pti;
  if (!_pti.satisfyConstraints())
  {
    tbvExceptionT::setUserExceptionTypeString("CONSTRAINT_VIOLATION");
    tbvExceptionT::reportUserException("CONSTRAINT_VIOLATION",
                                       tbvExceptionT::ERROR,
                                       "PTI field constraint violated");
   } 
} 


/**************************************************************************/
const tbvSmartUnsignedT &  tbvATMCellT::getGFC() const 
{
  if (_nni)
  {
    tbvExceptionT::setUserExceptionTypeString("INVALID_FIELD");
    tbvExceptionT::reportUserException("INVALID_FIELD",
                                       tbvExceptionT::ERROR,
                                       "Not an UNI cell");
  } 
  return _gfc;
} 

/************************************************************************/
bool tbvATMCellT::isHec() const{

  return _ishec;
}
/*************************************************************************/
void		tbvATMCellT::setHec() {

  _ishec = true;
  _hec = 0x0;

   setHeaderSize(5);
} 

/**************************************************************************/
 
tbvSmartUnsignedT &    tbvATMCellT::getVPI()  
{
  return _vpi;
} 


/**************************************************************************/
tbvSmartUnsignedT &    tbvATMCellT::getVCI()  
{
  return _vci;
} 


/**************************************************************************/
tbvSmartUnsignedT &  tbvATMCellT::getPTI()  
{
  return _pti;
} 


/**************************************************************************/
const bool  tbvATMCellT::getNNI() const 
{
  return _nni;
} 


/**************************************************************************/
void  tbvATMCellT::setPTIUseControlData() 
{
  unsigned int oam = getOAM().getUnsignedValue();
  unsigned int efci = getEFCI().getUnsignedValue();

  // EOP bit is left unchanged
   _pti = (_pti & 0x01) | (oam << 2) | (efci << 1);
} 


/**************************************************************************/


vector <uint8>  tbvATMCellT::getHeader() const
{
  vector <uint8> x = tbvCellT::getHeader();
  if(_ishec)
 	 x.resize(5);
  else
	x.resize(4);

  if (_nni)
    x[0] = (_vpi>>4) & 0xff; 
  else 
    x[0] = ((_gfc & 0x0f) << 4) | ((_vpi & 0xf0) >> 4);
  
  x[1] = ((_vpi & 0x0f) << 4) | ((_vci & 0xf000) >> 12);
  x[2] = (_vci & 0xff0) >> 4;
  x[3] = ((_vci & 0x0f) << 4) | ((_pti & 0x07)<<1) | _atm_header_clp; 
  if(_ishec)
  	x[4]=_hec;
  //x[4] = crc10(x,4);
  
  return x;

}


/*************************************************************************/
/*
vector <uint8>  tbvATMCellT::getHeader() const
{
 // vector<uint8>  y = _payload;
 // y.resize(4);
 // y.erase(y.begin()+4,y.end());
 // vector<uint8> &  x = y;
  return _header;
}
*/

/****************************************************/
vector<uint8>  tbvATMCellT::getPayload() const
{

  vector<uint8>  payloadP =  vector<uint8>(_payload);

  if(getOAM().getUnsignedValue()){
     (payloadP)[5] = (payloadP)[5]|0x08;
  }

  return payloadP;
}


/**************************************************************************/
void tbvATMCellT::createHeader(const vector<uint8> &y) 
{
    vector<uint8> x;
    if(_ishec){
   	 x.insert(x.begin(),y.begin(),y.begin()+5);
    }
    else{
	 x.insert(x.begin(),y.begin(),y.begin()+4); 
    }

  if (_nni)
    _vpi = (x[0]<<4 & 0xff0) | (x[1]>>4 & 0x00f);
    
  else{
    _gfc = x[0]>>4 & 0x0f;
    _vpi = (x[0]<<4 & 0xf0) | (x[1]>>4 & 0x0f);
  }
  
  _vci = ((x[1] & 0x000f)<<12) | ((x[2]&0xff)<<4) | ((x[3] & 0xf0)>>4);
    
   
   if(_ishec)
	_hec = x[4];
  _pti = (x[3] & 0x0e)>>1;
  _atm_header_clp = (x[3] & 0x01);
}


/**************************************************************************/
void tbvATMCellT::setHeader(const vector<uint8> &x) 
{
  if (_ishec == true & x.size() != 5) {
     tbvExceptionT::setUserExceptionTypeString("HEADER_SIZE_MISMATCH");
     tbvExceptionT::reportUserException("HEADER_SIZE_MISMATCH",
                                        tbvExceptionT::ERROR,
                                        "Atm cell header size should be 5 bytes not %d", x.size());
  }
  if(_ishec == false & x.size()!=4) {
	tbvExceptionT::setUserExceptionTypeString("HEADER_SIZE_MISMATCH");
        tbvExceptionT::reportUserException("HEADER_SIZE_MISMATCH",
        				    tbvExceptionT::ERROR,
					    "Atm cell header size should be 5 bytes not %d", x.size());
  }
createHeader(x);
}

/**************************************************************************/
void tbvATMCellT::setPayload(const vector<uint8> &x) 
{

  if (x.size() != 48) {

tbvExceptionT::setUserExceptionTypeString("PAYLOAD_SIZE_MISMATCH");
     tbvExceptionT::reportUserException("PAYLOAD_SIZE_MISMATCH",
                                        tbvExceptionT::ERROR,
                                        "ATM cell payload size should be 48 bytes not %d ", x.size());
  }

  _payload = x;

       
}


/**************************************************************************/
void tbvATMCellT::setPayload(const char * payload, int sz)
{
  vector <uint8> temp;
  for (int i = 0; i < sz; i++)
  {
    temp.push_back((uint8) *payload);
    payload++;
  }
  
  setPayload(temp);
}


/**************************************************************************/
void tbvATMCellT::setHeader(const char * header)
{ 
  vector <uint8> _header;
  if(_ishec) {
  	for (int i = 0; i < 5; i++)
  	{
    		_header.push_back((uint8) *header);
    		 header++;
  	}

  }
  else{
	for (int i = 0; i < 4; i++)
        {
                _header.push_back((uint8) *header);
                 header++;
        }
  }

  createHeader(_header);

}

/***********************************************************************/

/**************************************************************************/
unsigned long long tbvATMCellT::getKey() const {
    unsigned long long key = 0;
    key = ((unsigned long long)getInputPort()) << 28 | _vpi.getUnsignedValue() << 16 | _vci.getUnsignedValue();
    return key;
}

void tbvATMCellT::randomHeader() {
   _vpi.randomize();
   _vci.randomize();
   _pti.randomize();
   _atm_header_clp.randomize();
}
	

/**************************************************************************/
void tbvAAL1CellT::setHeader(const vector<uint8> &x) 
{createHeader(x);
}


/**************************************************************************/
void tbvAAL1CellT::setHeader(const char * header)
{ 
  vector <uint8> _header;
  
  
  for (int i = 0; i < 1; i++)
  {
    _header.push_back((uint8) *header);
    header++;
  }
  createHeader(_header);

}


/**************************************************************************/
void tbvATMCellT::createPayload(const vector<uint8> &x) 
{
  _payload.erase(_payload.begin(), _payload.end());
   if(_ishec){
 
	  _payload.insert(_payload.begin(), x.begin() + 5, x.begin()+53);
   }
   else {
	  _payload.insert(_payload.begin(), x.begin() + 4, x.begin()+52);
   }
}


/**************************************************************************/

DATACOM_TYPE tbvATMCellT::getType()
{
  return ATM_CELL;
}

/**************************************************************************/ 

vector <uint8>  tbvAAL1CellT::getHeader()const {

  vector <uint8> x = tbvCellT::getHeader();

  x.resize(1);
  
  x[0] = (getSN() & 0x07)<<4 | (crc(0xbU,getSN().getUnsignedValue()) << 1);
    
  return x;
  
}

/****************************************************/
vector<uint8>  tbvAAL1CellT::getPayload() const
{
  vector<uint8>  payloadP =  vector<uint8>(_payload);
    return payloadP;
}

/**************************************************************************/
void tbvAAL1CellT::createHeader(const vector<uint8> &x) {

  if (x.size() != 1) {
     tbvExceptionT::setUserExceptionTypeString("HEADER_SIZE_MISMATCH");
     tbvExceptionT::reportUserException("HEADER_SIZE_MISMATCH",
                                        tbvExceptionT::ERROR,
                                        "AAL1 cell header size should be 1 byte not %d", x.size());
  }
  setSN( (x[0] & 0x70) >> 4);
  if (crc(0xb,getSN()) != unsigned((x[0] & 0x0e) >> 1))
    tbvExceptionT::setUserExceptionTypeString("INVALID CRC");
  tbvExceptionT::reportUserException("INVALID CRC",
				     tbvExceptionT::ERROR,
				     "Invalid CRC");
  _csi = (x[0] & 0x80) >> 7;

}
/**************************************************************************/

void tbvAAL1CellT::createPayload(const vector<uint8> &x) {
 
  _payload.erase(_payload.begin(), _payload.end());
  _payload.insert(_payload.begin(), x.begin() + 1, x.end());
  _pointer = x[0];
}

/**************************************************************************/



/**************************************************************************/
tbvAAL1CellT::tbvAAL1CellT(bool csi = false,
			 const tbvControlDataT * controlP = NULL,
    		 	 const char * nameP = NULL) :
  tbvCellT(47, controlP, nameP),
  _pointer(this,"pointer",true)
{ 
  _csi = csi;
  setup(); 
}


/**************************************************************************/
tbvAAL1CellT::tbvAAL1CellT(
			 const char * header,
			 const char * payload,
			 bool csi = false,
		   	 const tbvControlDataT * controlP = NULL, 
    		   	 const char * nameP = NULL) :
	        tbvCellT(payload,47,controlP, nameP), 
                _pointer(this,"pointer",true)		
{ 
  _csi = csi;
  setHeader(header);
  setup();
}


/**************************************************************************/
tbvAAL1CellT::tbvAAL1CellT(
			   const vector<uint8> & header,
			   const vector<uint8> & payload, 
			   bool csi = false,
			   const tbvControlDataT * controlP = NULL,
    		 	 const char * nameP = NULL) :
	        tbvCellT(payload, controlP, nameP),
	_pointer(this,"pointer",true)

{
  _csi = csi;
  setHeader(header);
  setup();
} 


/**************************************************************************/
tbvAAL1CellT::tbvAAL1CellT(const tbvAAL1CellT & rhs, 
    		 const char * nameP = NULL) :
	         tbvCellT(rhs, nameP),
		_pointer(this,"pointer",true)

 { operator=(rhs);
   setup();
  } 


/**************************************************************************/
tbvAAL1CellT::tbvAAL1CellT( const tbvSmartDataT * parentP, const char * nameP = NULL,
		 bool derived = false) :
  tbvCellT(parentP, nameP, derived),
  _pointer(this,"pointer",true)
 { 
     setup(); }


/**************************************************************************/
tbvAAL1CellT & tbvAAL1CellT::operator=(const tbvAAL1CellT & rhs)
{
  tbvCellT::operator=(rhs);
   _csi = rhs.getCSI(); 
  return *this;
}


/**************************************************************************/
tbvAAL1CellT * tbvAAL1CellT::duplicate(const char * nameP = NULL) const
{
  return new tbvAAL1CellT(*this, nameP);
}


/**************************************************************************/
tbvAAL1CellT * tbvAAL1CellT::duplicate(const tbvSmartDataT * parentP,
                                     const char * nameP = NULL,
                                     bool derived = false ) const
{
  tbvAAL1CellT * a = new tbvAAL1CellT(parentP, nameP, derived);
  *a = *this;
  return a;
}


/**************************************************************************/
tbvAAL1CellT::~tbvAAL1CellT() { wrapup(); }
/**************************************************************************/


void tbvAAL1CellT::wrapup(){
}

/**************************************************************************/


void tbvAAL1CellT::setup(){
  setHeaderSize(1);
  _pointer.keepOnly(0,92);
}

/**************************************************************************/
void tbvAAL1CellT::setNextMsgPointer(const unsigned int pointer) 
{
  _pointer = pointer;
  if (!_pointer.satisfyConstraints() || (!_csi) )
  {
    tbvExceptionT::setUserExceptionTypeString("CONSTRAINT_VIOLATION");
    tbvExceptionT::reportUserException("CONSTRAINT_VIOLATION",
                                       tbvExceptionT::ERROR,
                                       "Pointer field constraint violated");
   } 
} 


/**************************************************************************/
const tbvSmartUnsignedT &  tbvAAL1CellT::getNextMsgPointer() const 
{
  if (!_csi)
  {
    tbvExceptionT::setUserExceptionTypeString("INVALID_FIELD");
    tbvExceptionT::reportUserException("INVALID_FIELD",
                                       tbvExceptionT::ERROR,
                                       "Not an UNI cell");
  } 
  return _pointer;
} 


/**************************************************************************/
const bool tbvAAL1CellT::getCSI() const 
{
  return _csi;
} 


/**************************************************************************/
void  tbvATMCellT::setATMHeaderCLP(const unsigned int clp)
{
    _atm_header_clp = clp;
}


/**************************************************************************/
const unsigned int   tbvATMCellT::getATMHeaderCLP() const
{
   return _atm_header_clp;
}
/**************************************************************************/
void  tbvATMCellT::setATMHeaderEFCI(const unsigned int efci)
{
  _pti = _pti | (efci << 1);
}


/**************************************************************************/
const unsigned int   tbvATMCellT::getATMHeaderEFCI() const
{
   return (_pti & 0x2)>>1;
}
/**************************************************************************/
void  tbvATMCellT::setATMHeaderOAM(const unsigned int oam)
{
  _pti = _pti | (oam << 2);
}


/**************************************************************************/
const unsigned int   tbvATMCellT::getATMHeaderOAM() const
{
  return (_pti & 0x4)>>2;
}
/**************************************************************************/
void  tbvATMCellT::setPTIEOP(const unsigned int eop){
  _pti = _pti | (eop);
}
/**************************************************************************/
unsigned short tbvATMCellT::getPTIEOP(){
  return _pti & 0x1;
}
