/*
 * SSLEngineManager.java
 *
 * Copyright Esmond Pitt, 2005. All rights reserved.
 *
 * Created on 30 November 2004, 18:00
 */

package javanet.ssl;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.net.*;
import java.util.*;
import javax.net.*;
import javax.net.ssl.*;

/**
 * The SSLEngineManager is a higher-level wrapper for javax.net.ssl.SSLEngine.
 * It takes care of all handshaking except for blocking tasks, so the only statuses the caller needs to be aware
 * of are the EngineStatus codes:
 *<dl>
 *<dt>OK
 *<dd>The operation completed successfully.
 *<dt>BUFFER_UNDERFLOW
 *<dd>This can be returned by SSLEngineManager.unwrap().
 * It means that there is insufficient data in the appRecvBuffer to proceed.
 * The application should not retry the SSLEngineManager.unwrap() until at least one byte of data has been read into the appRecvBuffer.
 * If the ReadableByteChannel supplied is non-null and non-blocking, the SSLEngineManager will already have
 * tried a read on the channel, and only returns this status if that read returned zero (which can
 * only happen if the channel is non-blocking). In this way the SSLEngineManager makes its best attempt
 * to proceed without ever blocking the application.
 *
 *<dt>BUFFER_OVERFLOW
 *<dd>This can be returned by SSLEngineManager.wrap() if there is insufficient room in the target buffer.
 * The application should not retry the SSLEngineManager.wrap() until at least one byte of net data
 * has been written from the netSendBuffer. If the ReadableByteChannel supplied is non-null and non-blocking,
 * the SSLEngineManager will already have tried a write on the channel, and only returns this status
 * if that write returned zero. In this way the SSLEngineManager makes its best attempt to proceed
 * without ever blocking the application.
 *<p>
 * BUFFER_OVERFLOW can also be returned by SSLEngineManager.unwrap() if there is unsufficient
 * room in the appRecvBuffer for the unwrapped data.
 * The application should remove application data from the AppRecvBuffer and retry the wrap.
 * The conditions under which this happens are dependent on the implementation of SSLEngine:
 * in Sun's JDK 1.5 implementation it is returned on unwrap() if the application receive buffer
 * isn't completely empty. This is pretty drastic but can be expected to improve in later releases.
 * In any case all the application has to do is respond to the condition by emptying the buffer
 * (i.e. draining it to an application-side buffer).
 *
 *<dt>CLOSED
 *<dd>This means that the SSLEngine is closed correctly. All the application can sensibly do
 * is close the connection directly (i.e. not via SSLEngineManager.close()).
 *</dl>
 * and the EngineResult.HandshakeStatus codes:
 *<dl>
 *<dt>NEED_TASK
 *<dd>As in SSLEngine, this means that a potentially blocking task must be executed.
 *</dl>
 * The application and net buffers are maintained on these principles:
 *<ul>
 *<li>
 * they are always ready for a read() or put() operation, or to be the target of a wrap or unwrap;
 *<li>
 * they need to be flipped before a write() or get() operation and compacted afterwards.
 * This also applies to the source buffer of a wrap() or unwrap().
 *</ul>
 *<p>
 * State machine:
 *<pre>
 * HandshakeStatus	Status				Action
 * NEED_UNWRAP		BUFFER_OVERFLOW		application must read appRecvBuffer and retry.
 * NEED_UNWRAP		BUFFER_UNDERFLOW	read from network into netRecvBuffer and retry.
 * NEED_WRAP		BUFFER_OVERFLOW		write from netSendBuffer to network and retry.
 * NEED_WRAP		BUFFER_UNDERFLOW	Doesn't seem possible.
 *</pre>
 * In Sun's implementation there are no other ways to get the BUFFER_UNDERFLOW/BUFFER_OVERFLOW handshake statuses.
 * In JDK 1.5.0 the UNWRAP/BUFFER_OVERFLOW condition is triggered far too often: presently it happens
 * if the target buffer isn't completely empty, regardless of how much room is really needed.
 * I expect Sun to improve this over time.
 * There is a discussion on the java-security mailing list about this.
 *<p>
 * In JDK 1.5 it should be noted that SSLSocket/SSLServerSocket don't actually use
 * the SSLEngine so its usage and reliability are open to question. 
 *<p>
 * Copyright (c) Esmond Pitt, 2005. All rights reserved.
 *
 * @author Esmond Pitt
 * @version $Revision: 6 $
 */
public class SSLEngineManager
{
	private SocketChannel	channel;
	private SSLEngine		engine;
	private ByteBuffer	appSendBuffer;
	private ByteBuffer	netSendBuffer;
	private ByteBuffer	appRecvBuffer;
	private ByteBuffer	netRecvBuffer;
	private SSLEngineResult	engineResult = null;
	boolean	atEOF = false;
	
	/**
	 * Creates new SSLEngineManager.
	 *
	 * @param channel Socket channel (the client or server end of TCP/IP connection)
	 * @param engine: SSLEngine: must be in the appropriate client or server mode
	 * and must be otherwise fully configured (e.g. want/need/client/server/auth, cipher
	 * suites, &c) before this manager is used.
	 */
	public SSLEngineManager(SocketChannel channel,SSLEngine engine) throws IOException
	{
		this.channel = channel;
		this.engine = engine;
		this.appSendBuffer = ByteBuffer.allocate(engine.getSession().getApplicationBufferSize());
		this.netSendBuffer = ByteBuffer.allocate(engine.getSession().getPacketBufferSize());
		this.appRecvBuffer = ByteBuffer.allocate(engine.getSession().getApplicationBufferSize());
		this.netRecvBuffer = ByteBuffer.allocate(engine.getSession().getPacketBufferSize());
	}
	
	/** @return the application receive buffer	*/
	public ByteBuffer	getAppRecvBuffer()
	{
		return appRecvBuffer;
	}

	/** @return the application send buffer */
	public ByteBuffer	getAppSendBuffer()
	{
		return appSendBuffer;
	}
	
	/** @return the SSLEngine */
	public SSLEngine	getEngine()
	{
		return engine;
	}
	
	/** @return the number of bytes that can be read immediately. */
	public int	available()
	{
		return atEOF ? -1 : appRecvBuffer.position();
	}
	
	public SSLEngineResult	getEngineResult()
	{
		return engineResult;
	}
	
	/**
	 * Read from the channel via the SSLEngine into the application receive buffer.
	 * Called in blocking mode when input is expected, or in non-blocking mode when the channel is readable.
	 */
	public int	read() throws IOException
	{
		flush();
		if (atEOF)
			return -1;
		// NB none, some, or all of the data required may already have been
		// unwrapped/decrypted, or EOF may intervene at any of these points.
		int	count = channel.read(netRecvBuffer);
//		System.out.println("read: read "+count+" netRecv="+netRecvBuffer);
		netRecvBuffer.flip();
		engineResult = engine.unwrap(netRecvBuffer,appRecvBuffer);
		netRecvBuffer.compact();
		processEngineResult();
		if (engineResult.getStatus() == SSLEngineResult.Status.CLOSED)
		{
			atEOF = true;
			return -1;
		}
		count = appRecvBuffer.position();
//		System.out.println("read: count="+count+" netRecv="+netRecvBuffer+" appRecv="+appRecvBuffer+" remaining="+appRecvBuffer.remaining()+"; produced="+engineResult.bytesProduced());
		return count;
	}
	
	/**
	 * Write from the application send buffer to the channel via the SSLEngine.
	 * Called in either blocking or non-blocking mode when application output is ready to send.
	 */
	public int	write() throws IOException
	{
		int	count = appSendBuffer.position();
//		System.out.println("write: position="+appSendBuffer.position());
		if (count > 0)
		{
//			System.out.println("write: before flip/wrap appSend="+appSendBuffer+" netSend="+netSendBuffer);
			appSendBuffer.flip();
			engineResult = engine.wrap(appSendBuffer,netSendBuffer);
			appSendBuffer.compact();
//			System.out.println("write: after flip/wrap/compact appSend="+appSendBuffer+" netSend="+netSendBuffer);
			processEngineResult();
		}
		count -= appSendBuffer.position();
		// return count of bytes written.
		return count;
	}
	
	public int	flush() throws IOException
	{
//		System.out.println("flush: netSend="+netSendBuffer);
		netSendBuffer.flip();
		int	count = channel.write(netSendBuffer);
		netSendBuffer.compact();
//		System.out.println("flush: wrote "+count+" netSend="+netSendBuffer);
		return count;
	}
	
	public void	close() throws IOException
	{
		// try a read
		// TODO what to do in blocking mode ...
		if (!atEOF && !channel.isBlocking())
			read();
//		System.out.println("close: closing outbound");
		engine.closeOutbound();
		processEngineResult();
		System.out.println("close: closeOutbound handshake");
		while (!engine.isOutboundDone())
		{
			flush();
			processEngineResult();
		}
		System.out.println("close: closeOutbound complete");
//		System.out.println("close: closing inbound");
		engine.closeInbound();
//		System.out.println("close: closeInbound handshake");
		while (!engine.isInboundDone())
		{
			read();
		}
//		System.out.println("close: closeInbound complete");
		channel.close();
		System.out.println("close: SSLEngine & channel closed");
	}
	
	void processEngineResult() throws IOException
	{
		while (processStatus() && processHandshakeStatus())
			continue;
	}
	
	boolean	processHandshakeStatus() throws IOException
	{
		int	count;
		// process handshake status
//		System.out.println(engineResult);
		switch (engine.getHandshakeStatus())
		{
		case NOT_HANDSHAKING:	// not presently handshaking => session is available
		case FINISHED:			// just finished handshaking, SSLSession is available
			return false;
		case NEED_TASK:
			runDelegatedTasks();
			// TODO need to do something to engineResult to stop it looping here forever
			return true;	// keep going
		case NEED_WRAP:
			// output needed
//			System.out.println("before flip/wrap/compact appSend="+appSendBuffer+" netSend="+netSendBuffer);
			appSendBuffer.flip();
			engineResult = engine.wrap(appSendBuffer,netSendBuffer);
			appSendBuffer.compact();
//			System.out.println("after flip/wrap/compact appSend="+appSendBuffer+" netSend="+netSendBuffer);
			return true;
		case NEED_UNWRAP:
			// Sometimes we get > 1 handshake messages at a time ...
//			System.out.println("unwrapping netRecv="+netRecvBuffer+" appRecv="+appRecvBuffer);
			netRecvBuffer.flip();
			engineResult = engine.unwrap(netRecvBuffer,appRecvBuffer);
			netRecvBuffer.compact();
//			System.out.println("after flip/unwrap/compact netRecv="+netRecvBuffer+" appRecv="+appRecvBuffer);
			if (engineResult.getStatus() == SSLEngineResult.Status.BUFFER_OVERFLOW)
				return false;	// read data is ready but no room in appRecvBuffer
			return true;
		default:	// unreachable, just for compiler
			return false;
		}
	}
	
	boolean	processStatus() throws IOException
	{
		int	count;
		// processs I/O
		System.out.println(engineResult);
		switch (engineResult.getStatus())
		{
		case OK:		// OK: packet was sent or received
			return true;	// true;
		case CLOSED:	// Orderly SSL termination from either end
			return engineResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING;
		case BUFFER_OVERFLOW:
			// output needed
			switch (engineResult.getHandshakeStatus())
			{
			case NEED_WRAP:
				// If we are wrapping we are doing output to the channel,
				// and we can continue if we managed to write it all.
				flush();
				return netSendBuffer.position() == 0;
			case NEED_UNWRAP:
				// If we are unwrapping we are doing input from the channel
				// but the overflow means there is no room in the appRecvBuffer,
				// so the application has to empty it.
				// fall through
				System.out.println("netSendBuffer="+netSendBuffer);
				System.out.println("netRecvBuffer="+netRecvBuffer);
				System.out.println("appSendBuffer="+appSendBuffer);
				System.out.println("appRecvBuffer="+appRecvBuffer);
				return false;
			default:
				return false;
			}
		case BUFFER_UNDERFLOW:
			// input needed, existing data too short to unwrap
			System.out.println("netSendBuffer="+netSendBuffer);
			System.out.println("netRecvBuffer="+netRecvBuffer);
			System.out.println("appSendBuffer="+appSendBuffer);
			System.out.println("appRecvBuffer="+appRecvBuffer);
			// Underflow can only mean there is no data in the netRecvBuffer,
			// so try a read. We can continue if we managed to read something,
			// otherwise the application has to wait (select on OP_READ).
			// First flush any pending output.
			flush();
			// now read
//			System.out.println("underflow: netRecv="+netRecvBuffer);
			count = channel.read(netRecvBuffer);
			System.out.println("underflow: read "+count+" netRecv="+netRecvBuffer);
			// If we didn't read anything we want to exit processEngineStatus()
			return count > 0;
		default:	// unreachable, just for compiler
			return false;
		}
	}
	
	/**
	 * Run delegated tasks.
	 * This implementation runs all the presently existing delegated tasks in a single new thread.
	 * Derived classes should override this method to use a strategy
	 * appropriate to their environment, e.g.
	 * <ul>
	 * <li>dispatching into a single existing thread,
	 * <li>a new thread per task,
	 * <li>a thread-pool, &c.
	 *</ul>
	 */
	int	threadNumber = 1;
	
	protected void	runDelegatedTasks()
	{
		Thread	delegatedTaskThread = new Thread("DelegatedTaskThread-"+(threadNumber++))
		{
			public void	run()
			{
				// run delegated tasks
				Runnable	task;
				while ((task = engine.getDelegatedTask()) != null)
				{
					System.out.println(this.getName()+".runDelegatedTask: "+task);
					task.run();
				}
			}
		};
		if (false)
			delegatedTaskThread.start();
		else
			delegatedTaskThread.run();
	}
}

