package javanet.tcp;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * TCPServer demonstrates the simplest possible TCP server.
 *
 * @author Esmond Pitt
 * @version $Revision: 4 $
 */
public class TCPServer implements Runnable
{
	static final int	Port = 1100;
	// class data
	static long[]	events = new long[1000000];
	// instance data
	ServerSocket	server;
	int	connections = 0;
	
	TCPServer() throws IOException
	{
		this.server = new ServerSocket(Port);
		System.out.println("Bound to port "+server.getLocalPort());
	}
	
	public void	run()
	{
		int	connections = 0;
		for (;;)
		{
			try
			{
				Socket	s = server.accept();
				connections++;
				handleConnection(s);
			}
			catch (SocketException e)
			{
				if (e.getMessage().equalsIgnoreCase("Socket closed"))
					break;
				e.printStackTrace();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
		}
		System.out.println(Thread.currentThread().getName()+" handled "+connections+"/"+this.connections+" connections");
	}
		
	void	handleConnection(Socket s)
	{
		try
		{
			// count the connection
			events[this.connections++] = System.currentTimeMillis()/1000;
			// do RMI overhead
			InetAddress	ha = s.getInetAddress();
			String	ipa = ha.getHostAddress();
			Thread.currentThread().setName(ipa+"/"+this.connections);
			// ...
			try
			{
				s.setTcpNoDelay(true);
			}
			catch (SocketException e)	{}
			handleConversation(s);
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		finally
		{
			try { s.close();	} catch (IOException e)	{}
		}
	}

	/**
	 * Handle all requests coming down the socket.
	 * In a real deployment this would be abstract.
	 * @param s Socket connected to client: do not close.
	 */
	public void	handleConversation(Socket s)
		throws IOException
	{
		System.out.println(new Date()+" accepted "+s.getInetAddress()+":"+s.getPort());
		InputStream	in = new BufferedInputStream(s.getInputStream());
		OutputStream	out = new BufferedOutputStream(s.getOutputStream());
		byte[]	buffer = new byte[8192];
		for (;;)
		{
			int	count = in.read(buffer);
			if (count < 0)	// EOF
				break;
			System.out.println(new Date()+" received "+count+" bytes from "+s.getInetAddress()+":"+s.getPort());
			out.write(buffer,0,count);
			out.flush();	// flush stream reply
		}
		// shutdown output when EOF received
		s.shutdownOutput();
	}
	
	static void	report(int connections)
	{
		int	i = 0;
		int	cps = 0;
		while (i < connections)
		{
			long	l = events[i];
			int	count = 1;	// i.e. [i]
			while (i+count < connections && events[i+count] == l)
				count++;
			if (cps < count)
			{
//				if (cps > 0)
//					System.out.println("i="+i+" count="+count+" cps="+cps);
				cps = count;
			}
			i += count;
		}
		System.out.println("max connections/second = "+cps);
	}
	
	public static void	main(String[] args) throws Exception
	{
		TCPServer	s = new TCPServer();
		new Thread(s).start();
		System.out.print("press <Enter> to terminate ... ");
		String	reply = new DataInputStream(System.in).readLine();
		s.server.close();
		report(s.connections);
	}
}

/**
 * Client for TCPServer, for benchmarking.
 *
 * @author Esmond Pitt
 * @version $Revision: 4 $
 */
class Client extends Thread
{
	static int	iterations	= 1000;
	static int	connections = 0;
	String	host;
	int	port;
	
	Client(String host,int port)	{ this.host = host; this.port = port;	}
	
	public void	run()
	{
		System.out.println(this+" starting "+iterations+" iterations");
		for (int i = 0; i < iterations; i++)
		{
			Socket	s = null;
			try
			{
				s = new Socket(host,port);
				System.out.println("recv buffer="+s.getReceiveBufferSize());
				System.out.println("send buffer="+s.getSendBufferSize());
				connections++;
				// ...
				try
				{
					s.setTcpNoDelay(true);
				}
				catch (SocketException e)	{}
				// send request
				OutputStream	out = new BufferedOutputStream(s.getOutputStream());
				ObjectOutputStream	oos = new ObjectOutputStream(out);
				// ...
				oos.writeUTF("request");
				// ...
				oos.flush();
				s.shutdownOutput();
				System.out.println("sent request & shutdown output");
				// receive reply
				InputStream	in = s.getInputStream();
				ObjectInputStream	oin = new ObjectInputStream(in);
				String	reply = oin.readUTF();
				System.out.println("received reply '"+reply+"'");
				// ...
			}
			catch (IOException e)
			{
				System.err.println(e);
				e.printStackTrace();
			}
			finally
			{
				try { if (s != null) s.close();	} catch (IOException e)	{}
//				System.out.print(".");
			}
		}
		System.out.println(this+" finished "+iterations+" iterations");
	}
	
	public static void	main(String[] args) throws Exception
	{
		int	threads = 100;
		if (args.length > 0)
			threads = Integer.parseInt(args[0]);
		if (args.length > 1)
			iterations = Integer.parseInt(args[1]);
		String	host = args.length > 2 ? args[2] : "localhost";
		int	port = args.length > 3 ? Integer.parseInt(args[3]) : TCPServer.Port;
		Client[]	clients = new Client[threads];
		long	start = System.currentTimeMillis();
		for (int i = 0; i < threads; i++)
		{
			clients[i] = new Client(host,port);
			clients[i].start();
		}
		for (int i = 0; i < threads; i++)
			clients[i].join();
		long	end = System.currentTimeMillis();
		int	expected = threads*iterations;
		System.out.println("Finished in "+(end-start)+" ms; made "+connections+"/"+expected+" connections");
	}
}
