proc(values,varprop,scores,harmcoef)=FDApca(coef,period,lambda,npc)
; --------------------------------------------------------
;   Library      smoother
; --------------------------------------------------------
;   See_also     Fouriertrans Fouriereval pca fft
; --------------------------------------------------------
;   Macro        FDApca
; --------------------------------------------------------
;   Description  Carries out a penalized functional 
;                principal component analysis (PCA)
;                based on the coefficient matrix for functional data.
;                It is possible to choose a smoothing parameter objectively.
; --------------------------------------------------------
;   Usage        {values,varprop,scores,harmcoef}=FDApca(coef,period,lambda{,npc})  
;   Input
;     Parameter    coef
;     Definition   KxN matrix, the coefficient matrix for functional data
;     Parameter    period
;     Definition   scalar, the period based on the grid points of original data
;     Parameter    lambda
;     Definition   scalar, a smoothing parameter
;     Parameter    npc
;     Definition   scalar, the number of principal components to be kept (Default is 4)
;   Output
;     Parameter    values
;     Definition   Px1 vector, eigenvalues
;     Parameter    varprop
;     Definition   Px1 vector, the proportion of each variance explained by eigenfunctions 
;     Parameter    scores
;     Definition   NxP matrix, PC scores 
;     Parameter    harmcoef
;     Definition   KxP matrix, the coefficient matrix for the eigenfunctions 
; --------------------------------------------------------
;   Notes    
; --------------------------------------------------------
;   Example  library("math")
;            library("smoother")
;            temp   = read("dailtemp")            ; read daily temperature data
;            nbasis = 30                          ; set the number of basis functions
;            tempcoef =  Fouriertrans(temp,nbasis); calculation of coefficient matrix
;            period = 365                         ; period
;            lambda = 10000                       ; smoothing parameter
;            npc = 4                              ; number of principal components
;            tempfpcaresult = FDApca(tempcoef,period,lambda,npc); functional PCA
;            tempycoef = tempfpcaresult.harmcoef  ; coefficient for eigenfunctions
;            K   = rows(tempycoef)                ; number of basis functions in algorithm
;            nresol = 100                         ; number of grid points for evaluation
;            phi = Fouriereval(K,nresol,period)   ; evaluation of basis functions
;            tempxifdval = (tempycoef' * phi)'    ; evaluation of eigenfunctions
;            x  = ((1:nresol)-0.5)/(nresol/period); generates grid
;            z1 = setmask(x~tempxifdval[,1], "line", "black"); black line for PC1 weight
;            z2 = setmask(x~tempxifdval[,2], "line", "blue") ; blue  line for PC2 weight
;            z3 = setmask(x~tempxifdval[,3], "line", "red")  ; red   line for PC3 weight
;            z4 = setmask(x~tempxifdval[,4], "line", "green"); green line for PC4 weight
;            plot(z1, z2, z3, z4)                 ; plots lines
;            setgopt(plotdisplay,1,1,"title","PC Weight Functions (lambda=10000)")
; --------------------------------------------------------
;   Result   PC1 weight function is displayed with black colour,
;            PC2 weight function is displayed with blue colour,
;            PC3 weight function is displayed with red colour,
;            PC4 weight function is displayed with green colour.
; --------------------------------------------------------
;   Keywords     FDA PCA smoothing
; --------------------------------------------------------
;   Reference    Ramsay, J. O. and Silverman, B.W.(1997). Functional Data Analysis, Springer.      
; --------------------------------------------------------
;   Link
; --------------------------------------------------------
;   Author       Yoshihiro Yamanishi, 010927
;--------------------------------------------------------- 

  error(period<0|lambda<0,"FDApca: Negative argument!") 
  N = cols(coef)        ; number of observations
  K = rows(coef)        ; number of basis functions

  if (exist("npc").<>1)
    npc = 4             ; Dafault for the number of PCs is 4
  endif

  Vmat = cov(coef')     ; covariance matrix of coefficients
  Jmat = diag(matrix(K)); identiy matrix for penalty matrix

  if (lambda == 0)
    JVJ = Jmat * Vmat * Jmat
    JVJ = (JVJ+JVJ')/2                ; symmetrizing
    result = eiggn(JVJ)               ; eigenvalue problem
    values = result.values.re         ; eigenvalues
    vectors = orthonormal(result.vectors.re); normalized eigenvectors
    harmcoef = vectors[,1:npc]        ; coefficient of eigenfunctions
  else
    omega = 2 * pi / period ; parameter that determines the period 2pi/omega
    Kmat = diag(matrix(K))  ; identiy matrix for penalty matrix
    j = 1
    nj = floor(K/2)         ; number of sines/cosines
    js = 1
    jc = 2
    Kmat[1,1] = 0           ; the derivative of the constant term is 0
    do
      kval = (j * omega)^4
      Kmat[js+1,js+1] = kval 
      Kmat[jc+1,jc+1] = kval
      js = js + 2
      jc = jc + 2
      j = j + 1
    until(j > nj)
    LLmat= Jmat + lambda * Kmat
    Lmat = chold(LLmat,K)               ; Cholesky decomposition
    Lmatinv = inv(Lmat)
    LJVJL = Lmatinv * Jmat * Vmat * Jmat * Lmatinv
    LJVJL = (LJVJL+LJVJL')/2            ; symmetrizing
    result = eiggn(LJVJL)               ; eigenvalue problem
    values = result.values.re           ; eigenvalues
    vectors = orthonormal(result.vectors.re); normalized eigenvectors
    harmcoef = Lmatinv * vectors[,1:npc]; coefficient of eigenfunctions
  endif
    
  varprop  = values/sum(values)      ; proportion of each variance
  values   = values[1:npc]           ; selected eigenvalues
  varprop  = varprop[1:npc]          ; selected proportion
  scores   = coef' * Jmat * harmcoef ; PC scores
endp

