proc(phi)=Fouriereval(nbasis,nresol,period)
; --------------------------------------------------------
;   Library      math
; --------------------------------------------------------
;   See_also     Fouriertrans FDApca 
; --------------------------------------------------------
;   Macro        Fouriereval
; --------------------------------------------------------
;   Description  Evaluates the basis functions of fourier series
;                based on the period and the grid points
;                where the functions are evaluated.                                    
; --------------------------------------------------------
;   Usage        phi = Fouriereval(nbasis,nresol,period)  
;   Input
;     Parameter    nbasis
;     Definition   scalar, the number of basis functions
;     Parameter    nresol
;     Definition   scalar, the number of points where the basis functions are evaluated
;     Parameter    period
;     Definition   scalar, the period based on the grid points for original data
;   Output
;     Parameter    phi
;     Definition   KxG matrix, the evaluated basis functions, where K is the number of basis functions and G is the number of grid points for evaluation 
; --------------------------------------------------------
;   Notes     
; --------------------------------------------------------
;   Example  library("math")
;            K=31           ; number of basis functions in algorithm
;            nresol=100     ; number of grid points
;            period=365     ; period
;            Fouriereval(K,nresol,period) 
; --------------------------------------------------------
;   Result   produces basis functions 
;            evaluated by using Fourier series
; --------------------------------------------------------
;   Keywords     Fourier series
; --------------------------------------------------------
;   Reference    Ramsay, J. O.and Silverman, B.W.(1997). Functional Data Analysis, Springer.      
; --------------------------------------------------------
;   Link
; --------------------------------------------------------
;   Author       Yoshihiro Yamanishi, 010927
;--------------------------------------------------------- 

  error(nbasis<0|nresol<0|period<0,"Fouriereval: Negative argument!")
   
  phi = matrix(nbasis,nresol)           ; for basis functions
  argval = 2*pi*((1:nresol)-0.5)/nresol ; a grid for evaluation
  phi[1,] = sqrt(2)/2                   ; constant term
  nk = floor(nbasis/2)                  ; number of sines/cosines

  argum = (1:nk) .* argval' 
  s = sin(argum)                        ; evaluation of sines
  c = cos(argum)                        ; evaluation of cosines
  phi[2:nbasis,] = trans(reshape((s')|(c'), #(nresol,2*nk))) 
  
  phi = phi / sqrt(period/2)  
endp

