proc(coef)=Fouriertrans(tmat,nbasis)
; --------------------------------------------------------
;   Library      math
; --------------------------------------------------------
;   See_also     FDApca Fouriereval
; --------------------------------------------------------
;   Macro        Fouriertrans
; --------------------------------------------------------
;   Description  Calculates the coefficient in applying
;                a basis expansion by using Fourier series.
;                In this case, it is assumed that 
;                the data unit can be expressed 
;                by a linear combination of finite terms 
;                of sine and cosine functions. 
; --------------------------------------------------------
;   Usage        coef = Fouriertrans(tmat ,nbasis)  
;   Input
;     Parameter    tmat
;     Definition   TxN matrix, the original data matrix
;     Parameter    nbasis
;     Definition   scalar, the number of basis functions on which the expansion is computed
;   Output
;     Parameter    coef
;     Definition   KxN matrix, the coefficient in applying a Fourier basis expansion
; --------------------------------------------------------
;   Notes     
; --------------------------------------------------------
;   Example  library("math")
;            library("smoother")
;   temp=read("dailtemp.dat")            ; read daily temperature data
;   nbasis = 30                          ; set the number of basis functions
;   tempcoef =  Fouriertrans(temp,nbasis); calculation of coefficient matrix
;   K = rows(tempcoef)                   ; number of basis functions in algorithm
;   nresol = 100                         ; number of grid points
;   period = 365                         ; period
;   phi = Fouriereval(K,nresol,period)   ; evaluation of basis functions
;   tempfdval = tempcoef' * phi          ; evaluation of data functions
;   bp = grpcp (tempfdval,0)             ; parallel cordinates plot
;   dispfd  = createdisplay(1,1)         ; create display
;   show (dispfd, 1, 1, bp)              ; plots lines
;   setgopt(dispfd,1,1,"title","Temperature Functions")
; --------------------------------------------------------
;   Result   produces a plot of functional data 
; --------------------------------------------------------
;   Keywords     FDA Fourier coefficient
; --------------------------------------------------------
;   Reference    Ramsay, J. O.and Silverman, B.W.(1997). Functional Data Analysis, Springer.      
; --------------------------------------------------------
;   Link
; --------------------------------------------------------
;   Author       Yoshihiro Yamanishi, 010927
;--------------------------------------------------------- 

  N = cols(tmat)        ; number of observations
  ngrid = rows(tmat)    ; number of grid points

  error(nbasis<0,"Fouriertrans: Negative argument!") 
  error(nbasis==0|nbasis==1,"Fouriertrans: The number of basis functions is too small!")
  error(nbasis>ngrid,"Fouriertrans: The number of basis functions is too large!")
  
  nbasis = nbasis + nbasis%2 + 1 ; make the number of sines and cosines same and add a constant
       
  nresol = ngrid        ; number of grid points for evaluation
  period = ngrid        ; period
  phimat = Fouriereval(nbasis, nresol, period) ; evaluates Fourier basis funcions
  coef = phimat * tmat                         ; computes the coefficient
endp

