; --------------------------------------------------------------------
; --------------------------------------------------------------------
; FUNCTIONAL KERNEL ESTIMATOR - DISCRIMINATION
; --------------------------------------------------------------------
; --------------------------------------------------------------------
; INPUT
;========
; Response = vector of observed groups (1, ..., G)
;
; curves = matrix of curves (stored row by row)
;          for which the group
;          is observed (learning sample)
;
; pred = matrix of new curves (stored row by row)
;        for which the group
;        has to be predicted
;
; param =   dimension for the reduced dimensional space [if we use semi = "pca"]
;         or
;            list(q, nknot, rangegrid)  [if we use semi = "der"]
;            with q = order of derivation, nknot = number of interior knots, 
;            rangegrid = #(t1,tp) [a vector with the limits of the grid]
;
; semi =     "der" -> semi-metric deriv; 
;        or
;           "pca" -> semi-metric Functional PCA;
;
;  
; OUTPUT
;========
; EstimatedProb = matrix containing the posterior probabilities:
;                 the ith row gives the G posterior probabilities
;                 for the ith  curve  in the learning sample ("curves")
;
; PredictedProb = matrix containing the posterior probabilities:
;                 the ith row gives the G posterior probabilities
;                 for the ith  curve  in the  sample of new curves ("pred")
;
; EstimatedGroups = vector containing the estimated categorical
;                    responses for  the learning sample ("curves")
;
; PredictedGroups = vector containing the predicted categorical
;                    responses for the new curves ("pred")
;
; MAE = misclassification rate for the curves in the learning
;       sample ("curves")
;
; --------------------------------------------------------------------
; --------------------------------------------------------------------

proc(EstimatedProb, PredictedProb, EstimatedGroups, PredictedGroups, MAE) = FuncKerDiscrim(Response, curves, pred, param, semi)

	G = max(Response)
	n1 = rows(curves)
	n2 = rows(pred)

	DummyResp = zeros(n1, G)
	EstimatedProb = zeros(n1, G)
	PredictedProb = zeros(n2, G)

	j = 1
	while(j <= G)
		DummyResp[,j] = (Response == j) 
		ris = FuncKerDiscKnnCV(DummyResp[,j], curves, pred, param, semi)
		EstimatedProb[,j] = ris.EstimatedGroups
		PredictedProb[,j] = ris.PredictedGroups
		j = j + 1
	endo
	
	EstimatedGroups = zeros(n1)
	i = 1
	while(i <= n1)
		EstimatedGroups[i] = order(-EstimatedProb[i,]')[1]
		i = i + 1
	endo

	MAE = mean(Response!=EstimatedGroups)

	PredictedGroups = zeros(n2)
	i = 1
	while(i <= n2)
		PredictedGroups[i] = order(-PredictedProb[i,]')[1]
		i = i + 1
	endo

endp

; --------------------------------------------------------------------
; --------------------------------------------------------------------

proc(EstimatedGroups, PredictedGroups, Band, MAE ) = FuncKerDiscKnnCV(Response, curves, pred, param, semi)

	error( ((semi=="pca")||(semi=="der")) == 0, "Define semi-metric 'pca' or 'der'" )
	
	if (semi=="pca")
		SEMIMETRIC1 = SemiMetricPCA(curves, param, curves)
	endif
		
	if (semi=="der")
		q = param{1}
		nknot = param{2}
		rangegrid = param{3}
 		SEMIMETRIC1 = SemiMetricDer(curves, q, nknot, rangegrid, curves)	
	endif	
	
	n1 = cols(SEMIMETRIC1)
	step = ceil (n1/100)
  	if(step == 0)
  		step = 1
	endif
	Knearest = aseq(2, (floor( n1 / 2 ) - 1)/step , step)
	kmax = max(Knearest)

	; the vector Knearest contains the sequence of the 
	; k-nearest neighbours used for computing the optimal bandwidth
        
	ResponseEstimated = zeros(n1)
	BandwidthOpt  = zeros(n1)

	i = 1
	while(i <= n1)
		NormDiff = SEMIMETRIC1[,i]
  		; "norm.order" gives the sequence k_1, k_2,... such that 
   		; dq(X_{k_1},X_i) < dq(X_{k_2},X_i) < ...
		NormOrder = order( NormDiff )
		; "zz" contains dq(X_{k_2},X_i), dq(X_{k_3},X_i),...,
		; dq(X_{j_{kamx+2}},X_i)
		zz = sort(NormDiff)[2:(kmax + 2)]
		; Bandwith[l-1] contains (dq(X_{j_l},X_i) + 
		; dq(X_{j_l},X_i))/2 for l=2,...,kmax+2
		Bandwidth = (zz[2:(kmax+1)] + zz[ 1:kmax ]) / 2
		z = zz[1:kmax ]
		ZMAT = matrix(kmax) .* z'
		UMAT = ZMAT / Bandwidth
		KMAT = quad(UMAT)
		
		Ind1 = 1:kmax
		Ind2 = kmax / (1:kmax)
		KMAT  = KMAT .* ( Ind2 * Ind1' <= kmax )
		IndCurves = NormOrder[2:(kmax + 1)]
		IndResp = Response[IndCurves]
		YMAT = matrix(kmax) .* IndResp'
		HatResp = sum( (YMAT[Knearest,  ] .* KMAT[Knearest,  ])' ) / sum( (KMAT[Knearest,  ])' )
		Criterium = abs(HatResp - Response[i])
		Index = order(Criterium')[1]
		ResponseEstimated[i] = HatResp[,Index]
		BandwidthOpt[i]  =  Bandwidth[Index]      
		i = i + 1
	endo

	MaeEstimated = sum( abs(rint(ResponseEstimated) - Response) ) / n1

	if (semi=="pca")
		SEMIMETRIC2 = SemiMetricPCA(curves, param, pred)
	endif	
	
	if (semi=="der")
 		SEMIMETRIC2 = SemiMetricDer(curves, q, nknot, rangegrid, pred)	
	endif	

	n2 = cols(SEMIMETRIC2)
	Bandwidth2 = zeros(n2)

	k = 0
	while(k < n2)
		k = k+1
		Bandwidth2[k] = BandwidthOpt[order(SEMIMETRIC2[,k])[1] ]
	endo
	KERNEL = quad( SEMIMETRIC2 / Bandwidth2' )
	flag2 = 1 - ((KERNEL < 0) || (KERNEL >= 1))
	KERNEL = KERNEL .* flag2
	Denom = sum( KERNEL )
	Logic = (Denom' == 0)

	while(sum(Logic)>=1) 
		CorLogit = Logic * 0.1 + 1
  	Bandwidth2 = CorLogit .* Bandwidth2
		KERNEL = quad( SEMIMETRIC2 / Bandwidth2' )
		flag = 1 - ((KERNEL < 0) || (KERNEL >= 1))
		KERNEL = KERNEL .* flag
		Denom = sum( KERNEL )
		Logic = (Denom' == 0)
	endo

	RESPKERNEL = KERNEL .* Response
	ResponsePredicted = sum(RESPKERNEL) / Denom
	
	EstimatedGroups = ResponseEstimated
	PredictedGroups = ResponsePredicted
	Band = BandwidthOpt
	MAE = MaeEstimated

endp

; --------------------------------------------------------------------
; --------------------------------------------------------------------
 
proc(kernel) = quad(u)
	kernel = 1-u^2
endp

; --------------------------------------------------------------------
; --------------------------------------------------------------------

