proc(Ttree, P, Q, AD, LocVol, Onodes, Oprobs, Time) = ITT(S,r,div,time,volafunc,skewconst)
; ---------------------------------------------------------------------
; Library     finance
; ---------------------------------------------------------------------
; See_also    grITTcrr, grITTstsp, grITTspd, ITTad, ITTcrr, ITTnewnodes, ITTterm, ITTtermder, plotITT, IBTdk, bitree, nmnewton
; ---------------------------------------------------------------------
; Macro       ITT
; ---------------------------------------------------------------------
; Description the main function for the Derman/Kani/Chriss method of 
;             implied trinomial trees (ITT). It computes the nodes of the ITT, 
;             the probability matrices, Arrow-Debreu prices and the local volatility matrix.
; ---------------------------------------------------------------------
; Usage       {Ttree,P,Q,AD,LocVol,Onodes,Oprobs,Time}=ITT(S,r,div,time,volafunc{,skewconst}) or <br>
;             {Ttree,P,Q,AD,LocVol,Onodes,Oprobs,Time}=ITT(volafunc)
; Input       
; Parameter   S
; Definition  scalar; spot price of the underlying
; Parameter   r
; Definition  scalar; interest rate, from interval (0,1)
; Parameter   div
; Definition  scalar; the dividend rate, from interval (0,1)
; Parameter   time
; Definition  vector; time points (in years) when the levels of the tree occured 
;             (they are supposed to increase)
; Parameter   volafunc
; Definition  string; if there is only slow variation of implied volatilities, volafunc
;             has only one row containing the name of the volatility function. 
;             This function has three input parameters: (S, K, tau)
;             where K denotes the exercise price and tau is the time to maturity
;             on each level. <br>
;             If the volatility varies significantly with strike or time to expiration, 
;             volafunc can have up to 3 rows: <br>
;             1st row contains the name of the volatility function for the term 
;             structure: term(t). 2nd row contains the name of the first derivative of this
;             volatility funciton: term'(t). Both of these two functions have only one 
;             parameter time and it is necessary that they can use vectors.
;             3rd row is the name of the function for the skew structure: skew(t). 
;             If there is no term structure, but significant skew structure, 
;             insert volafunc[1]="".
; Parameter   skewconst
; Definition  vector; the skew constant c (see the reference) - default is 0.1
; Output      
; Parameter   Ttree
; Definition  matrix; the Implied Trinomial Tree (root is in the left upper corner
;             elements which do not belong to the tree are NaN's)
; Parameter   P
; Definition  matrix; the up transition probabilities
; Parameter   Q
; Definition  matrix; the down transition probabilities
; Parameter   AD
; Definition  matrix; the Arrow-Debreu prices for the computed tree
; Parameter   LocVol
; Definition  matrix; the local volatilities
; Parameter   Onodes
; Definition  matrix; the overwritten nodes of the Implied Trinomial Tree,
;             not overridden elements are NaN's. If there are no overwritten 
;             nodes Onodes is NaN (only scalar).
; Parameter   Oprobs
; Definition  ((1+4*k) x 2) matrix; coordinates of the nodes where overwritten probabilities
;             occured. First row is only auxiliary, but after that each four rows correspond
;             to one overwritten probability. First element of this quartet is the parental node,
;             the three other elements are daughter nodes. If there are no overwritten 
;             probabilities Oprobs is NaN (only scalar).
; Parameter   Time
; Definition  vector; usefull only when there is a significant term structure - only then 
;             it differs from the input parameter time.
; ---------------------------------------------------------------------
; Notes       (1) Option prices are computed from the ideal (not overwritten) tree <br>
;             (2) Interactive menu will be invoked if "volafunc" is the only parameter 
; ---------------------------------------------------------------------
; Example 
;    library("finance")
;
;    proc(sigma)=volafunc(S,K,time)	
;        sigma=0.15 + (K-S)/10 * 0.005 
;    endp
;
;    S = 100	        ; current index level
;    r = 0.1		; interest rate
;    div = 0.05         ; dividend yield
;    time = 0|0.5|1	; 2 half-year periods
;
;    t=ITT(S, r, div, time, "volafunc")
;    t.Ttree
; ---------------------------------------------------------------------
; Result      
;	Contents of Ttree
;
;	[1,]      100   116.18   134.99 
;	[2,]     +NAN      100   116.18 
;	[3,]     +NAN   86.071      100 
;	[4,]     +NAN     +NAN   86.071 
;	[5,]     +NAN     +NAN   74.082 
; ---------------------------------------------------------------------
; Example 
;    library("finance")
;
;    proc(sigma)=volafunc(S,K,time)	
;        sigma=0.15 + (K-S)/10 * 0.005 
;    endp
;
;    t=ITT("volafunc")
; ---------------------------------------------------------------------
; Result
;	several interactive menus are invoked to let the user  
;	type the input values
; ---------------------------------------------------------------------
; Example 
;    library("finance")
;    library("nummath")
;; now define the term volatility function and its first derivative
;    proc(fx)=t1(t)	
;        T=3	
;        q=0.3
;        k=-q/(T+2)
;        fx = sqrt(k*t+q)
;    endp
;    proc(fder)=t1der(t)	
;        T=3
;        q=0.3
;        k=-q/(T+2)
;        fder = k / 2 / sqrt(k*t+q)
;    endp
;; define the skew volatility function
;    proc(sigma)=skew(S)
;        K=100	; exercise price
;        sigma = (S/(3*K+S))^2
;    endp
;; and compute     
;    S = 100	        ; current index level
;    r = 0.1		; compounded riskless interest rate
;    div = 0.05         ; dividend yield
;    time = 0|1|2|3	; time vector
;    volaf = "t1"|"t1der"|"skew"
;
;    t=ITT(S, r, div, time, volaf)
;    plotITT(t,1)
;    t.LocVol
; ---------------------------------------------------------------------
; Result      
;    --------------------------------
;                 Warning!           
;    ================================
;     probabilities of 6 node(s)
;     had to be overwritten in order 
;     to avoid the arbitrage         
;    --------------------------------
;
;    Contents of LocVol
;    [1,]  0.044084  0.043385  0.038446 
;    [2,]     +NAN  0.031908  0.02952 
;    [3,]     +NAN  0.032168  0.027637 
;    [4,]     +NAN     +NAN  0.031259 
;    [5,]     +NAN     +NAN  0.042086 
; ---------------------------------------------------------------------
; Keywords    implied volatilities, Black Scholes model, option pricing
; ---------------------------------------------------------------------
; Reference   E. Derman, I. Kani and N. Chriss (1996): 
;             Implied Trinomial Trees of the Volatility Smile <br>
;             K. Komorad (2002): Implied Trinomial Trees and Their Implementation with XploRe
; ---------------------------------------------------------------------
; Link        www.gs.com/qs
; ---------------------------------------------------------------------
; Author      K. Komorad 20020319
; ---------------------------------------------------------------------
;
;=============== interactive input
    if(exist("r")!=1)   ; interactive
        iact = 1
        volafunc = S
        
        st = "Price of the underlying asset"|"Annual interest rate"|"Dividend rate"
        va = readvalue(st,100|0.1|0.05)
        S     = va[1]
        r     = va[2]
        div   = va[3]
        
        t=readvalue("The starting time point",0)
        error(t!=0,"you must allways start with 0")
        time = t
        t=1
        while(t>0)
            t=readvalue("next time point (<=0  for end)",t)
            time = time | t
        endo  
        error(rows(time)<3,"you must insert at least 2 valid time points")
        time=time[1:rows(time)-1]	; the last value is negative
    else
        iact = 0
    endif	; end interactive
    
;=========== initial phase
    dims=dim(S)|dim(r)|dim(div)
    error(sum(dims)!=3,"S, r and div must be scalars")
    error(dim(dim(time))!=1,"time must be a vector")
    error(S<=0,"S must be positive")
    error((r<0)||(r>1),"r must be from (0,1)")
    error((div<0)||(div>1),"div must be from (0,1)")
    error(rows(time)<2,"time must have at least two rows")
    error(time[1]!=0,"time[1] must be 0")
    steps=rows(time)-1          ; # of time steps
    error(sum(sort(time)!=time),"time must be increasing")
    tmp=time[2:rows(time)]-time[1:rows(time)-1]
    error(sum(tmp<=0),"time must be increasing")

; optional parameter
    if(exist("skewconst")!=1)
        skewconst=0.1*matrix(steps) 
    else
        error(rows(skewconst)!=steps,"skewconst has wrong dimension")
    endif

; allocate the output arrays
    Ttree = NaN * matrix(2*steps+1,steps+1)	
    Onodes = Ttree
    AD = Ttree
    LocVol = NaN * matrix(2*steps-1,steps)
    P = LocVol
    Q = LocVol
; initializations    
    tau = time[steps+1]-time[1] ; time to maturity
    deltat = time[2]-time[1]    ; time difference on the first level
    tte = tau                   ; time to expiration on the first level
    r = log(1+r)                ; continuous interest rate
    div = log(1+div)            ; continuous dividend yield
    AD[1,1] = 1
    Ttree[1,1] = S
    ITTprobs = 0            ; # of overwritten probabilities
    ITTnodes = 0            ; # of overwritten nodes
    Oprobs = 0~0

;===================== significant volatility variation
    skew = 0
    term = 0
    constsigma = 1
    if(rows(volafunc)==1)                    ;==== no significant structure
        nostructure=volafunc
        constsigma=_nostructure(S,S,tte)	
        error((constsigma<=0)||(constsigma>=1),"volatility function is not from (0,1)")
    else
        if(volafunc[1]!="")                  ;==== there is term structure
            term=1
            if(steps>1) ; if(steps==1) then you can't change the time
                ITTtermparameter  = time[rows(time)]
                ITTtermstructure  = volafunc[1]
                ITTtermderivative = volafunc[2]
                putglobal("ITTtermparameter")
                putglobal("ITTtermstructure")
                putglobal("ITTtermderivative")
                z=nmnewton("ITTterm","ITTtermder",0*matrix(steps-1))
                time=0|z.x|ITTtermparameter
                error(sum(sort(time)!=time),"time must be increasing")
                tmp=time[2:rows(time)]-time[1:rows(time)-1]
                error(sum(tmp<=0),"time must be increasing")            
            endif
            constsigma=_ITTtermstructure(time[2])
            error((constsigma<=0)||(constsigma>=1),"volatility function is not from (0,1)")
        endif   ; term
        
        if(rows(volafunc)>2)	             ;==== there is skewed structure
            skew=1
            skewstructure=volafunc[3]
            constsigma=_skewstructure(S) * constsigma
            error((constsigma<=0)||(constsigma>=1),"volatility function is not from (0,1)")
            ; build up the skewed state space  
            crr=ITTcrr(S, S, 0, constsigma, time, 0, 0)
            Ttree=crr.ttree

            j1 = 2    ; modify the mesh - go from the left to the right
            while(j1 <= steps+1) 
                newn = Ttree[1:2*j1-1,j1]   ; choose the actual level
                np1=(rows(newn)+1)/2        ; middle index
                copy=newn	            ; copy of original nodes from the actual level
                k=1
                while(k<np1) ; go in the level from the central node
                  ; upper spot prices
                    lgu = log(copy[np1-k]/copy[np1-k+1])
                    sgu = _skewstructure(copy[np1-k])
                    error((sgu<=0)||(sgu>=1),"volatility function is not from (0,1)")
                    newn[np1-k]=newn[np1-k+1]*exp(skewconst[j1-1]/sgu*lgu)
                  ; lower spot prices
                    lgd = log(copy[np1+k-1]/copy[np1+k])
                    sgd = _skewstructure(copy[np1+k-1])
                    error((sgd<=0)||(sgd>=1),"volatility function is not from (0,1)")
                    newn[np1+k]=newn[np1+k-1]/exp(skewconst[j1-1]/sgd*lgd)

                    k=k+1    
                endo
                Ttree[1:2*j1-1,j1]=newn
                j1=j1+1    
            endo
        else
            ; the state space hasn't been built yet
            crr=ITTcrr(S, S, 0, constsigma, time, 0, 0)
            Ttree=crr.ttree            
        endif   ; skew
        deltas=time[2:steps+1]-time[1:steps]
        Ttree[,2:steps+1]=Ttree[,2:steps+1].*exp((r-div)*deltas)'       ; grow the prices along the forward curve
    endif    ;  if(rows(volafunc)==1) 	- end of structure in volatility    
    
;============ by the root
    if((!term) && (!skew))
        newn = ITTnewnodes(S,r,constsigma,deltat,div)
        Ttree[1:3,2] = newn.newnodes
        Onodes[1:3,2] = newn.overwritten
    endif
    crr = ITTcrr(S, S, r, constsigma, time[1:2], 0, div)
    put = crr.optprice    
    F = S*exp((r-div)*deltat)                             ; Forward price
    Q[1,1] = exp(r*deltat)*put/(S-Ttree[3,2])             ; first lower prob.
    P[1,1] = (F+Q[1,1]*(S-Ttree[3,2])-S)/(Ttree[1,2]-S)   ; first upper prob.
    ; stop arbitrage	- only the middle probability can be zero
    if((P[1,1]<=0) || (P[1,1]>=1) || (Q[1,1]<=0) || (Q[1,1]>=1) || (P[1,1]+Q[1,1]>1))
        ITTprobs = ITTprobs + 1
        Oprobs = Oprobs | ( (time[1]|time[2]*matrix(3))~(S|Ttree[1:3,2]) )
        if(F>=Ttree[2,2])
            Q[1,1] = (Ttree[1,2]-F)/(Ttree[1,2]-Ttree[3,2])/2
            P[1,1] = (F-Ttree[2,2])/(Ttree[1,2]-Ttree[2,2])
            P[1,1] = (P[1,1] + (F-Ttree[3,2])/(Ttree[1,2]-Ttree[3,2]) ) / 2
        else
            P[1,1] = (F-Ttree[3,2])/(Ttree[1,2]-Ttree[3,2])/2
            Q[1,1] = (Ttree[2,2]-F)/(Ttree[2,2]-Ttree[3,2])
            Q[1,1] = (Q[1,1] + (Ttree[1,2]-F)/(Ttree[1,2]-Ttree[3,2]) ) / 2       
        endif
    endif
    LocVol[1,1] = P[1,1]*(Ttree[1,2]-F)^2+(1-P[1,1]-Q[1,1])*(S-F)^2+Q[1,1]*(Ttree[3,2]-F)^2     
    LocVol[1,1] = sqrt(LocVol[1,1]/F^2/deltat)            ; local volatility

;============= generalize
; go through the tree from the left to the right and from the upside downwards   
    j1 = 2
    while(j1 <= steps) 
        lastn = 2*j1+1       ; index of the last element of the tree on the new level
        lasta = 2*j1-1       ; index of the last element of the tree on actual level
        lastp = 2*j1-3       ; index of the last element of the tree on previous level
        deltatprev = deltat            ; previous deltat
        deltat = time[j1+1]-time[j1]
        tte = tte - deltatprev         ; time to expiration on the new level
        if((!term) && (!skew))
            newn = ITTnewnodes(Ttree[1:lasta,j1],r,constsigma,deltat,div)
            Ttree[1:lastn,j1+1] = newn.newnodes
            Onodes[1:lastn,j1+1] = newn.overwritten
        endif
        F = Ttree[1:lasta,j1]*exp((r-div)*deltat)   ; vector of Forward prices
        lambda = ITTad(P[1:lastp,j1-1], Q[1:lastp,j1-1], AD[1:lastp,j1-1], r, deltatprev)        
        AD[1:lasta,j1] = lambda
        j2 = 1  
        while(j2 <= lasta)       ; go from the upper side downwards
          ; determine the volatility
            switch 
              case((!term) && (!skew))
                  sigma=_nostructure(Ttree[j2,j1],S,tte)
                  break
              case(term && (!skew))
                  sigma=_ITTtermstructure(time[j1])
                  break
              case((!term) && skew)
                  sigma=_skewstructure(Ttree[j2,j1])
                  break
              case(term && skew)
                  a = _ITTtermstructure(time[j1])
                  b = _skewstructure(Ttree[j2,j1])
                  sigma=a*b
            endsw          
            error((sigma<=0)||(sigma>=1),"volatility function is not from (0,1)")            
            ; j1 is also the index of the middle element on actual level
            if(j2<j1)  ; the upper triangle
              ; p_i        
                if(j2==1)  ; no higher knots
                    Sm = 0                                ; Sm is the sum from the paper
                else
                    Sm = trans(lambda[1:j2-1])*(F[1:j2-1]-Ttree[j2+1,j1+1]) 
                endif
                crr = ITTcrr(S,Ttree[j2+1,j1+1],r,sigma,time[1:j1+1],1,div)
                call = crr.optprice
                num = exp(r*deltat)*call-Sm 
                den = lambda[j2]*(Ttree[j2,j1+1]-Ttree[j2+1,j1+1])
                P[j2,j1] = num/den
              ; q_i
                num = F[j2] - P[j2,j1]*(Ttree[j2,j1+1]-Ttree[j2+1,j1+1])-Ttree[j2+1,j1+1]
                Q[j2,j1]=num/(Ttree[j2+2,j1+1]-Ttree[j2+1,j1+1])
            else  ; the lower triangle
              ; q_i            
                if(j2==lasta)  ; no lower knots
                    Sm = 0                                ; Sm is the sum from the paper
                else
                    Sm = trans(lambda[j2+1:lasta])*(Ttree[j2+1,j1+1]-F[j2+1:lasta])
                endif
                crr = ITTcrr(S,Ttree[j2+1,j1+1],r,sigma,time[1:j1+1],0,div)
                put = crr.optprice
                num = exp(r*deltat)*put - Sm
                den = lambda[j2]*(Ttree[j2+1,j1+1]-Ttree[j2+2,j1+1])
                Q[j2,j1]=num/den
              ; p_i                
                num = F[j2] + Q[j2,j1]*(Ttree[j2+1,j1+1]-Ttree[j2+2,j1+1])-Ttree[j2+1,j1+1]
                P[j2,j1]=num/(Ttree[j2,j1+1]-Ttree[j2+1,j1+1])
            endif 
            ; stop the arbitrage (only the middle probability can be zero)
            if((P[j2,j1]<=0) || (P[j2,j1]>=1) || (Q[j2,j1]<=0) || (Q[j2,j1]>=1) || (P[j2,j1]+Q[j2,j1]>1))
                ITTprobs = ITTprobs + 1
                newoverwr = (time[j1]|time[j1+1]*matrix(3))~(Ttree[j2,j1]|Ttree[j2:j2+2,j1+1]) 
                Oprobs = Oprobs | newoverwr  ; coordinates of the changed nodes (for the graph)
                if(F[j2]>=Ttree[j2+1,j1+1])
                    Q[j2,j1] = (Ttree[j2,j1+1]-F[j2])/(Ttree[j2,j1+1]-Ttree[j2+2,j1+1])/2
                    P[j2,j1] = (F[j2]-Ttree[j2+1,j1+1])/(Ttree[j2,j1+1]-Ttree[j2+1,j1+1])
                    P[j2,j1] = (P[j2,j1] + (F[j2]-Ttree[j2+2,j1+1])/(Ttree[j2,j1+1]-Ttree[j2+2,j1+1]) ) / 2
                else
                    P[j2,j1] = (F[j2]-Ttree[j2+2,j1+1])/(Ttree[j2,j1+1]-Ttree[j2+2,j1+1])/2
                    Q[j2,j1] = (Ttree[j2+1,j1+1]-F[j2])/(Ttree[j2+1,j1+1]-Ttree[j2+2,j1+1])
                    Q[j2,j1] = (Q[j2,j1] + (Ttree[j2,j1+1]-F[j2])/(Ttree[j2,j1+1]-Ttree[j2+2,j1+1]) ) / 2       
                endif
            endif
            ; local volatility
            num = P[j2,j1]*(Ttree[j2,j1+1]-F[j2])^2+(1-P[j2,j1]-Q[j2,j1])*(Ttree[j2+1,j1+1]-F[j2])^2+Q[j2,j1]*(Ttree[j2+2,j1+1]-F[j2])^2
            LocVol[j2,j1] = sqrt(num / (F[j2]^2 * deltat))
            j2 = j2 + 1
        endo
        j1 = j1 + 1
    endo
; the last column of Arrow-Debreu Prices    
    AD[,steps+1] = ITTad(P[,steps], Q[,steps], AD[1:2*steps-1,steps], r, deltat)
    Time = time
   
    ITTnodes=sum(sum(Onodes!=NaN),2)    
; reduce Onodes if there are only NaN's
    if(!ITTnodes)
        Onodes = NaN   ; save the memory - no need for a big matrix of NaN's
    endif
    if(!ITTprobs)
        Oprobs = NaN   
    endif
; text output 
    if(ITTnodes+ITTprobs>0)
        old1 = getenv("outputstringformat")
        old2 = getenv("outlineno")
        old3 = getenv("outheadline")
        setenv("outheadline","\n")
        setenv("outputstringformat", "%s")
        setenv("outlineno", "")
        otext=      "--------------------------------"
        otext=otext|"             Warning!           "
        otext=otext|"================================"
        if(ITTnodes>0)
            otext=otext|string(" values of %.f node(s) ",ITTnodes)
        endif
        if(ITTprobs>0)
            if(ITTnodes>0)
                row = " and"
            else
                row = ""
            endif
            row = row + string(" probabilities of %.f node(s)",ITTprobs)
            otext=otext|row
        endif    
        otext=otext|" had to be overwritten in order "
        otext=otext|" to avoid the arbitrage         "
        otext=otext|"--------------------------------"
        otext
        setenv("outputstringformat", old1)
        setenv("outlineno", old2)
        setenv("outheadline", old3)
    endif  ; text output is worth to be shown

; plot the results if interactive
    if(iact)
        out=list(Ttree, AD, P, Q, LocVol, Onodes, Oprobs, Time)
        headline = "What do you want to plot?"
        items = "State Space of the ITT"|"Probability Trees"|"Local Volatility Tree"|"Arrow-Debreu Prices"|"State Price Density"
        is = selectitem(headline, items)    
        if(is[1])
            text = selectitem("State space with described nodes?","yes"|"no","single")    
            prtext = selectitem("Arrows with probabilities?","yes"|"no","single")    
            text = sum(text==(1|0))/2
            prtext = sum(prtext==(1|0))/2
            plotITT(out,is,r,text,prtext)
        else
            plotITT(out,is,r)
        endif
    endif  ; if(iact)
endp

