proc(newad) = ITTad(up, down, ad, r, deltat)
; ---------------------------------------------------------------------
; Library     finance
; ---------------------------------------------------------------------
; See_also    ITTcrr, ITTnewnodes, ITT
; ---------------------------------------------------------------------
; Macro       ITTad
; ---------------------------------------------------------------------
; Description An auxiliary macro for ITT. It computes the Arrow-Debreu 
;             prices of an Implied Trinomial Tree on one particular level.
; ---------------------------------------------------------------------
; Usage       newad=ITTad(up,down,ad,r,deltat)
; Input       
; Parameter   up
; Definition  vector; up probabilities on the given level
; Parameter   down
; Definition  vector; down probabilities on the given level
; Parameter   ad
; Definition  vector; Arrow-Debreu prices on the last level
; Parameter   r
; Definition  scalar; CONTINUOUS interest rate from the interval (0,1)
; Parameter   deltat
; Definition  scalar; time step from the last level to the new one
; Output      
; Parameter   newad
; Definition  vector; Arrow-Debreu prices on the following level
; ---------------------------------------------------------------------
; Author      K. Komorad 20020126
; ---------------------------------------------------------------------
;
    dims = dim(up)|dim(down)|dim(ad)
    error(rows(dims)!=3,"up, down and ad must be vectors")
    steps = rows(ad)
    error(sum(steps!=dims),"up, down and ad must have the same dimension")
    error(sum(dim(r)|dim(deltat))!=2,"r and deltat must be scalars")
    error((r<0)||(r>1),"r must be from (0,1)")
;
    levels  = steps + 2
    newad = NaN*matrix(levels)
    dis = exp(-r*deltat)        ; discount rate
;
    if(steps == 1)       ; initialize for the first time step  
        newad[1] = dis * up            ; discounted p
        newad[2] = dis * (1-up-down)   ; discounted 1-p-q
        newad[3] = dis * down          ; discounted q
    else                 ; compute the Arrow-Debreu prices on higher level
        j=1
        while(j <= levels)
            switch
              case(j==1)    ; the highest up probability - only p goes into this knot
                  newad[j] = dis * up[1] * ad[1]
                  break
              case(j==2)    ; the 2nd highest up prob. - only p and (1-p-q) go into this knot
                  tmp = up[2] * ad[2] + (1-up[1]-down[1]) * ad[1]
                  newad[j] = dis * tmp
                  break
              case(j==levels) ; the lowest down probability - only q goes to this knot
                  newad[j] = dis * down[steps] * ad[steps]
                  break
              case(j==levels-1) ; the 2nd lowest prob. - only q and (1-p-q) go to this knot
                  tmp = down[steps-1] * ad[steps-1] + (1-up[steps]-down[steps]) * ad[steps]
                  newad[j] = dis * tmp
                break
              case((j>2)&&(j<levels-1)) ; anything else - there are 3 probs. going to this knot
                  tmp = up[j] * ad[j]
                  tmp = tmp + (1-up[j-1]-down[j-1]) * ad[j-1]
                  tmp = tmp + down[j-2] * ad[j-2]
                  newad[j] = dis * tmp
                  break
            endsw  
            j = j+1
        endo
    endif
endp

