proc(ttree, optprice) = ITTcrr(S, K, r, sigma, time, opt, div)
; ---------------------------------------------------------------------
; Library     finance 
; ---------------------------------------------------------------------
; See_also    ITTad, ITTnewnodes, ITT, bitree, grITTcrr
; ---------------------------------------------------------------------
; Macro       ITTcrr
; ---------------------------------------------------------------------
; Description builds up a constant-volatility trinomial tree and computes
;             option price of given option with given strike price
; ---------------------------------------------------------------------
; Usage       {ttree,optprice}=ITTcrr(S,K,r,sigma,time,opt,div)
; Input       
; Parameter   S
; Definition  scalar; the spot price of the underlying
; Parameter   K
; Definition  scalar; the exercise price
; Parameter   r
; Definition  scalar; the CONTINUOUS interest rate from the interval (0,1)
; Parameter   sigma
; Definition  scalar; constant volatility
; Parameter   time
; Definition  vector; the time points
; Parameter   opt
; Definition  scalar; 1 for a call option or 0 for a put option
; Parameter   div
; Definition  scalar; the dividend rate from the interval (0,1)  
; Output      
; Parameter   ttree
; Definition  matrix; the constant volatility trinomial tree - possible 
;             values of the stock price are in the upper "trinagular".
; Parameter   optprice
; Definition  scalar; the price of given option
; ---------------------------------------------------------------------
; Notes       the used method combines two steps of a CRR binomial tree
; ---------------------------------------------------------------------
; Example     
;    library("finance")
;    library("graphic")
; 
;    S = 100         ; current index level
;    K = 120         ; strike price
;    r = 0.1         ; compounded riskless interest rate
;    sigma = 0.2     ; constant volatility
;    time = 0|1|3|6  ; time vector
;    opt = 1         ; call option
;    div = 0.05      ; dividend yield
;  
;    t=ITTcrr(S, K, r, sigma, time, opt, div)
;    t.optprice	   ; show the option price
;  
;    tr=grITTcrr(t.ttree,time,0|0,1,2)   
;    d=createdisplay(1,1)
;    show(d,1,1,tr)
; ---------------------------------------------------------------------
; Result      
;    Contents of optprice
;    [1,]   14.992 
;    
;    A display with a trinomial tree with constant logarithmic spacing 
;    is also returned.
; ---------------------------------------------------------------------
; Keywords    Black Scholes model, option pricing
; ---------------------------------------------------------------------
; Author      K. Komorad 20020126
; ---------------------------------------------------------------------
;
    dims=dim(S)|dim(K)|dim(r)|dim(sigma)|dim(opt)|dim(div)
    error(sum(dims)!=6,"S, K, r, sigma, opt and div must be scalars")
    error(dim(dim(time))>1,"time must be vector")
    error(rows(time)==1,"time must have at least two rows")
    error((r<0)||(r>1),"r must be from (0,1)")
    error((sigma<0)||(sigma>1),"sigma must be from (0,1)")
    error((div<0)||(div>1),"div must be from (0,1)")
    error((opt!=0)&&(opt!=1),"opt must be either 0 or 1")
    steps   = rows(time) - 1
;======= the tree nodes
    nodes = 0*matrix(2*steps+1)            ; the last column
    ttree = NaN*matrix(2*steps+1,steps+1)   
    middle = steps + 1                     ; index of the middle value
    nodes[middle] = S
    ttree[1,1] = S
    i = 1
    while(i <= steps)
        deltat = time[i+1] - time[i]
        nodes[middle-i] = nodes[middle-i+1] * exp(sigma * sqrt(2*deltat) )  
        nodes[middle+i] = nodes[middle+i-1] * exp(-sigma * sqrt(2*deltat) ) 
        ttree[1:2*i+1,i+1] = nodes[middle-i:middle+i]
        i = i+1
    endo
;======= transition probabilities
    denominator = exp(sigma*sqrt(deltat/2))-exp(-sigma*sqrt(deltat/2))
    p = exp((r-div)*deltat/2)-exp(-sigma*sqrt(deltat/2))   ; numerator only
    q = exp(sigma*sqrt(deltat/2))-exp((r-div)*deltat/2)    ; numarator only
    p = (p / denominator)^2
    q = (q / denominator)^2 
;======= option price
    ow = NaN * matrix(rows(nodes),steps+1)
; option inner value
    if(opt == 1)    ; Call
        iw = (nodes-K).*(nodes-K>0)
    else    ; Put
        iw = (K-nodes).*(K-nodes>0)
    endif ; (opt == 1)
; option price
    ow[,steps+1]=iw       ; pay-off in the last column
    j1 = steps
    while(j1 > 0)
        deltat = time[j1+1]-time[j1]
        dis = exp(-r*deltat)
        j2 = 2*j1-1
        while(j2 > 0)
            ow[j2,j1]=p*ow[j2,j1+1]+(1-p-q)*ow[j2+1,j1+1]+q*ow[j2+2,j1+1]
            ow[j2,j1]=dis*ow[j2,j1]
            j2 = j2 - 1
        endo
        j1 = j1 - 1
    endo
    optprice = ow[1,1]
endp

