proc(newnodes,overwritten) = ITTnewnodes(old, r, sigma, deltat, div)
; ---------------------------------------------------------------------
; Library     finance
; ---------------------------------------------------------------------
; See_also    ITTcrr, ITTad, ITT
; ---------------------------------------------------------------------
; Macro       ITTnewnodes
; ---------------------------------------------------------------------
; Description An auxiliary macro for ITT. It computes new level for an 
;             implied trinomial tree, so that the forward condition is 
;             not violated
; ---------------------------------------------------------------------
; Usage       {newnodes,overwritten}=ITTnewnodes(old,r,sigma,deltat,div)
; Input       
; Parameter   old
; Definition  vector; last level of the trinomial tree
; Parameter   r
; Definition  scalar; CONTINUOUS interest rate from interval (0,1)
; Parameter   sigma
; Definition  scalar; constant volatility from interval (0,1)
; Parameter   deltat
; Definition  scalar; the time step between the last and the new level
; Parameter   div
; Definition  scalar; the CONTINUOUS dividend rate from interval (0,1)
; Output      
; Parameter   newnodes
; Definition  vector; new level of the ITT where the forward price is 
;             allways between the relevant tree nodes
; Parameter   overwritten
; Definition  vector; overwritten nodes (because of the arbitrage) in 
;             the new level of the ITT
; ---------------------------------------------------------------------
; Author      K. Komorad 20020131
; ---------------------------------------------------------------------
;
    dims=dim(r)|dim(sigma)|dim(deltat)|dim(div)
    error(sum(dims)!=4,"r, sigma, deltat and div must be scalars")
    error(dim(dim(old))>1,"old must be vector")
    error((r<0)||(r>1),"r must be from (0,1)")
    error((sigma<0)||(sigma>1),"sigma must be from (0,1)")
    error(deltat<=0,"deltat must be positive")
    error((div<0)||(div>1),"div must be from (0,1)")
; initialization and auxiliary variables
    rowa = rows(old)               ; rows in the actual level
    rown = rowa + 2                ; rows in the new level
    newnodes = NaN*matrix(rown)   
    overwritten = newnodes   
;    
    F = old * exp((r-div)*deltat)  ; vector of Forward prices
    if(rowa==1)  ; from the root
        newnodes[1] = old * exp(sigma * sqrt(2*deltat) )
        if(newnodes[1] < F)      	; stop the arbitrage 
            newnodes[1] = 1.01*F 	; 1% larger than F
            overwritten[1] = newnodes[1]
        endif       ; if(newnodes[1]<F)
        newnodes[2] = old 
        newnodes[3] = old * exp(-sigma * sqrt(2*deltat) )
        if(newnodes[3] > F)		; stop the arbitrage
            newnodes[3] = 0.99*F	; 1% smaller than F
            overwritten[3] = newnodes[3]
        endif       ; if(newnodes[3]>F)
    else  ; higher levels
        j = 1
        while(j <= rown)        ; go through the new level from the upside downwards
            ovri = NaN		; no arbitrage occured yet
            switch
            case(j==1)  ; highest knot - must be higher than the forward price only
                new = old[j] * exp(sigma * sqrt(2*deltat) )
                if(new < F[j])		; stop the arbitrage 
                    new = 1.01*F[j]	; 1% larger than F
                    ovri = new
                endif       ; if(new < F[j])
                break
            case(j==2)  ; 2nd highest knot - increasing arrow must be higher than the forward,
                        ; but no condition for decreasing arrow (there is none)
                new1 = old[j-1]
                new2 = old[j] * exp(sigma * sqrt(2*deltat) )
                new = (new1+new2)/2
                if(new < F[j])   ; stop the arbitrage
                    new = (F[j-1]+F[j]) / 2     ; set to the middle of the two forward prices
                    ovri = new
                endif
                break
            case(j==rown)   ; lowest knot - must be lower than the forward price only
                new = old[j-2] * exp(-sigma * sqrt(2*deltat) )
                if(new > F[j-2])      	; stop the arbitrage
                    new = 0.99*F[j-2]	; 1% smaller than F
                    ovri = new
                endif            
                break
            case(j==rown-1) ; 2nd lowest knot - decreasing arrow must be lower than the forward,
                            ; but no condition for increasing arrow (none exists)
                new1 = old[j-2] * exp(-sigma * sqrt(2*deltat) )
                new2 = old[j-1]
                new = (new1+new2)/2
                if(new > F[j-2])   ; stop the arbitrage
                    new = (F[j-2]+F[j-1]) / 2     ; set to the middle of the two forward prices
                    ovri = new
                endif
                break
            case((j>2)&&(j<rown-1)) ; something else                    ; old[j-2] \
                new1 = old[j-2] * exp(-sigma * sqrt(2*deltat) )         ;           \    
                new2 = old[j-1]                                         ; old[j-1] - new
                new3 = old[j] * exp(sigma * sqrt(2*deltat) )            ;           /
                new  = (new1 + new2 + new3) / 3                         ; old[j]   / 
                if((new<F[j])||(new>F[j-2]))   ; stop the arbitrage
                    new = (F[j]+F[j-2])/2
                    ovri = new
                endif
                break          
            endsw
            overwritten[j] = ovri
            newnodes[j] = new
            j = j + 1
        endo
    endif    ; if(rowa==1)         ; from the root
endp


