proc(nodes)=grITTcrr(onetree,time,col,scale,prec)
; ---------------------------------------------------------------------
; Library     finance 
; ---------------------------------------------------------------------
; See_also    grITTspd, grITTstsp, ITT, plotITT
; ---------------------------------------------------------------------
; Macro       grITTcrr
; ---------------------------------------------------------------------
; Description generates a trinomial tree built from the standard CRR tree
;             and describes it with the given values
; ---------------------------------------------------------------------
; Usage       nodes=grITTcrr(onetree{,time{,col{,scale{,prec}}}})
; Input       
; Parameter   onetree
; Definition  matrix; correspond to any output matrix of ITT(.): Ttree, P, Q, AD
;             or LocVol.
; Parameter   time
; Definition  vector; time points corresponding to onetree. Default is
;             time with constant time step (equidistant time axe).
; Parameter   col
; Definition  (2x1) vector; 1st row is the color of the mesh, 
;             2nd is the text color. Default is col=0|1.
; Parameter   scale
; Definition  scalar; 1 if the tree should be scaled by the values given in onetree.
;             0 if a constant-volatility trinomial tree should be plot and the values of onetree
;             will be used only like description (possible usage for P,Q,LocVol...),
;             this is default.
; Parameter   prec
; Definition  scalar; the precision of the description
; Output      
; Parameter   nodes
; Definition  graphical object; the nodes of the tree with corresponding
;             description
; ---------------------------------------------------------------------
; Example    
;    library("finance")
;    library("graphic")
;
;    proc(sigma)=volafunc(S,K,time)	
;        sigma=0.15 + (S-K)/10 * 0.005 
;    endp
;
;    S = 100	        ; current index level
;    r = 0.1		; compounded riskless interest rate
;    div = 0.05         ; dividend yield
;    time = 0|1|3|6	; time vector
;    col=1|0            ; blue mesh with black text
;    prec=3		; desired precision
;
;    t=ITT(S, r, div, time, "volafunc")  ; compute the tree
;    tr=grITTcrr(t.AD,time,col,0,prec)   ; Arrow-Debreu prices
;    d=createdisplay(1,1)
;    axesoff()
;    show(d,1,1,tr)
;    axeson()
; ---------------------------------------------------------------------
; Result      plot of Arrow-Debreu prices
; ---------------------------------------------------------------------
; Keywords    implied volatilities, Black Scholes model, option pricing
; ---------------------------------------------------------------------
; Reference   E. Derman, I. Kani and N. Chriss (1996): 
;             Implied Trinomial Trees of the Volatility Smile <br>
;             K. Komorad (2002): Implied Trinomial Trees and Their Implementation with XploRe
; ---------------------------------------------------------------------
; Link        www.gs.com/qs
; ---------------------------------------------------------------------
; Author      K. Komorad 20020326
; ---------------------------------------------------------------------
;
; dimensions
    k  = rows(onetree)
    l  = cols(onetree)
    kl = k*l
    error(k!=2*l-1,"onetree has wrong dimensions")
; optional parameters
    if(exist("time")!=1)    ; equidistant time axe
        time = 0:l-1
    else
        error(sum(sort(time)!=time),"time must be increasing")
        tmp=time[2:rows(time)]-time[1:rows(time)-1]
        error(sum(tmp<=0),"time must be increasing")
        error(rows(time)!=l,"rows(time) <> cols(onetree)")
    endif
    if(exist("col")!=1)    
        col = 0|1
    else
        error(rows(col)!=2,"wrong col vector")
    endif    
    if(exist("scale")!=1)    
        scale = 0
    else
        error((scale!=1)&&(scale!=0),"scale must be either 0 or 1")
    endif    
    if(exist("prec")!=1)
        prec="2"
    else
        error(dim(prec)!=1,"prec must be scalar")
        error((floor(prec)!=prec)||(prec<0),"prec must be positive integer")
        prec=string("%.0f",prec)
    endif
;   
    if(!scale)
      ; compute a constant CRR tree
        crr = ITTcrr(100,100,0,0.1,time,0,0)
        crr = log(crr.ttree)		    ; constant spacing between the nodes
    else
        crr = onetree
    endif 
    
; build up the tree 
    xtime = vec(matrix(k)*trans(time))      ; x coordinates
    ytree = vec(crr)                        ; y coordinates
    text = vec(onetree)                     ; description
    nodes = xtime~ytree~text
    ; delete NaN's
    ind1 = (1:kl) .* (ytree != NaN)
    ind1 = paf(ind1,ind1>0)
    nodes  = nodes[ind1,]		; coordinates of all tree nodes
    rpo = rows(nodes)

; connect the nodes 
    po = 1:(l-1)^2           ; (l-1)^2 is the # of nodes up to the last level
    pmat = vec(po'|po'|po')  ; the starting knots (3 lines go from each of them)

    tmp = 2                  ; the final knots on each level
    j = 2
    while(j <= l-1)              
        b = tmp[rows(tmp)]+3
        tmp = tmp | (b:(b+2*j-2))
        j = j+1
    endo
    tmp = vec(trans(tmp ~ tmp+1 ~ tmp+2)) ; 3 daughter nodes for each parental node
    pmat = pmat ~ tmp

    descr=string("%."+prec+"f",nodes[,3])       ; description of the nodes    
    nodes=nodes[,1:2]
; set up the masks
    setmaskl(nodes,pmat,col[1],1,1)
    setmaskp(nodes, 0, 8, 3)
    setmaskt(nodes, descr, col[2], 12, 12)
endp

