proc(nodes,probs,axe,on,op)=grITTstsp(itree,text,col,prec)
; ---------------------------------------------------------------------
; Library     finance 
; ---------------------------------------------------------------------
; See_also    grITTcrr, grITTspd, ITT, plotITT
; ---------------------------------------------------------------------
; Macro       grITTstsp
; ---------------------------------------------------------------------
; Description generates the state space of an implied trinomial tree
; ---------------------------------------------------------------------
; Usage       {nodes,probs,axe,on,op}=grITTstsp(itree{,text{,col{,prec}}})
; Input       
; Parameter   itree
; Definition  list; output of ITT(.) <br>
;             Ttree - the Implied Trinomial Tree, P - the upper probabilities
;             Q - the lower probabilities, AD - the Arrow-Debreu prices
;             LocVol - the local volatilities, Onodes - the overwritten nodes,
;             Oprobs - the overwritten probabilities and Time - time points.
; Parameter   text
; Definition  scalar; 1 for the description of the tree nodes with the values
;             of the underlying (default), 0 for no description.
; Parameter   col
; Definition  (5x1) vector; each row correspond in the following order to the color of:
;             the tree mesh, the description of the nodes, the description of the probabilities,
;             the overwritten nodes and the overwritten probabilities.
;             Default is col=0|1|2|4|5.
; Parameter   prec
; Definition  (2 x 1) vector; prec[1] is the precision of the nodes description,
;             prec[2] is the precision of the transition probabilities description.
; Output      
; Parameter   nodes
; Definition  graphical object; the knots of the implied trinomial tree 
; Parameter   probs
; Definition  graphical object; the description for the probabilities
; Parameter   axe
; Definition  graphical object; the time axe
; Parameter   on
; Definition  graphical object; overwritten nodes, denoted with rhombus
;             (if no nodes were overwritten, on=NaN)
; Parameter   op
; Definition  graphical object; overwritten probabilities
;             (if no probabilities were overwritten, op=NaN)
; ---------------------------------------------------------------------
; Example     
;    library("finance")
;    library("graphic")
;
;    proc(sigma)=volafunc(S,K,time)	
;        sigma=0.15 + (S-K)/10 * 0.005 
;    endp
;
;    S = 100	        ; current index level
;    r = 0.1		; compounded riskless interest rate
;    div = 0.05         ; dividend yield
;    time = 0|1|3|6	; time vector
;    col = 0*matrix(5)  ; black only
;    prec = 1|3		; desired precision
;
;    t=ITT(S, r, div, time, "volafunc")  ; compute the tree
;    stsp=grITTstsp(t,0,col,prec)        ; don't describe the nodes
;    d=createdisplay(1,1)
;    axesoff()
;    show(d,1,1,stsp.nodes,stsp.probs)
;    axeson()
; ---------------------------------------------------------------------
; Result      
;    An implied trinomial tree with described transition probabilities.
; ---------------------------------------------------------------------
; Keywords    implied volatilities, Black Scholes model, option pricing
; ---------------------------------------------------------------------
; Reference   E. Derman, I. Kani and N. Chriss (1996): 
;             Implied Trinomial Trees of the Volatility Smile <br>
;             K. Komorad (2002): Implied Trinomial Trees and Their Implementation with XploRe
; ---------------------------------------------------------------------
; Link        www.gs.com/qs
; ---------------------------------------------------------------------
; Author      K. Komorad 20020326
; ---------------------------------------------------------------------
;
; get the input parameters
    ttree = itree.Ttree
    ad    = itree.AD
    probp = itree.P
    probq = itree.Q
    lv    = itree.LocVol
    on    = itree.Onodes
    op    = itree.Oprobs
    time  = itree.Time
    steps = rows(time)-1
; optional parameters
    if(exist("text")!=1)
        text = 1
    else
        error((text!=1) && (text!=0),"text must be either 0 or 1")
    endif
    if(exist("col")!=1)
        col=0|1|2|4|5
    else
        error(rows(col)!=5,"wrong col vector")
    endif
    if(exist("prec")!=1)
        prec1="2"
        prec2="3"
    else
        error(rows(prec)!=2,"prec must have two rows")
        error(sum((floor(prec)!=prec)||(prec<0)),"prec must contain positive integers")
        prec1=string("%.0f",prec[1])
        prec2=string("%.0f",prec[2])
    endif
; dimensions
    k  = rows(ttree)
    l  = cols(ttree)
    kl = k*l
    
;============= the state space
; determine the regular nodes
    xtime = vec(matrix(k)*trans(time))
    ytree = vec(ttree)
    nodes = xtime~ytree
    ; delete NaN's
    ind1 = (1:kl) .* (ytree != NaN)
    ind1 = paf(ind1,ind1>0)
    nodes  = nodes[ind1,]		; coordinates of all tree nodes
    rpo = rows(nodes)

; connect the nodes 
    po = 1:(l-1)^2           ; (l-1)^2 is the # of nodes up to the last level
    pmat = vec(po'|po'|po')  ; the starting knots (3 lines go from each of them)

    tmp = 2                  ; the final knots on each level
    j = 2
    while(j <= l-1)              
        b = tmp[rows(tmp)]+3
        tmp = tmp | (b:(b+2*j-2))
        j = j+1
    endo
    tmp = vec(trans(tmp ~ tmp+1 ~ tmp+2))     ; 3 daughter nodes for each parental node
    pmat = pmat ~ tmp

;============= time axe
    xmin =time[1]-0.2*(time[2]-time[1])		; shift it a little bit to the left
    xmax =time[steps+1]+0.1*(time[2]-time[1])	; shift it a little bit to the right
    xlim = xmin|xmax
    ymax = max(max(ttree),2)
    ymin = min(min(ttree),2)
    ymin = ymin - 0.2*(ymax-ymin)
    ylim = ymin|ymax
    if(sum(time!=floor(time)))
        if(sum((100*time)!=floor(100*time)))
            xfmtstr = "%.4f"  	; this should be large enough
        else
            xfmtstr = "%.2f"	
        endif
    else
        xfmtstr = "%.0f"
    endif      
    axe=graxes(xlim~ylim,"origin",xmin|ymin,"xticks",time,"xfmtstr",xfmtstr,"yticks",ymin,"ytextpos",-1,"ycolor",7,"ytcolor",7)

;============= other features 
; determine the overridden probabilities
    if(rows(op)>1)
        op = op[2:rows(op),]
        nov = rows(op) / 4	; # of overrides
        error(nov != floor(rows(op) / 4),"op is wrong")        
        st = aseq(1,nov,4)
        fi = vec((st+1)'|(st+2)'|(st+3)')
        st = vec(st'|st'|st')
        opmat = st ~ fi
    else
        op = NaN	; set it up to be consistent with on
    endif

; determine the overridden nodes
    if(rows(on) > 1)
        ytree2 = vec(on)
        xtime = vec(matrix(k)*trans(time))
        on = xtime~ytree2
        ; delete NaN's
        ind2 = (1:kl) .* (ytree2 != NaN)
        ind2 = paf(ind2,ind2>0)
        on = on[ind2,]
    endif

; description of the nodes    
    if(text==1)
        ttext=string("%."+prec1+"f",nodes[,2])
    endif
    
; description of the arrows
  ; get the probabilities
    w = rows(probp)*cols(probp)
    probp = vec(probp)
    probq = vec(probq)
    ; delete NaN's
    ind3 = (1:w) .* (probp != NaN)
    ind3 = paf(ind3,ind3>0)
    probp  = probp[ind3,]	
    probq  = probq[ind3,]	
    rpr = rows(probp)
  ; text with the probabilities
    probstext=string("%."+prec2+"f",probp)|string("%."+prec2+"f",1-probp-probq)|string("%."+prec2+"f",probq)        

  ;========== prepare the adjusting 'shifts'
  ; shift in the horizontal direction
    xshift = matrix(k)*trans(time[2:steps+1]-time[1:steps])    ; time steps
    xshift = vec(xshift~0*matrix(k))	; last column only pro forma
    ; delete NaN's
    ind4 = (1:rows(xshift)) .* (ytree != NaN)
    ind4 = paf(ind4,ind4>0)
    xshift = xshift[ind4[1:rows(ind4)-k]]

  ; shift in the vertical direction
    ; indices for deleting NaN's
    smalltr = vec(ttree[1:k-2,1:l-1])	; any tree; one level smaller than ttree
    ind5 = (1:(k-2)*(l-1)) .* (smalltr != NaN)
    ind5 = paf(ind5,ind5>0)
  ; differences: (upper son) - parent
    diffr = ttree[1:k-2,2:l] - ttree[1:k-2,1:l-1]
    yshiftup = vec(diffr)[ind5]	; delete NaN's and vectorize
  ; differences: (lower son) - parent
    nan=NaN|NaN
    diffr = ttree[,l] - (nan|ttree[1:k-2,l-1])	; differences in the vertical direction
    m=1
    while(m+1<l)	; make the differences from the back in the tree
        tmp = (nan|ttree[1:k-m-1,l-m]) - (nan|NaN|NaN|ttree[1:k-m-3,l-m-1])
        diffr = (tmp[2*m+1:k]|nan) ~ diffr
        nan = NaN|NaN|nan
        m=m+1
    endo
    diffr = diffr[3:k,]
    yshiftdown = vec(diffr)[ind5]	; delete NaN's and vectorize
  ; differences: (middle son) - parent
    diffr = ttree[,l] - (NaN|ttree[1:k-1,l-1])	; differences in the vertical direction
    nan=NaN
    m=1
    while(m+1<l)	; make the differences from the back in the tree
        tmp = (nan|ttree[1:k-m,l-m]) - (nan|NaN|ttree[1:k-m-1,l-m-1])
        diffr = (tmp[m+1:k]|nan) ~ diffr
        nan = NaN|nan
        m=m+1
    endo
    diffr = diffr[2:k-1,]
    yshiftmid = vec(diffr)[ind5]	; delete NaN's and vectorize

  ; finaly determine the coordinates of the text with probabilities
    probs = nodes[1:rpr,] | nodes[1:rpr,] | nodes[1:rpr,]
    probs[,1] = probs[,1] + (0.25*xshift | 0.5*xshift | 0.75*xshift)
    probs[,2] = probs[,2] + (0.4*yshiftup | 0.5*yshiftmid | 0.8*yshiftdown)
;========= postscript shift - activate for printing the display    
    probs[,2] = probs[,2] - 0.25*(yshiftup | 0.2*yshiftmid | 0.25*yshiftdown) 
;========= set up the masks
    setmaskl(nodes,pmat,col[1],1,1)
    setmaskp(nodes, 0, 8, 3)
    setmaskt(probs,probstext, col[3], 12, 12)
    setmaskp(probs, 0, 0, 0)
    if(on != NaN)
        setmaskp(on, col[4], 9, 5)
    endif        
    if(op != NaN)
        setmaskl(op,opmat,col[5],1,1)
        setmaskp(op, 0, 0, 0)
    endif        
    if(text==1)
        setmaskt(nodes, ttext, col[2], 12, 12)
    endif        
endp

