proc()=plotITT(itree,what,r,text,prtext)
; ---------------------------------------------------------------------
; Library     finance 
; ---------------------------------------------------------------------
; See_also    grITTcrr, grITTstsp, grITTspd, ITT
; ---------------------------------------------------------------------
; Macro       plotITT
; ---------------------------------------------------------------------
; Description plots desired components of an implied trinomial tree
; ---------------------------------------------------------------------
; Usage       plotITT(itree,what{,r{,text{,prtext}}})
; Input       
; Parameter   itree
; Definition  list; output of ITT(.) <br>
;             Ttree - the Implied Trinomial Tree, P - the upper probabilities
;             Q - the lower probabilities, AD - the Arrow-Debreu prices
;             LocVol - the local volatilities, Onodes - the overwritten nodes,
;             Oprobs - the overwritten probabilities and Time - the time vector.
; Parameter   what
; Definition  vector; with maximal 5 rows, the rows correspond to: <BR>
;             (1) state space of the ITT, (2) transition probabilities, 
;             (3) tree of local volatilities, (4) the tree of Arrow-Debreu prices and 
;             (5) the state price density. Row for each item 
;             which should be plotted must contain a non-zero.
; Parameter   r
; Definition  scalar; interest rate, from interval (0,1). Necessary only when the 
;             state price density should be plotted.
; Parameter   text
; Definition  scalar; 1 for the description of the tree nodes with the values
;             of the underlying (default), 0 for no description.
;             Applicable only when the state space of the ITT should be plotted. 
; Parameter   prtext
; Definition  scalar; 1 for the description of the tree arrows with corresponding 
;             probabilities, 0 for no description (default).  
;             Applicable only when the state space of the ITT should be plotted. 
; Output      
; ---------------------------------------------------------------------
; Notes       A plot of the corresponding implied trinomial tree will 
;             be returned. The tree is growing from the root on the left
;             hand side to the right. Each tree node has a text label
;             with the spot price. Any node which violated the forward
;             condition is designated with a red rhombus.
;             The tree arrows are denoted with the corresponding probability. 
;             If the original probability fell out of the interval (0,1) 
;             it had to be overwritten and all three arrows are now magenta.
; ---------------------------------------------------------------------
; Example     
;    library("finance")
;    library("plot")
;
;    proc(sigma)=volafunc(S,K,time)	
;        sigma=0.15 + (S-K)/10 * 0.005 
;    endp
;
;    S = 100	        ; current index level
;    r = 0.1		; compounded riskless interest rate
;    div = 0.05         ; dividend yield
;    time = 0|1|3|6	; time vector
;
;    t=ITT(S, r, div, time, "volafunc")
;    plotITT(t,0|0|0|0|1,r,time) ; plot the state price density
; ---------------------------------------------------------------------
; Result      the state price density curve 
; ---------------------------------------------------------------------
; Example     
;    library("finance")
;    library("plot")
;
;    proc(sigma)=volafunc(S,K,time)	
;        sigma=0.15 + (S-K)/10 * 0.01 
;    endp
;
;    S = 100	        ; current index level
;    r = 0.1		; compounded riskless interest rate
;    div = 0.05         ; dividend yield
;    time = 0|3|5|6	; time vector
;    what = 0|1|1	; plot the probabilities and loc. volatilities
;
;    t=ITT(S, r, div, time, "volafunc")
;    plotITT(t,what)
; ---------------------------------------------------------------------
; Result      one display with the transition probabilities and 
;             one display showing the local volatilities
; ---------------------------------------------------------------------
; Keywords    implied volatilities, Black Scholes model, option pricing
; ---------------------------------------------------------------------
; Reference   E. Derman, I. Kani and N. Chriss (1996): 
;             Implied Trinomial Trees of the Volatility Smile <br>
;             K. Komorad (2002): Implied Trinomial Trees and Their Implementation with XploRe
; ---------------------------------------------------------------------
; Link        www.gs.com/qs
; ---------------------------------------------------------------------
; Author      K. Komorad 20020326
; ---------------------------------------------------------------------
;
    methods=5	; maximal number of rows of vector what
    error(exist("what")!=1,"what parameter missing")
    error(dim(dim(what))!=1,"what must be a vector")
    error(dim(what) > methods,"what has too many rows")
    what=what|0*matrix(methods+1-rows(what))    ; extend what to the maximal length
; get the input parameters
    ttree = itree.Ttree
    ad    = itree.AD
    probp = itree.P
    probq = itree.Q
    lv    = itree.LocVol
    time  = itree.Time
; optional parameters
    if(exist("text")!=1)
        text = 1
    else
        error((text!=1) && (text!=0),"text must be either 0 or 1")
    endif
    if(exist("prtext")!=1)
        prtext = 0
    else
        error((prtext!=1) && (prtext!=0),"prtext must be either 0 or 1")
    endif

    axesoff()        
; split in order to the required job
    switch
        case(what[1])   ; plot the main ITT state space
            d1 = createdisplay(1,1)
            o  = grITTstsp(itree,text)
            if(prtext)
                show(d1,1,1,o.axe,o.nodes,o.probs)
            else
                show(d1,1,1,o.axe,o.nodes)
            endif
            if(o.on!=NaN)
                adddata(d1,1,1,o.on)
            endif
            if(o.op!=NaN)
                adddata(d1,1,1,o.op)
            endif
            setgopt(d1,1,1,"title","Implied Trinomial Tree")
        case(what[2])   ; plot the probabilities
            d2 = createdisplay(1,3)
            n1 = grITTcrr(probp)
            n2 = grITTcrr(1-probp-probq)
            n3 = grITTcrr(probq)
            show(d2,1,1,n1)
            show(d2,1,2,n2)
            show(d2,1,3,n3)
            setgopt(d2,1,1,"title","Upper Probabilities")
            setgopt(d2,1,2,"title","Middle Probabilities")
            setgopt(d2,1,3,"title","Lower Probabilities")
        case(what[3])   ; plot the local volatility
            d3 = createdisplay(1,1)
            o  = grITTcrr(lv)
            show(d3,1,1,o)
            setgopt(d3,1,1,"title","Local Volatility")
        case(what[4])   ; plot Arrow-Debreu price
            d4 = createdisplay(1,1)
            o  = grITTcrr(ad,time)
            show(d4,1,1,o)
            setgopt(d4,1,1,"title","Arrow-Debreu Prices")
        case(what[5])   ; plot the SPD
            error(exist("r")!=1,"r parameter is missing")
            axeson()
            d5 = createdisplay(1,1)
            o  = grITTspd(ttree,ad,r,time)
            o  = setmask(o,"line","blue")
            show(d5,1,1,o)
            setgopt(d5,1,1,"title","State Price Density")
            setgopt(d5,1,1,"xlabel","underlying price","ylabel","probability")
    endsw
    axeson()    
endp


