function y = MW2cdf(X,N)
%MW2CDF   Mann-Whitney's U cumulative distribution function. 
%Probability of obtaining a Mann-Whitney's U of two random variables with continuous
%cumulative distribution. This procedure is highly recommended for sample sizes
%7< nx & ny <=40. For nx & ny <=7 it is recommended to use the MW1cdf function;
%otherwise, the asymptotic p-value may be a poor approximation. For nx & ny >40 you
%must to use the normal approximation. 
%(The variable x will be called stochastically smaller than y if f(a) > g(a) for every a).
%[Based on the Fortran77 algorithm AS 62 Appl. Statist. (1973)]
%
%   Syntax: function y = MW2cdf(X,N) 
%      
%     Inputs:
%          X - Mann-Whitney's U statistic.
%          N - 2-element vector of sample sizes for the two samples []. 
% The input quantities should be scalars.
%     Outputs:
%          y - cumulative probability value associated to the given Mann-Whitney's statistic.
%
%    Example: From the random data of two independent samples we got the
%             Mann-Whitney's statistic U = 337. Sample sizes were n1 = 17,
%             n2 = 28. We are interested to get the associated cumulative 
%             probability value.
%
%                              U=337;N=[17,28];
%
%     Calling on Matlab the function: 
%             y = MW2cdf(X,N)
%
%       Answer is:
%
%                 0.9906    
%

%  Created by A. Trujillo-Ortiz and R. Hernandez-Walls
%             Facultad de Ciencias Marinas
%             Universidad Autonoma de Baja California
%             Apdo. Postal 453
%             Ensenada, Baja California
%             Mexico.
%             atrujo@uabc.mx
%
%  May 10, 2003.
%
%  To cite this file, this would be an appropriate format:
%  Trujillo-Ortiz, A. and R. Hernandez-Walls. (2003). MW2cdf: Mann-Whitney's U cumulative
%    distribution function: nx or ny >7. A MATLAB file. [WWW document]. URL http://
%    www.mathworks.com/matlabcentral/fileexchange/loadFile.do?objectId=3494&objectType=FILE
%
%  References:
% 
%  Mann, H. B. and Whitney, D. R. (1947), On a test of whether one of two   
%           random variables is stochastically larger than the other. Annals
%           of Mathematical Statistics, 18: 50-60.
%  Algorithm AS 62 (1973). Journal of Applied Statistics, 22(2):1-3.
%

if nargin <  2,
   error('Requires two input arguments.');
end

U=X;  %Mann-Whitney's U statistic.

Um=prod(N);  %maximum Mann-Whitney's U.

if U > Um;
   fprintf('Error: For this 2-sample sizes the maximum Mann-Whitney''s U value must be: %2i\n\n', Um);
end
   
nmin = min(N);  %largest sample size.
nmax = max(N);  %smallest sample size.

if (nmin <= 7) & (nmax <= 7);
   fprintf('Warning: For nx and ny <= 7, the p-value may be a poor approximation.\n'); 
   fprintf('It is recommended to use the MW1cdf function you can find on the\n');
   fprintf('Matlab>File Exchange Antonio Trujillo-Ortiz'' Author Page.\n');
   disp(' ');
   cont=input('Do you want to continue anyway (y/n):','s');
   
   if (cont=='y');
      disp('Here it goes.');
      
      mn1 = prod(N)+1;
      n1 = nmax+1;
      freq = [ones(n1,1); zeros(mn1-n1,1)];
      
      lwrk = floor((mn1+1)/2 + nmin);
      work = zeros(lwrk,1);
      
% Generate successively higher-order distributions
      in = nmax;
      for i = 2:nmin
         in = in+nmax;
         n1 = in+2;
         l = 1 + in/2;
         k = i;
         
% Generate complete distribution from outside inwards
         for j = 1:l
            k = k+1;
            n1 = n1-1;
            summ = freq(j) + work(j);
            freq(j) = summ;
            work(k) = summ - freq(n1);
            freq(n1) = summ;
         end;
      end;
      
      freq = freq/sum(freq);  % Make distribution relative
      
% Cumulative frequency distribution
      cumfreq = cumsum(freq);
      
      UU = [0:length(freq)-1];  %vector of all the possible Mann-Whitney's U values.
      
      u = find(UU == U);  %location of the interested Mann-Whitney's U on all the possible U's for this 2-sample sizes.
      y = cumfreq(u);  %association of the interested Mann-Whitney's U with its cumulative distribution.
      
   else
   end
else
   mn1 = prod(N)+1;
   n1 = nmax+1;
   freq = [ones(n1,1); zeros(mn1-n1,1)];
   
   lwrk = floor((mn1+1)/2 + nmin);
   work = zeros(lwrk,1);
   
% Generate successively higher-order distributions
   in = nmax;
   for i = 2:nmin
      in = in+nmax;
      n1 = in+2;
      l = 1 + in/2;
      k = i;
      
% Generate complete distribution from outside inwards
      for j = 1:l
         k = k+1;
         n1 = n1-1;
         summ = freq(j) + work(j);
         freq(j) = summ;
         work(k) = summ - freq(n1);
         freq(n1) = summ;
      end;
   end;
   
   freq = freq/sum(freq);  % Make distribution relative
   
   % Cumulative frequency distribution
   cumfreq = cumsum(freq);
   
   UU = [0:length(freq)-1];  %vector of all the possible Mann-Whitney's U values.
   
   u = find(UU == U);  %location of the interested Mann-Whitney's U on all the possible U's for this 2-sample sizes.
   y = cumfreq(u);  %association of the interested Mann-Whitney's U with its cumulative distribution.
end
