function [pWert,AnovTab,TestStat] = anova1B(X,Gruppe,varargin)
%
% Funktion anova1B
%
% Aufruf: [pWert,AnovTab,TestStat] = anova1B(X,Gruppe,P1,P2,...)
%
% Diese Funktion realisiert eine EINFAKTORIELLE VARIANZANALYSE
% nach Art der STATISTICS-TOOLBOX-Funktion anova1.
%
% Die Funktion hat im Wesentlichen die gleiche Aufrufschnittstelle
% wie anova1 und kann als ERSATZFUNKTION eingesetzt werden.
% Die Funktion hat gegenber anova1 EINGESCHRNKTE FUNKTIONALITT.
% Insbesondere erfolgt keine grafische Ergebnisausgabe, wie bei anova1.
%
% Eingangsdaten:	  X          Matrix oder SPALTENVektor der Daten
%                                Bei einer MATRIX wird von einem BALANCIERTEN
%                                Versuchsplan ausgegangen. Die Daten sind 
%                                SPALTENWEISE organisiert.
%                                Ein unbalancierter Versuchsplan muss ber
%                                einen Datenvektor erfolgen, wobei die Zuordnung
%                                zur Gruppe ber den Parameter Gruppe erfolgt.
%
%                     Gruppe     Cell-Array von Strings mit
%                                den Gruppennamen. Falls X ein Vektor ist,
%                                muss Gruppe die gleiche Dimension haben. Die
%                                Daten werden dann jeweils den Namen in Gruppe
%                                zugeordnet. Falls X eine Matrix ist, kann Gruppe
%                                auch weggelassen werden. Dann werden den Spalten
%                                Gruppennummern zugeordnet.
%
%                     varargin   alle anderen Parameter werden ignoriert
%                                 
%
% Ausgangsdaten:	  pWert      der p-Wert
%                                  
%                     AnovTab    Cell-Array mit der ANOVA-Tabelle 
%    
%                     TestStat   Struktur mit der Teststatistik
%
%
% Es werden nur wenige Fehleingaben abgefangen! 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 2.01
% Datum: 29.8.2004/12.9.2004



% Eingangsparameterliste checken

inputparams = nargin;
typ = [];

switch inputparams
    case 0
        error('anova1B muss mindestens einen Parameter haben! Abbruch!');
    case 1
        [n, m] = size(X);
        if m<2
            error('Die Datenmatrix X muss mindestens zwei Spalten haben! Abbruch!')
        else
            typ = 'balanciert1';   % balancierter Versuchman ohne Gruppennamen
        end
    otherwise
        if ~iscell(Gruppe)
            error('Gruppennamen als Cell-Array von Strings eingeben! Abbruch!');
        end
                                  % Gruppe ist ein Cell-Array
        [nG, mG] = size(Gruppe);
        if mG ~= 1
            error('Gruppennamen mssen ein Cell-Array-Spaltenvektor von Strings sein! Abbruch!');
        end
              
        [n, m] = size(X);
        
        if n==1 & m>1             % Zeilenvektor statt Spaltenvektor wurde bergeben
            X = X';               % in Spaltenvektor umwandeln
            n = m; m = 1;
        end
                   
        if m>=2 & nG == m         % soviel Namen wie Spalten in X
            typ = 'balanciert2';  % balancierter Versuch mit Gruppennamen
        elseif  m<2 & n == nG     % soviel Namen wie Elemente im Vektor X
            typ = 'unbalanciert'; 
        else
            error('Bitte Aufrufkonvention beachten (help anova1B)! Abbruch!');
        end
end
    
% ANOVA-Tabelle vorinitialisieren

AnovTab = cell(4,6);
AnovTab{1,1} = 'Quelle'; AnovTab{1,2} = 'SS'; AnovTab{1,3} = 'df'; AnovTab{1,4} = 'MS'; AnovTab{1,5} = 'F'; AnovTab{1,6} = 'Prob>F';
AnovTab{2,1} = 'Spalten'; 
AnovTab{3,1} = 'Restfehler';
AnovTab{4,1} = 'Totalvariation';

% Teststatistik initialisieren

TestStat.GrpNamen = [];
TestStat.n = [];
TestStat.Mittelwerte = [];
TestStat.Freiheitsgrade = [];
TestStat.s = [];

% Varianzanalyse berechnen


if strcmp(typ, 'balanciert1')|strcmp(typ, 'balanciert2')
        
         if strcmp(typ, 'balanciert1')
             TestStat.GrpNamen = num2cell(cumsum(ones(1,m)));
         else
             TestStat.GrpNamen = Gruppe;
         end
         
         % Abweichungsquadratsumme q1 innerhalb der Gruppen (Spalten von X)
         
         q1 = 0;        % Vorinitialisierung
        
         for k=1:m
            q1 = q1 + var(X(:,k));
         end
        
         q1 = q1*(n-1);
         AnovTab{3,2} = q1; AnovTab{2,3} = m-1;
         TestStat.n = n*ones(1,m);
         TestStat.Mittelwerte = mean(X);
        
         % Abweichungsquadratsumme q2 zwischen den Stichproben berechnen.
         % Am einfachsten: den schon berechneten Wert von q1 von der Gesamt-
         % Abweichungsquadratsumme abziehen
         
         allX = X(:);
         N = length(allX);
         qges = var(allX)*(N-1);
         AnovTab{4,2} = qges; AnovTab{4,3} = N-1; AnovTab{3,3} = N-m;
         TestStat.Freiheitsgrade = N-m;
         q2 = qges - q1;
         AnovTab{2,2} = q2;
         
         nq2 = (1/(m-1))*q2;     % Normierung der Quadratsummen
         nq1 = (1/(N-m))*q1;     % Normierung der Quadratsummen
         AnovTab{2,4} = nq2; AnovTab{3,4} = nq1;
         TestStat.s = sqrt(nq1);
         
         t=nq2/nq1;              % Wert der Testvariablen
         AnovTab{2,5} = t;
         
         % Bestimmung des p-Wertes
         
         pWert = 1-fcdfB(t, m-1, N-m);
         AnovTab{2,6} = pWert;
         
else % unbalanciert
    
        % Gruppennamen aufsteigend sortieren
    
        [GrpX,Ind] = sort(Gruppe);
        
        % Datenvektor entsprechend umsortieren
        
        Y = X(Ind);
        
        % Gruppen identifizieren
        
        [grpindx] = identgrp(GrpX);
        
        % Zahl Gruppen und der Elemente pro Gruppe
        
        m = length(grpindx); grpindx = [grpindx, length(X)+1];
        TestStat.n = diff(grpindx);
        
        % Abweichungsquadratsumme q1 innerhalb der Gruppen (Spalten von X)
         
         q1 = 0;        % Vorinitialisierung
         TestStat.Mittelwerte = zeros(1,m);
        
         for k=1:m
            mk = grpindx(k+1)-grpindx(k);
            q1 = q1 + (mk-1)*var(X(grpindx(k):grpindx(k+1)-1));
            TestStat.Mittelwerte(k) = mean(X(grpindx(k):grpindx(k+1)-1));
         end
         
         AnovTab{3,2} = q1; AnovTab{2,3} = m-1;
         
         % Abweichungsquadratsumme q2 zwischen den Stichproben berechnen.
         % Am einfachsten: den schon berechneten Wert von q1 von der Gesamt-
         % Abweichungsquadratsumme abziehen
                 
         N = length(X);
         qges = var(X)*(N-1);
         
         AnovTab{4,2} = qges; AnovTab{4,3} = N-1; AnovTab{3,3} = N-m;
         TestStat.Freiheitsgrade = N-m;
         q2 = qges - q1;
         AnovTab{2,2} = q2;
         
         nq2 = (1/(m-1))*q2;     % Normierung der Quadratsummen
         nq1 = (1/(N-m))*q1;     % Normierung der Quadratsummen
         AnovTab{2,4} = nq2; AnovTab{3,4} = nq1;
         TestStat.s = sqrt(nq1);
         
         
         t=nq2/nq1;              % Wert der Testvariablen
         AnovTab{2,5} = t;
         
         % Bestimmung des p-Wertes
         
         pWert = 1-fcdfB(t, m-1, N-m);
         AnovTab{2,6} = pWert;      
end




%------------------------------------------------------------
%
% Subfunction zur Identifikation der Gruppennamen
%

function [grpind] = identgrp(Grps)

grpind = [1];  Grlng = length(Grps);
k = 1; inx = cumsum(ones(1,Grlng));

while k < Grlng
    TF = strcmp(Grps{k},Grps);
    k = inx(diff(TF)==-1)+1;
    grpind = [grpind, k];
end;











