function [beta, varargout] = glmfitB(x,y,distr,link,varargin)
% 
% Funktion glmfitB
%
% Aufruf: [beta] = glmfitB(x,y,distr,link,P1,P2,...)
%
% Diese Funktion schtzt die Koeffizienten eines VERALLGEMEINERTEN
% LINEAREN REGRESSIONSMODELLS
%
%         F(mu) = <X,beta> = SUM(i=1:r  x_i*beta_i) + beta_0
%
% Die Funktion hat im Wesentlichen die gleiche Aufrufschnittstelle
% wie glmfit und kann als ERSATZFUNKTION eingesetzt werden.
% Die Funktion hat gegenber glmfit jedoch eine SEHR EINGESCHRNKTE 
% FUNKTIONALITT, da lediglich das LOGIT- und das PROBIT-Modell fr
% binomialverteilte Beobachtungsgren berechnet werden und auer den
% Regressionskoeffizienten keine weiteren statistischen Informationen
% zurckgeliefert werden.
%
%
% Eingangsdaten:	  x          nxr-Matrix der unabhngigen Variablen
%
%                     y          nx2-Vektor der Beobachtungswerte (1. Spalte)
%                                und der Zahl der Versuche (2.Spalte)
%                                (die 2. Spalte wird lediglich aus Kompati-
%                                bilittsgrnden zu glmfit bentigt)
%
%                     distr      wird intern immer ignoriert (wird lediglich 
%                                aus Kompatibilittsgrnden zu glmfit bentigt)
%
%                     link       Link-Funktion: hier sind nur die Werte 'logit'
%                                und 'probit' zugelassen.
%
%                     varargin   Werte die noch an glmfitB bergeben werden 
%                                knnen, aber ignoriert werden (wird lediglich 
%                                aus Kompatibilittsgrnden zu glmfit bentigt)
%
% Ausgangsdaten:	  beta       geschtzte Koeffizienten
%
%                     varargout  alle anderen Rckgabewerte werden ignoriert
%
% Es werden keine Fehleingaben abgefangen! 
%                                 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 1.01
% Datum: 9.9.2004

inargs = nargin;

if inargs < 4
    error('nlinfitB bentigt mindestens vier Argumente. Abbruch!');
end

[n,r] = size(x); 
if n==1      % Zeilenvektor zulassen
    X = X';  % Spaltenvektor daraus machen
    n = r; r = 1;
end

[m,p] = size(y); 

if m~=n | p~=2    
    error('y muss soviel Zeilen haben, wie x und muss 2 Spalten haben. Abbruch!');
end

fall = [];

if strcmp(link,'logit')
    fall = 'logit';
elseif strcmp(link,'probit')
    fall = 'probit';
else
    error('Der Parameter link muss den Wert ''logit'' oder ''probit'' haben. Abbruch!');
end

if inargs > 4
    warning('Alle Eingabeparameter ab Nummer 5 werden igroriert!');
end

switch fall
    case 'logit'
        [beta] = nlinfitB([ones(n,1),x],y(:,1),@logitfunkt,zeros(1,r+1));
    
    case 'probit'
        [beta] = nlinfitB([ones(n,1),x],y(:,1),@probitfunkt,zeros(1,r+1));
end
  
    


% Subfunctions:
%----------------------------------------------------------------------------------------
% Lokale Funktionen mit dem Logit- und dem Probit-Modell

function [ydach] = logitfunkt(betalf,X)

n=length(betalf);
ydach = 0;
for k=1:n
    ydach = ydach + betalf(k)*X(k);
end

ydach = 1/(1+exp(-ydach));

function [ydach] = probitfunkt(betalf,X)

n=length(betalf);
ydach = 0;
for k=1:n
    ydach = ydach + betalf(k)*X(k);
end
ydach = normcdfB(ydach);