function [Tent, pWert, ConfInt, TStat] = ttest2B(datenx,dateny,alpha,altHyp)
%
% Funktion ztest2B
%
% Aufruf: [Tent, pWert, ConfInt, TStat] = ttest2B(datenx,dateny,alpha,altHyp)
%
% Diese Funktion fhrt fr die Daten in den Vektoren datenx und dateny einen 
% so genannten ZWEI-STICHPROBEN t-Test (Test des Mittelwerts zweier 
% NORMALVERTEILTER Stichproben auf Gleicheit) durch!
% (vgl. Abschnitt 5.4.2)
%
% Die Funktion hat gegenber der Statistics-Toolbox-Funktion  EINGESCHRNKTE
% FUNKTIONALITT. 
%
% Eingangsdaten:	  datenx     1. Stichprobe (Vektor von Werten, Umfang>1)
%
%                     dateny     2. Stichprobe (Vektor von Werten, Umfang>1)
%
%                     alpha      Signifikanzniveau (0<=alpha<=1)
%                                default ist 0.05
%
%                     altHyp     Art des Tests (0 = zweiseitig,
%                                1 = einseitig grer, -1 = einseitig kleiner)
%                                default ist 0
%
% Ausgangsdaten:	  Tent       Testentscheidung (0=nicht abgelehnt, 1=abgelehnt)
%
%                     pWert      p-Wert
%
%                     ConfInt    Konfidenzintervall
%
%                     TStat      Struktur mit dem Testergebnis
%
% Es werden i.W. keine Fehleingaben abgefangen! 
%                                 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 3.01
% Datum: 5.9.2004/12.9.2004/15.9.2004

nargs = nargin;

if sum(size(datenx)>[1,1])>=2 | sum(size(dateny)>[1,1])>=2  % daten keine Vektoren
    error('ttest2B kann fr datenx,dateny nur vektorielle Argumente verarbeiten. Abbruch!');
end


if nargs == 2            % Defaultfall 1
    alpha = 0.05; altHyp = 0;
elseif nargs == 3        % Defaultfall 2
    altHyp = 0;
elseif nargs<2 | nargs>4
    error('Aufrufstruktur von ttest2B beachten. Abbruch!');
else
    % tue nichts
end

% Vorinitialisierungen

n = length(datenx); m = length(dateny);
sigmaxq = var(datenx); sigmayq = var(dateny);
TStat.Testniveau = alpha;
if altHyp<-eps
    TStat.Testart = 'Zwei-Stichproben-t-Test einseitig nach unten';
elseif altHyp >eps
    TStat.Testart = 'Zwei-Stichproben-t-Test einseitig nach oben';
else
    TStat.Testart = 'Zwei-Stichproben-t-Test zweiseitig';
end

TStat.Vertrauensintervall = [];

% Wert der Testvariablen T

faktor = sqrt(n*m/(n+m));
nenner = sqrt(((n-1)*sigmaxq+(m-1)*sigmayq)/(n+m-2));
TStat.Testvariable_T = faktor*(mean(datenx)-mean(dateny))/nenner;

TStat.Freiheitsgrade_T = n+m-2;

% p-Wert
TStat.pWert = [];

TStat.Testentscheidung = [];

% Berechnung des Vertrauensintervalls

if altHyp>eps        % einseitig nach oben
    quantil = tinvB(1-alpha,TStat.Freiheitsgrade_T);
    TStat.Vertrauensintervall = ...
        [mean(datenx)-mean(dateny)-quantil*nenner/faktor, Inf];
    TStat.pWert = 1-tcdfB(TStat.Testvariable_T,TStat.Freiheitsgrade_T);
elseif altHyp <-eps   % einseitig nach unten
    quantil = tinvB(1-alpha,TStat.Freiheitsgrade_T);
    TStat.Vertrauensintervall = ...
        [-Inf, mean(datenx)-mean(dateny)+quantil*nenner/faktor];
    TStat.pWert = tcdfB(TStat.Testvariable_T,TStat.Freiheitsgrade_T);
else                 % zweiseitiger Test
    quantil = tinvB(1-alpha/2,TStat.Freiheitsgrade_T);
    TStat.Vertrauensintervall = ...
        [mean(datenx)-mean(dateny)-quantil*nenner/faktor, ...
              mean(datenx)-mean(dateny)+quantil*nenner/faktor];
    TStat.pWert = min(2*tcdfB(TStat.Testvariable_T,TStat.Freiheitsgrade_T),...
                               2*(1-tcdfB(TStat.Testvariable_T,TStat.Freiheitsgrade_T)));
end

ConfInt = TStat.Vertrauensintervall;
pWert = TStat.pWert;
Tent = TStat.pWert < alpha; 

if Tent
     TStat.Testentscheidung = 'Die Nullhypothese wird abgelehnt!';
else
     TStat.Testentscheidung = 'Die Nullhypothese wird nicht abgelehnt!';
end