function [TEnt,pWert,KonfInt,Entscheidung] = ztestB(daten,mu,sigma,alpha,altHyp)
%
% Funktion ztestB
%
% Aufruf: [TEnt,pWert,KonfInt,Entscheidung] = ztestB(daten,mu,sigma,alpha,altHyp)
%
% Diese Funktion fhrt fr die Daten im Vektor daten einen Z-Test 
% (Test des Mittelwerts einer Normalverteilung bei bekannter Varianz) durch!
% (vgl. Abschnitt 5.4.2)
%
%
% Eingangsdaten:	  daten      Stichprobe (Vektor von Werten)
%
%                     mu, sigma  Parameter der Normalverteilung
%                                default ist mu=0 und sigma=1
%
%                     alpha      Signifikanzniveau (0<=alpha<=1)
%                                default ist 0.05
%
%                     altHyp     Art des Tests (0 = zweiseitig,
%                                1 = einseitig grer, -1 = einseitig kleiner)
%                                default ist 0
%
% Ausgangsdaten:	  TEnt       Testergebnis
%
%                     pWert      p-Wert
%
%                     KonfInt    Konfidenzintervall
%
%                     Entscheidung      
%                                Struktur mit dem Testergebnis
%
% Es werden i.W. keine Fehleingaben abgefangen! 
%                                 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 2.01
% Datum: 5.9.2004/14.9.2004

nargs = nargin;

if sum(size(daten)>[1,1])>=2   % daten kein Vektor
    error('ztestB kann fr daten nur vektorielle Argumente verarbeiten. Abbruch!');
end


if nargs == 1            % Defaultfall 1
    mu = 0; sigma = 1; alpha = 0.05; altHyp = 0;
elseif nargs == 3        % Defaultfall 2
    alpha = 0.05; altHyp = 0;
elseif nargs == 4        % Defaultfall 3
    altHyp = 0;
elseif nargs == 2 | nargs > 5
    error('Aufrufstruktur von ztestB beachten. Abbruch!');
else     
    % tue nichts
end

% Vorinitialisierungen

n = length(daten);
Entscheidung.Testniveau = alpha;
if altHyp<-eps
    Entscheidung.Testart = 'Z-Test einseitig nach unten';
elseif altHyp >eps
    Entscheidung.Testart = 'Z-Test einseitig nach oben';
else
    Entscheidung.Testart = 'Z-Test zweiseitig';
end

Entscheidung.Vertrauensintervall = [];

% Wert der Testvariablen Z
Entscheidung.Testvariable_Z = sqrt(n)*(mean(daten)-mu)/sigma;

% p-Wert
Entscheidung.pWert = [];

Entscheidung.Testentscheidung = [];

% Berechnung des Vertrauensintervalls

if altHyp>eps        % einseitig nach oben
    quantil = norminvB(1-alpha);
    Entscheidung.Vertrauensintervall = ...
        [mean(daten)-quantil*sigma/sqrt(n), Inf];
    Entscheidung.pWert = 1-normcdfB(Entscheidung.Testvariable_Z);
elseif altHyp <-eps   % einseitig nach unten
    quantil = norminvB(1-alpha);
    Entscheidung.Vertrauensintervall = ...
        [-Inf, mean(daten)+quantil*sigma/sqrt(n)];
    Entscheidung.pWert = normcdfB(Entscheidung.Testvariable_Z);
else                 % zweiseitiger Test
    quantil = norminvB(1-alpha/2);
    Entscheidung.Vertrauensintervall = ...
        [mean(daten)-quantil*sigma/sqrt(n), mean(daten)+quantil*sigma/sqrt(n)];
    Entscheidung.pWert = min(2*normcdfB(Entscheidung.Testvariable_Z),...
                               2*(1-normcdfB(Entscheidung.Testvariable_Z)));
end

pWert= Entscheidung.pWert;
KonfInt = Entscheidung.Vertrauensintervall;
TEnt = Entscheidung.pWert < alpha; 

if TEnt
     Entscheidung.Testentscheidung = 'Die Nullhypothese wird abgelehnt!';
else
     Entscheidung.Testentscheidung = 'Die Nullhypothese wird nicht abgelehnt!';
end