function x = MW1cv(P,N)
%MW1CV   critical Mann-Whitney's U associated to a p-value. 
%It obtain a Mann-Whitney's U of two random variables with continuous cumulative
%distribution associated to a p-value. This procedure is highly recommended for sample sizes
%nx & ny <=7. For greater sample sizes it will takes a longer run-time. Alternatively, it is
%recommended to use the MW2cv function.
%It works with a procedure to get the nearest cumulative distribution relative value to P.
%
%   Syntax: function x = MW1cv(P,N) 
%      
%     Inputs:
%          P - cumulative probability value of interest.
%          N - 2-element vector of sample sizes for the two samples []. 
% The input quantities should be scalars.
%     Outputs:
%          x - Mann-Whitney's U statistic.
%
%    Example: For two independent samples we are interested to get the
%             Mann-Whitney's statistic U with an associated cumulative
%             probability P = 0.90. Sample sizes are n1 = 5 and n2 = 6.
%
%                              P = 0.90; N = [5,6];
%
%     Calling on Matlab the function: 
%             x = MW1cv(P,N)
%
%       Answer is:
%
%                 22    
%

%  Created by A. Trujillo-Ortiz and R. Hernandez-Walls
%             Facultad de Ciencias Marinas
%             Universidad Autonoma de Baja California
%             Apdo. Postal 453
%             Ensenada, Baja California
%             Mexico.
%             atrujo@uabc.mx
%
%  May 20, 2003.
%
%  To cite this file, this would be an appropriate format:
%  Trujillo-Ortiz, A. and R. Hernandez-Walls. (2003). MW1cv: Critical Mann-Whitney's U 
%    associated to a p-value: nx and ny <=7. A MATLAB file. [WWWdocument].URL http://
%    www.mathworks.com/matlabcentral/fileexchange/loadFile.do?objectId=3554&objectType=FILE
%
%  References:
% 
%  Mann, H. B. and Whitney, D. R. (1947), On a test of whether one of two   
%           random variables is stochastically larger than the other. Annals
%           of Mathematical Statistics, 18: 50-60.
%

if nargin <  2,
   error('Requires two input arguments.');
end
   
n = max(N);  %largest sample size.
m = min(N);  %smallest sample size.

if (n > 7) | (m > 7);
   fprintf('Warning: For nx or ny > 7, procedure will takes a bit long run-time.\n'); 
   fprintf('It is recommended to use the MW2cv function you can find on the\n');
   fprintf('Matlab>File Exchange Antonio Trujillo-Ortiz'' Author Page.\n');
   disp(' ');
   cont=input('Do you want to continue anyway (y/n):','s');
   
   if (cont=='y');
      disp('On your own choose...Please, wait.');
      
      for i = 1:n
         x(i) = 0;
      end;
      
      for j = 1:m
         y(j) = 1;
      end;
      
      V = [x y];
      
      N = sum(N);
      M = 2^N-1;
      
%Procedure to get the matrix of the total number of combinations among the O's and 1's:
%the rows are the N!/n!m! different ways (vectors) of arranging.
%Also, for each arrangement estimates the placements of x(i) [the number of y's less
%than x(i)].
      kk = 1:M;
      bip = dec2bin(kk,N);
      clear kk
      p = find((sum(double(bip'))-N*48) == n);
      Bip = bip(p,:);
      clear bip
      nB = length(Bip);
      v=[];
      for k = 1:nB
         w = Bip(k,:);
         ns = find(w == '1');
         nw = length(ns);
         summ = 0;
         for kk = 1:nw
            r = w(ns(kk)+1:end);
            nic = length(find(r == '0'));
            summ = summ+nic;
         end;
         v = [v;summ];
      end;
      
      Bip;  %matrix of the different ways of arranging.
      v;  %vector of the sum of placements of x(i).
      
      vo = sort(v);  %ordered vector of the sum of placements of x(i).
      
      %Frequency distribution.
      F = [];
      for i = 0:max(vo)
         f = sum(vo == i);
         F = [F,f];
      end;
      
      Fc = cumsum(F);  %cumulative frequency distribution.
      fc = Fc*1/sum(F);  %make distribution relative.
      
%Location of the interested Mann-Whitney's U on all the possible U's for this 2-sample sizes.
%Here we are using a procedure to get the nearest fc value to P.
      fc=fc-P;
      u = find(abs(fc)==min(abs(fc(:))));
      
      UU = [0:length(F)-1];  %vector of all the possible Mann-Whitney's U values.
      
%Association of the interested Mann-Whitney's U with its cumulative distribution.
      x = UU(u);  
   else
   end
else
   
   for i = 1:n
      x(i) = 0;
   end;
   
   for j = 1:m
      y(j) = 1;
   end;
   
   V = [x y];
   
   N = sum(N);
   M = 2^N-1;
   
%Procedure to get the matrix of the total number of combinations among the O's and 1's:
%the rows are the N!/n!m! different ways (vectors) of arranging.
%Also, for each arrangement estimates the placements of x(i) [the number of y's less
%than x(i)].
   kk = 1:M;
   bip = dec2bin(kk,N);
   clear kk
   p = find((sum(double(bip'))-N*48) == n);
   Bip = bip(p,:);
   clear bip
   nB = length(Bip);
   v=[];
   for k = 1:nB
      w = Bip(k,:);
      ns = find(w == '1');
      nw = length(ns);
      summ = 0;
      for kk = 1:nw
         r = w(ns(kk)+1:end);
         nic = length(find(r == '0'));
         summ = summ+nic;
      end;
      v = [v;summ];
   end;
   
   Bip;  %matrix of the different ways of arranging.
   v;  %vector of the sum of placements of x(i).
   
   vo = sort(v);  %ordered vector of the sum of placements of x(i).
   
%Frequency distribution.
   F = [];
   for i = 0:max(vo)
      f = sum(vo == i);
      F = [F,f];
   end;
   
   Fc = cumsum(F);  %cumulative frequency distribution.
   fc = Fc*1/sum(F);  %make distribution relative.
   
%Location of the interested Mann-Whitney's U on all the possible U's for this 2-sample sizes.
%Here we are using a procedure to get the nearest fc value to P.
   fc=fc-P;
   u = find(abs(fc)==min(abs(fc(:))));
   
   UU = [0:length(F)-1];  %vector of all the possible Mann-Whitney's U values.
   
%Association of the interested Mann-Whitney's U with its cumulative distribution.
   x = UU(u);
end
