function [pWerte,AnovTab,TestStat] = anova2B(X,MpZ)
%
% Funktion anova2B
%
% Aufruf: [pWerte,AnovTab,TestStat] = anova2B(X,MpZ)
%
% Diese Funktion realisiert eine ZWEIFAKTORIELLE VARIANZANALYSE
% fr einen BALANCIERTEN VERSUCHSPLAN nach Art der 
% STATISTICS-TOOLBOX-Funktion anova2.
%
% Die Funktion hat im Wesentlichen die gleiche Aufrufschnittstelle
% wie anova2 und kann als ERSATZFUNKTION eingesetzt werden.
% Die Funktion hat gegenber anova2 EINGESCHRNKTE FUNKTIONALITT.
%
% Eingangsdaten:	  X          Matrix der Daten
%                                Bei einer MATRIX wird von einem BALANCIERTEN
%                                Versuchsplan ausgegangen. Die Daten sind 
%                                SPALTENWEISE organisiert.
%                                Die Zeilen werden entsprechend dem Faktor
%                                MpZ (default=1, einfache Besetzung) zu 
%                                Faktoren mit MpZ Beobachtungen zusammengefasst.
%
%                     MpZ        Zahl der Beobachtungen pro Zeilenfaktor. 
%                                Die Zeilenzahl von X muss dann durch MpZ
%                                teilbar sein.
%                                 
%
% Ausgangsdaten:	  pWerte     die p-Werte
%                                  
%                     AnovTab    Cell-Array mit der ANOVA-Tabelle 
%    
%                     TestStat   Struktur mit der Teststatistik
%
% Falls anova12B mehr als zwei Eingangsparameter hat, werden diese ignoriert.
% Insbesondere erfolgt keine grafische Ergebnisausgabe, wie bei anova2.
%
% Es werden nur wenige Fehleingaben abgefangen! 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 1.01
% Datum: 1.9.2004



% Eingangsparameterliste checken

inputparams = nargin;

switch inputparams
    case 0
        error('anova2B muss mindestens einen Parameter haben! Abbruch!');
    case 1
        [m, n] = size(X);
        if m<2 | n<2
            error('Die Datenmatrix X muss mindestens zwei Zeilen und zwei Spalten haben! Abbruch!')
        else
            MpZ = 1;                    % default-Einstellung, einfache Besetzung
        end
    otherwise  
        [m, n] = size(X);
        if m<2 | n<2
            error('Die Datenmatrix X muss mindestens zwei Zeilen und zwei Spalten haben! Abbruch!')
        elseif MpZ<1 | abs(MpZ-floor(MpZ))>eps
            error('MpZ muss eine ganze zahl >=1 sein! Abbruch!');
        elseif  abs(m/MpZ-floor(m/MpZ))>eps  
            error('Die Zeilenzahl von X muss durch MpZ teilbar sein! Abbruch!');
        else
            % balancierter Versuch, mehrfache Besetzung
        end
end
    
% ANOVA-Tabelle vorinitialisieren

AnovTab = cell(6,6);
AnovTab{1,1} = 'Quelle'; AnovTab{1,2} = 'SS'; AnovTab{1,3} = 'df'; AnovTab{1,4} = 'MS'; AnovTab{1,5} = 'F'; AnovTab{1,6} = 'Prob>F';
AnovTab{2,1} = 'Spalten'; 
AnovTab{3,1} = 'Zeilen';
AnovTab{4,1} = 'Interaktion';
AnovTab{5,1} = 'Restfehler';
AnovTab{6,1} = 'Totalvariation';

% Teststatistik initialisieren

TestStat.RestfehlerVarianz = [];
TestStat.Spaltenmittel = [];
TestStat.WerteProSpaltenfaktor = m;
TestStat.Zeilenmittel = [];
ZF = m/MpZ;                    % Zahl der Zeilen-Faktoren berechnen
TestStat.WerteProZeilenfaktor = MpZ*n;
TestStat.pWertInteraktion = [];
TestStat.FreiheitsgradeRestfehler = [];

% Varianzanalyse berechnen

% Mittelwerte berechen

xiqq = mean(X);                % Mittel ber die Spalten
TestStat.Spaltenmittel = xiqq;

xjqqz = mean(X');              % Mittel ber die Zeilen
xjqqz = reshape(xjqqz,MpZ,ZF);
xjqqz = [xjqqz;zeros(size(xjqqz))];
xjqq = 2*mean(xjqqz);
TestStat.Zeilenmittel = xjqq;

Z = reshape(X,MpZ,ZF*n);       % Mittel ber die Zellen
Z = [Z; zeros(MpZ,ZF*n)];
Z = 2*mean(Z);
xijq = reshape(Z,ZF,n);

Z = X(:);                      % Gesamtmittel aller Werte
xqqq = mean(Z);

% Quadratsummen berechnen

QA = ZF*MpZ*sum((xiqq-xqqq).^2);     % Variation zwischen den Spalten
AnovTab{2,2} = QA;

QB = n*MpZ*sum((xjqq-xqqq).^2);      % Variation zwischen den Zeilen
AnovTab{3,2} = QB;


xiqqAB = repmat(xiqq, [ZF,1]);
xjqqAB = repmat(xjqq', [1,n]);
                                     % Wechselwirkungen
QAB = MpZ*sum(sum((xijq-xiqqAB-xjqqAB+xqqq).^2));    
AnovTab{4,2} = QAB;

if MpZ>1
                                     % Restvariation
    xijqR = repmat(xijq(:)', MpZ,1);
    xijqR = reshape(xijqR,size(X));
    QR = sum(sum(sum((X-xijqR).^2)));  
    AnovTab{5,2} = QR;
else
    AnovTab{4,2} = [];
    AnovTab{5,2} = QAB;
end
                                     % Gesamtsumme der Abweichungsquadrate
QT = sum(sum(sum((X-xqqq).^2)));    
AnovTab{6,2} = QT;


AnovTab{2,3} = n-1;                 % Freiheitsgrade
AnovTab{3,3} = ZF-1;
AnovTab{4,3} = (ZF-1)*(n-1);
AnovTab{6,3} = n*ZF*MpZ-1;
if MpZ>1
    AnovTab{5,3} = n*ZF*(MpZ-1);
    TestStat.FreiheitsgradeRestfehler = n*ZF*(MpZ-1);
else
    AnovTab{5,3} = AnovTab{4,3};
    AnovTab{4,3} = [];
    TestStat.FreiheitsgradeRestfehler = AnovTab{5,3};
end

                                     % Normierte Quadratsummen
MA = QA/(n-1); AnovTab{2,4} = MA;                 
MB = QB/(ZF-1); AnovTab{3,4} = MB;
MAB = QAB/((ZF-1)*(n-1)); AnovTab{4,4} = MAB;
if MpZ>1
    MR = QR/(n*ZF*(MpZ-1)); AnovTab{5,4} = MR;
else
    MR = MAB;
    AnovTab{5,4} = MR; 
    AnovTab{4,4} = []; 
end;
TestStat.RestfehlerVarianz = MR;


                                     % Testvariablenwerte
tA = MA/MR; AnovTab{2,5} = tA;                 
tB = MB/MR; AnovTab{3,5} = tB;
                                     % P-Werte
                                     
pWerte(1) = 1-fcdfB(tA, n-1, AnovTab{5,3});
AnovTab{2,6} = pWerte(1);    

pWerte(2) = 1-fcdfB(tB, ZF-1, AnovTab{5,3});
AnovTab{3,6} = pWerte(2);

if MpZ>1
    tAB = MAB/MR; AnovTab{4,5} = tAB;
    pWerte(3) = 1-fcdfB(tAB, (ZF-1)*(n-1), AnovTab{5,3});
    AnovTab{4,6} = pWerte(3);  
    TestStat.pWertInteraktion = pWerte(3);
else
    AnovTab{4,5} = [];
    AnovTab{4,6} = [];  
    TestStat.pWertInteraktion = [];
end