function [ydach, varargout] = glmvalB(beta,x,link,varargin)
% 
% Funktion glmvalB
%
% Aufruf: [ydach, varargout] = glmvalB(beta,x,link,P1,P2,...)
%
% Diese Funktion berechnet die Werte ydach eines VERALLGEMEINERTEN
% LINEAREN REGRESSIONSMODELLS
%
%         ydach = F^{-1}(<X,beta>) = F^{-1}(SUM(i=1:r  x_i*beta_i) + beta_0)
%
% Die Funktion hat im Wesentlichen die gleiche Aufrufschnittstelle
% wie glmval und kann als ERSATZFUNKTION eingesetzt werden.
% Die Funktion hat gegenber glmval jedoch eine SEHR EINGESCHRNKTE 
% FUNKTIONALITT, da lediglich das LOGIT- und das PROBIT-Modell berechnet 
% werden und auer dem Wert ydach keine weiteren statistischen Informationen
% zurckgeliefert werden.
%
%
% Eingangsdaten:	  beta       r-Koeffizientenvektor des Modells
%
%                     x          nxr-Matrix der unabhngigen Variablen
%
%                     link       Link-Funktion: hier sind nur die Werte 'logit'
%                                und 'probit' zugelassen.
%
%                     varargin   Werte die noch an glmvalB bergeben werden 
%                                knnen, aber ignoriert werden (wird lediglich 
%                                aus Kompatibilittsgrnden zu glmval bentigt)
%
% Ausgangsdaten:	  ydach      nx1-Vektor der berechneten Werte des Modells
%
%                     varargout  alle anderen Rckgabewerte werden ignoriert
%
% Es werden keine Fehleingaben abgefangen! 
%                                 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 1.01
% Datum: 11.9.2004

inargs = nargin;

if inargs < 3
    error('nlinfitB bentigt mindestens vier Argumente. Abbruch!');
end

[n,r] = size(x); 
if n==1      % Zeilenvektor zulassen
    X = X';  % Spaltenvektor daraus machen
    n = r; r = 1;
end

X = [ones(n,1),x];

[m,p] = size(beta); 
if m==1            % Zeilenvektor zulassen
    beta = beta';  % Spaltenvektor daraus machen
    m = p; p = 1;
end

if m~=r+1 
    error('beta muss soviel Elemente haben, wie x Spalten hat. Abbruch!');
end

for k=2:nargout
    varargin(k)={[]};
end


fall = [];

if strcmp(link,'logit')
    fall = 'logit';
elseif strcmp(link,'probit')
    fall = 'probit';
else
    error('Der Parameter link muss den Wert ''logit'' oder ''probit'' haben. Abbruch!');
end

if inargs > 4
    warning('Alle Eingabeparameter ab Nummer 5 werden igroriert!');
end

switch fall
    case 'logit'
        ydach = logitmodel(beta,X);
    
    case 'probit'
        ydach = probitmodel(beta,X);
end
  
    


% Subfunctions:
%----------------------------------------------------------------------------------------
% Lokale Funktionen mit dem Logit- und dem Probit-Modell

function [ydach] = logitmodel(betalf,X)

[n,r] = size(X); 
for j=1:n
    ydach(j) = 0;
    for k=1:r
        ydach(j) = ydach(j) + betalf(k)*X(j,k);
    end

    ydach(j) = 1/(1+exp(-ydach(j)));
end

ydach = ydach(:);


function [ydach] = probitmodel(betalf,X)

[n,r] = size(X); 
for j=1:n
    ydach(j) = 0;
    for k=1:r
        ydach(j) = ydach(j) + betalf(k)*X(j,k);
    end

    ydach(j) = normcdfB(ydach(j));
end

ydach = ydach(:);
