function [mu,sigma,VImu,VIsigma] = normfitB(daten, alpha)
%
% Funktion normfitB
%
% Aufruf: [mu,sigma,VImu,VIsigma] = normfitB(daten, alpha)
%
% Diese Funktion bestimmt fr eine Stichdaten daten einer NORMALVERTEILTEN
% Grundgesamtheit die erwartungstreuen Schtzungen fr die Parameter
% mu und sigma sowie die zugehrigen VERTRAUENSINTERVALLE (bei unbekannter
% varianz).
% Man vergleiche hierzu Abschnitt 5.3.2 und insbesondere die Datei
% VertIntvNV.m der Begleitsoftware!
%
%
% Eingangsdaten:	  daten      Stichdaten (Vektor von Werten)
%        
%                     alpha      Signifikanzniveau fr die Vertrauensintervalle
%                                (default ist 0.05)
%
% Ausgangsdaten:	  mu,sigma   Schtzung der Parameter
%
%                     VImu       Vertrauensintervall fr mu
%
%                     VIsigma    Vertrauensintervall fr sigma
%
% Es werden i.W. keine Fehleingaben abgefangen! 
%                                 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 1.01
% Datum: 5.9.2004

nargs = nargin;

if nargs == 1      
    alpha = 0.05;
end

if sum(size(daten)>[1,1])>=2   % daten kein Vektor
    error('normfitB kann fr daten nur vektorielle Argumente verarbeiten. Abbruch!');
end

% Bestimmung des Stichdatennumfangs n
n = length(daten);

% Bestimmung des empirischen Mittels xquer
xquer = mean(daten); mu = xquer;

% Bestimmung der empirischen Varianz sq und der empirischen
% Streuung s
sq = var(daten);
s = sqrt(sq); sigma = s;

% Bestimmung des zweiseitigen (1-alpha/2)-Quantils
% (einseitigen (1-alpha/2)-Quantils) der t-Verteilung
% mit n-1 Freiheitsgraden
ugammatilde = tinvB(1-alpha/2, n-1);

% Untere Vertrauensintervallgrenze bei unbekannter Varianz
cutilde = xquer - ugammatilde*(s/sqrt(n));

% Obere Vertrauensintervallgrenze bei unbekannter Varianz
cotilde = xquer + ugammatilde*(s/sqrt(n));

VImu = [cutilde,cotilde];

% Bestimmung des einseitigen (alpha/2)-Quantils
% der Chi-Quadrat-Verteilung mit n-1 Freiheitsgraden
ug2u = chi2invB(alpha/2, n-1);

% Bestimmung des einseitigen (1-alpha/2)-Quantils
% der Chi-Quadrat-Verteilung mit n-1 Freiheitsgraden
ug2o = chi2invB(1-alpha/2, n-1);

% Untere Vertrauensintervallgrenze fr die Varianz
cuhat = (n-1)*sq/ug2o;

% Obere Vertrauensintervallgrenze fr die Varianz
cohat = (n-1)*sq/ug2u;

VIsigma = [cuhat,cohat];

