function [pWert, H, TestErgebnis] = ranksumB(x,y,alpha)
%
% Funktion ranksumB
%
% Aufruf: [pWert, H, TestErgebnis] = ranksumB(X)
%
% Diese Funktion realisiert einen WILCOXON'schen RANGSUMMEN-TEST 
% (U-TEST) nach Art der STATISTICS-TOOLBOX-Funktion ranksum.
%
% Die Funktion hat im Wesentlichen die gleiche Aufrufschnittstelle
% wie ranksum und kann als ERSATZFUNKTION eingesetzt werden.
% Im Gegensatz zu ranksum wird das Testergebnis in Form der Struktur
% TestErgebnis zurckgeliefert. Die Funktion hat gegenber der
% Statistics-Toolbox-Funktion ranksum EINGESCHRNKTE FUNKTIONALITT.
%
% Eingangsdaten:	  x              erste Stichprobe
%                     y              zweite Stichprobe  (Vektoren)
%
%                     alpha          Signifikanzniveau
%                                    (default = 0.05)
%
% Ausgangsdaten:	  pWert           der p-Wert
%
%                     H              die Testentscheidung
%
%                     TestErgebnis   Struktur mit dem Ergebnis des Tests
%
% Es werden i.W. keine Fehleingaben abgefangen! 
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 2.01
% Datum: 6.9.2004/14.9.2004

nargs = nargin;

if sum(size(x)>[1,1])>=2 | sum(size(y)>[1,1])>=2  % x-daten oder y-daten kein Vektor
    error('ranksumB kann vektorielle Argumente verarbeiten. Abbruch!');
end

n = length(x); m = length(y);

if ~(n>=4 & m>=4 & n+m>=20)
    warnung('Stichproben sehr klein.ranksumB liefert nur ungenaue Werte!');
end

if nargs<=2
    alpha = 0.05;    % default-Niveau
end

% Ausgabestruktur vorinitialisieren

TestErgebnis.untererKritischerWert = [];
TestErgebnis.obererKritischerWert = [];
TestErgebnis.U = [];
TestErgebnis.pWert = [];
TestErgebnis.TestEntscheidung = [];

% Punkte in einem Vektor hintereinander schreiben und sortieren

pkte = [x, y];
ranks = cumsum(ones(1,n+m));       % vorkommende Rnge
[psort, indx] = sort(pkte);        % aufsteigend sortierte Punkte

% Feststellen, wo die sortierten Werte sich ndern bzw.
% wo gleiche Werte auftreten

chng = sign([1, diff(psort)]); 

% zugehrige Rangziffern bestimmen

chgranks = ranks(logical(chng));
z = [ranks(logical(chng)), ranks(end)+1];   % Hilfsgre

% Durchschnittsrnge berechnen

dranks =(1/2)*(1./(z(2:end)-z(1:end-1))).*...
    ( z(2:end).*(z(2:end)-1) - z(1:end-1).*(z(1:end-1)-1));

% Rnge in ranks durch Durchschnittsrnge ersetzen

xdranks = [];
for k=2:length(z)
    xdranks = [xdranks, repmat(dranks(k-1), 1, z(k)-z(k-1))];
end;

% Zuordnung der Durchschnittsrnge zu den Vektoren x und y

ranksX = xdranks(indx<=n);
ranksY = xdranks(indx>n);

% Berechnung der Rangsummen

xrankSum = sum(ranksX);
yrankSum = sum(ranksY);

% Bestimmung des Werts der Testvariablen U

U = xrankSum -n*(n+1)/2;
TestErgebnis.U = U;

% Zahl der Bindungen feststellen und Korrektursummand
% berechnen

rgdiff  = z(2:end)-z(1:end-1);
bindx = rgdiff>1;               % Index der Rangdifferenzen > 1
bindungen = rgdiff(bindx);      % Bindungen
K = length(bindungen);
                                % Korrektursummand bei Bindungen
korrekt = (sum(bindungen.^3)-sum(bindungen))/K;


% Bestimmung des kritischen Wertes
% (mit einseitigem alpha/2-Quantil der Normalverteilung
% bestimmen)

mu = n*m/2;
sigma = sqrt(n*m*(n+m+1)/12 - korrekt);
                                % Quantil der Standard-Normalverteilung
u_quantil =  norminvB(1-alpha/2);

k = floor( mu - u_quantil*sigma );

TestErgebnis.untererKritischerWert = n*m-k;
TestErgebnis.obererKritischerWert = k;

% Testentscheidung (1, falls Ablehnung, 0 sonst)

H = ~((U>k) & (U<n*m-k)); 
if H
    TestErgebnis.TestEntscheidung = 'Die Nullhypothese wird abgelehnt!';
else
    TestErgebnis.TestEntscheidung = 'Die Nullhypothese wird nicht abgelehnt!';
end

TestErgebnis.pWert = normcdfB((k-mu)/sigma)+1-normcdfB((m*n-k-mu)/sigma);
pWert = TestErgebnis.pWert;
