function [outliers, sbar, UEG, OEG] = schartD(data,conf,specs)
%
% Funktion schartD
%
% Aufruf: [outliers, sbar, UEG, OEG] = schartD(data,conf,specs)
%
% Modifizierte Version der Statistics Toolbox Funktion
% schart.
%
%   ------ Hilfe der Originalversion----------------------------------
%   SCHART S chart for monitoring the standard deviation.
%   SCHART(DATA,CONF,SPECS) produces an S chart of
%   the grouped responses in DATA. The rows of DATA contain
%   replicate observations taken at a given time. The rows
%   must be in time order.
%
%   CONF (optional) is the confidence level of the upper and 
%   lower plotted confidence limits. CONF is 0.9973 by default.
%   This means that 99.73% of the plotted points should fall 
%   between the control limits.
%
%   SPECS (optional) is a two element vector for the lower and
%   upper specification limits of the response.
%
%   OUTLIERS = SCHART(DATA,CONF,SPECS) returns  a vector of 
%   indices to the rows where the standard deviation of DATA is 
%   out of control.
%
%   [OUTLIERS, H] = SCHART(DATA,CONF,SPECS) also returns a vector
%   of handles, H, to the plotted lines.

%   Reference: Montgomery, Douglas, Introduction to Statistical
%   Quality Control, John Wiley & Sons 1991 p. 235.

%   B.A. Jones 2-13-95
%   Copyright 1993-2000 The MathWorks, Inc. 
%   $Revision: 2.10 $  $Date: 2000/05/26 18:53:41 $
%   ------------------------------------------------------------------
%
%   Modifikationen:
%
%   Eingangsparameter:       conf      Zahl oder 2-komponentiger Vektor
%
%                                      Ist conf eine Zahl, so entspricht seine
%                                      Bedeutung der URSPRNGLICHEN schart-
%                                      Version. Die Funktion kann als VORLAUF-
%                                      Funktion zur Ermittlung von OEG, UEG
%                                      verwendet werden.
%
%                                      Ist conf = [s, DeltaS] ein Vektor, so sind
%                                      die Komponenten der vorgegebene Streuwert
%                                      und der Abstand zu den Eingriffsgrenzen. 
%                                      Die Funktion kann als KARTEN-
%                                      Funktion verwendet werden.
%
%  Ausgangsparameter:        sbar      Schtzung von sigma (ermittelt oder vorgegeben)
%
%                            UEG, OEG  Eingriffsgrenzen (Kontrollgrenzen)
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 1.01
% Datum: 4.2.2004



if nargin < 2
   conf = 0.9973;
end

if isempty(conf)
  conf = 0.9973;
end

if all(size(conf) ==[1,1]) 
    version = 'Vorlauf';
elseif all(size(conf) ==[1,2])
    version = 'Karte'; 
else
    error('Parameter conf muss Zahl oder 1x2-Vektor sein! Abbruch!');
end;


ciprob = 1-(1-conf)/2;

[m,n] = size(data);
xbar  = mean(data')';
s     = (std(data'))';

if strcmp(version, 'Vorlauf')
    sbar   = mean(s);

    c4 = sqrt(2/(n-1)).*gamma(n/2)./gamma((n-1)/2);
    cicrit = norminvB(ciprob);
    b3 = 1 - cicrit*sqrt(1-c4*c4)/c4;
    b4 = 1 + cicrit*sqrt(1-c4*c4)/c4;

    UEG = b3*sbar;
    if UEG < 0, UEG = 0; end
    OEG = b4*sbar;
else
    sbar  = conf(1);
    OEG = sbar + conf(2);
    UEG = sbar - conf(2);
end;

tmp = NaN;
incontrol = tmp(1,ones(1,m));
outcontrol = incontrol;

greenpts = find(s > UEG & s < OEG);
redpts = find(s <= UEG | s >= OEG);

incontrol(greenpts) = s(greenpts);
outcontrol(redpts) = s(redpts);

samples = (1:m);

hh  = plot(samples,s,samples,OEG(ones(m,1),:),'r-',samples,sbar(ones(m,1),:),'g-',...
           samples,UEG(ones(m,1),:),'r-',samples,incontrol,'b+',...
         samples,outcontrol,'r+');

if any(redpts)
  for k = 1:length(redpts)
     text(redpts(k) + 0.5,outcontrol(redpts(k)),num2str(redpts(k)));
  end
end

t1 = text(m+0.5,OEG,'OEG');
t2 = text(m+0.5,UEG,'UEG');
text(m+0.5,sbar,'s');
         
if nargin == 3
   set(gca,'NextPlot','add');
   TolU = specs(1);
   TolO = specs(2);
   t3 = text(m + 0.5,TolO,'TolO');
   t4 = text(m + 0.5,TolU,'TolU');
   hh1 = plot(samples,TolU(ones(m,1),:),'g-',samples,TolO(ones(m,1),:),'g-');
   set(gca,'NextPlot','replace');
   hh = [hh; hh1];
end

if nargout > 0
  outliers = redpts;
end

if nargout == 2
 h = hh;
end         

set(hh([3 5 6]),'LineWidth',2);
xlabel('Stichproben');
ylabel('Messungen');
