function [outliers, avg, s, UEG, OEG] = xbarplotD(data,conf,specs,sigmaest)
%
% Funktion xbarplotD
%
% Aufruf: [outliers, avg, s, UEG, OEG] = xbarplotD(data,conf,specs,sigmaest)
%
% Modifizierte Version der Statistics Toolbox Funktion
% xbarplot.
%
%   ------ Hilfe der Originalversion----------------------------------
%   XBARPLOT X-bar chart for monitoring the mean.
%   XBARPLOT(DATA,CONF,SPECS,SIGMAEST) produces an xbar chart of
%   the grouped responses in DATA. The rows of DATA contain
%   replicate observations taken at a given time. The rows
%   should be in time order.
%
%   CONF (optional) is the confidence level of the upper and 
%   lower plotted confidence limits. CONF is 0.9973 by default.
%   This means that 99.73% of the plotted points should fall 
%   between the control limits if the process is in control.
%
%   SPECS (optional) is a two element vector for the lower and
%   upper specification limits of the response.
%
%   SIGMAEST (optional) specifies how XBARPLOT should estimate
%   sigma.  Possible values are 'std' (the default) to use the
%   average within-subgroup standard deviation, 'range' to use the
%   average subgroup range, and 'variance' to use the square root
%   of the pooled variance.
%
%   OUTLIERS = XBARPLOT(DATA,CONF,SPECS,SIGMAEST) returns a vector
%   of indices to the rows where the mean of DATA is out of
%   control.
%
%   [OUTLIERS, H] = XBARPLOT(DATA,CONF,SPECS,SIGMAEST) also returns
%   a vector of handles, H, to the plotted lines.
%
%   B.A. Jones 9/30/94
%   Modified 4/5/99 by Tom Lane
%   Copyright 1993-2000 The MathWorks, Inc. 
%   $Revision: 2.10 $  $Date: 2000/05/26 18:53:58 $
%   ------------------------------------------------------------------
%
%   Modifikationen:
%
%   Eingangsparameter:       conf      Zahl oder 2-komponentiger Vektor
%
%                                      Ist conf eine Zahl, so entspricht seine
%                                      Bedeutung der URSPRNGLICHEN xbarplot-
%                                      Version. Die Funktion kann als VORLAUF-
%                                      Funktion zur Ermittlung von OEG, UEG
%                                      verwendet werden.
%
%                                      Ist conf = [Mittel, DeltaKG] ein Vektor, so sind
%                                      die Komponenten der vorgegebene Mittelwert
%                                      (etwa der Sollwert) und
%                                      der Abstand zu den Eingriffsgrenzen. 
%                                      Die Funktion kann als KARTEN-
%                                      Funktion verwendet werden.
%
%  Ausgangsparameter:        avg       empirischer Mittelwert
%   
%                            s         Schtzung von sigma, je nach Verfahren
%
%                            UEG, OEG  Eingriffsgrenzen (Kontrollgrenzen)
%
%
% Autor: Prof. Dr. Ottmar Beucher
%        FH Karlsruhe
% Version: 2.01
% Datum: 10.2.2004
 

if nargin < 2
   conf = 0.9973;
end

if isempty(conf)
  conf = 0.9973;
end


if all(size(conf) ==[1,1]) 
    version = 'Vorlauf';
elseif all(size(conf) ==[1,2])
    version = 'Karte'; 
else
    error('Parameter conf muss Zahl oder 1x2-Vektor sein! Abbruch!');
end;

urwertkarte = 0;      % default-Einstellung: keine Urwertkarte

[m,n] = size(data);
xbar  = mean(data')';
avg   = mean(xbar);
if (n < 2)
   error('data muss mehr als 2 Spalten haben! Abbruch!');
end

% Need a sigma estimate to compute control limits
if (nargin < 4)
   sigmaest = 's';
elseif ((strcmp(sigmaest,'range') | strcmp(sigmaest,'r')) & (n>25))
   error(['XBARPLOT cannot use a range estimate if subgroups have' ...
          ' more than 25 observations.']);
end
if (strcmp(sigmaest,'variance') | strcmp(sigmaest,'v'))  % use pooled variance
   s = sqrt(sum(sum(((data - xbar(:,ones(n,1))).^2)))./(m*(n-1)));
elseif (strcmp(sigmaest,'range') | strcmp(sigmaest,'r'))  % use average range
   r = (range(data'))';
   d2 = [0.000 1.128 1.693 2.059 2.326 2.534 2.704 2.847 2.970 3.078 ...
         3.173 3.258 3.336 3.407 3.472 3.532 3.588 3.640 3.689 3.735 ...
         3.778 3.819 3.858 3.895 3.931];
   s = mean(r ./ d2(n));
else                                 % estimate sigma using average s
   svec = (std(data'))';
   c4 = sqrt(2/(n-1)).*gamma(n/2)./gamma((n-1)/2);
   s = mean(svec ./ c4);
end

smult = norminvB(1-.5*(1-conf));

if strcmp(version, 'Vorlauf')
    OEG = avg + smult*s./sqrt(n);
    UEG = avg - smult*s./sqrt(n);
    MW = avg;
else
    MW  = conf(1);
    OEG = MW + conf(2);
    UEG = MW - conf(2);
end;

tmp = NaN;
incontrol = tmp(1,ones(1,m));
outcontrol = incontrol;

greenpts = find(xbar > UEG & xbar < OEG);
redpts = find(xbar <= UEG | xbar >= OEG);

incontrol(greenpts) = xbar(greenpts);
outcontrol(redpts) = xbar(redpts);

samples = (1:m);

hh  = plot(samples,xbar,samples,OEG(ones(m,1),:),'r-',samples,MW(ones(m,1),:),'g-',...
           samples,UEG(ones(m,1),:),'r-',samples,incontrol,'b+',...
         samples,outcontrol,'r+');

if any(redpts)
  for k = 1:length(redpts)
     text(redpts(k) + 0.5,outcontrol(redpts(k)),num2str(redpts(k)));
  end
end

text(m+0.5,OEG,'OEG');
text(m+0.5,UEG,'UEG');
text(m+0.5,MW,'MW');
         
if nargin >= 3
   set(gca,'NextPlot','add');
   TolU = specs(1);
   TolO = specs(2);
   t3 = text(m + 0.5,TolO,'TolO');
   t4 = text(m + 0.5,TolU,'TolU');
   hh1 = plot(samples,TolU(ones(m,1),:),'g-',samples,TolO(ones(m,1),:),'g-');
   set(gca,'NextPlot','replace');
   hh = [hh; hh1];
end

if nargout > 0
  outliers = redpts;
end

if nargout == 2
 h = hh;
end         

set(hh([3 5 6]),'LineWidth',2);
xlabel('Stichproben');
ylabel('Messungen');
