/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef TRANSFORMATION_E1_H
#define TRANSFORMATION_E1_H

#include <cassert>
#include <cmath>
using std::cos;
using std::sin;

#include "point_e1.h"
#include "segment_e1.h"
#include "vector_e1.h"

#include "../geometry_lib/transformations.h"

extern Identity IDENTITY;
extern Scale SCALE;
extern Translation TRANSLATION;

template<typename T>
class Transformation_E1
{
protected:
    T m00, m01;

    // m10 = 0 , m11 = 1
    void setToIdentity();
public:    
    Transformation_E1();
    Transformation_E1(const Identity& t);
    Transformation_E1(const Scale& s, const Vector_E1<T>& v);
    Transformation_E1(const Translation& t, const Vector_E1<T>& v);

    Transformation_E1(const T& m00, const T& m01)
	: m00(m00), m01(m01)
    {}

    T get_m00() { return m00; }
    T get_m01() { return m01; }

    Transformation_E1 inverse() const;

    Point_E1<T>  transform(const Point_E1<T>& p) const;
    Vector_E1<T>  transform(const Vector_E1<T>& p) const;
    Segment_E1<T> transform(const Segment_E1<T>& s) const;
    Point_E1<T>  operator()(const Point_E1<T>& p) const;
    Vector_E1<T>  operator()(const Vector_E1<T>& p) const;
    Segment_E1<T>  operator()(const Segment_E1<T>& p) const;

    Transformation_E1<T> operator*(const Transformation_E1<T>& TR) const;
};

template<typename T>
void
Transformation_E1<T>::setToIdentity()
{
    m00 = 1;
    m01 = 0;
}
template<typename T>
Transformation_E1<T>::Transformation_E1()
{
    setToIdentity();
}

template<typename T>
Transformation_E1<T>::Transformation_E1(const Identity& t)
{
    setToIdentity();
}

template<typename T>
Transformation_E1<T>::Transformation_E1(const Scale& s, const Vector_E1<T>& v)
{
    setToIdentity();
    m00 = v.x();
}

template<typename T>
Transformation_E1<T>::Transformation_E1(const Translation& t, const Vector_E1<T>& v)
{
    setToIdentity();
    m01 = v.x();
}

template<typename T>
Transformation_E1<T>
Transformation_E1<T>::inverse() const
{
    T unity(1);
    T inv = unity/m00;
    return Transformation_E1<T>(inv, - m01 * inv);
}

template<typename T>
Point_E1<T>
Transformation_E1<T>::transform(const Point_E1<T>& p) const
{
    return Point_E1<T>(m00 * p.x() + m01);
}

template<typename T>
Vector_E1<T>
Transformation_E1<T>::transform(const Vector_E1<T>& v) const
{
    return Vector_E1<T>(m00 * v.x());
}

template<typename T>
Segment_E1<T>
Transformation_E1<T>::transform(const Segment_E1<T>& seg) const
{
    Point_E1<T> s = transform(seg.source());
    Point_E1<T> t = transform(seg.target());

    return Segment_E1<T>(s,t);
}

template<typename T>
Point_E1<T>
Transformation_E1<T>::operator()(const Point_E1<T>& p) const
{
    return transform(p);
}

template<typename T>
Vector_E1<T>
Transformation_E1<T>::operator()(const Vector_E1<T>& v) const
{
    return transform(v);
}

template<typename T>
Segment_E1<T>
Transformation_E1<T>::operator()(const Segment_E1<T>& s) const
{
    return transform(s);
}

template<typename T>
Transformation_E1<T>
Transformation_E1<T>::operator*(const Transformation_E1<T>& TR) const
{
    // Notice the order of composition:
    //      *this is applied second and TR is applied first
    return Transformation_E1<T>(m00 * TR.m00 ,
				m00 * TR.m01 + m01);
}

#endif // TRANSFORMATION_E1_H
