/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef BBOX_E2_H
#define BBOX_E2_H

#include "point_e2.h"
#include "line_e2.h"

template<typename T>
class Bbox_E2
{
    Point_E2<T> _LL;
    Point_E2<T> _UR;
public:
    Bbox_E2() : _LL(), _UR() {}

    Bbox_E2(const Point_E2<T>& p) : _LL(p), _UR(p) {}

    Bbox_E2(const Point_E2<T>& pLL, const Point_E2<T>& pUR)
	: _LL(pLL), _UR(pUR) {}

    T get_width()  const { return _UR.x() - _LL.x(); }
    T get_height() const { return _UR.y() - _LL.y(); }

    Bbox_E2<T>&
    operator+=(const Point_E2<T>& p)
    {
	T xll = _LL.x();
	T yll = _LL.y();
	T xur = _UR.x();
	T yur = _UR.y();
	if(p.x() < xll)
	    xll = p.x();
	if(p.y() < yll)
	    yll = p.y();
	if(p.x() > xur)
	    xur = p.x();
	if(p.y() > yur)
	    yur = p.y();

	_LL = Point_E2<T>(xll,yll);
	_UR = Point_E2<T>(xur,yur);

	return *this;
    }

    bool operator==(const Bbox_E2<T>& box) const
    {
	return box._LL == _LL && box._UR == _UR;
    }

    Point_E2<T> LL() const { return _LL; }
    Point_E2<T> UR() const { return _UR; }

    Point_E2<T> center() const { return Point_E2<T>((_LL.x() + _UR.x())/2.0,
						    (_LL.y() + _UR.y())/2.0); }

    // Bbox is to the left of the four oriented lines
    Line_E2<T> getEastLine() const
    {
	return Line_E2<T>(Point_E2<T>(_UR.x(), _LL.y()),
			  _UR);
    }
    Line_E2<T> getNorthLine() const
    {
	return Line_E2<T>(_UR,
			  Point_E2<T>(_LL.x(), _UR.y()));
    }
    Line_E2<T> getWestLine() const
    {
	return Line_E2<T>(Point_E2<T>(_LL.x(), _UR.y()),
			  _LL);
    }
    Line_E2<T> getSouthLine() const
    {
	return Line_E2<T>(_LL,
			  Point_E2<T>(_UR.x(), _LL.y()));
    }
    
};

template<class S>
std::ostream&
operator<<(std::ostream& os, const Bbox_E2<S>& B)
{
    return os << B.LL() << " --- " << B.UR();
}

#endif // BBOX_E2_H
