/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef DIRECTION_E2_H
#define DIRECTION_E2_H

#include <cmath>

#include "../misc/determinant.h"
#include "vector_e2.h"

template<typename T>
class Direction_E2
{
    T _x, _y;

public:
    Direction_E2() : _x(1), _y(0) {}
    Direction_E2(const T& x, const T& y) : _x(x), _y(y) {}
    Direction_E2(const Vector_E2<T>& v) : _x(v.x()), _y(v.y()) {}

    T x() const { return _x; }
    T y() const { return _y; }

    Vector_E2<T> getVector_E2() const {
	return Vector_E2<T>(_x,_y);
    }

    bool
    operator==(const Direction_E2<T>& d) const {
	return (this == &d) ||
	    determinant(_x, _y, d._x, d._y) == 0 &&
	    inner_product(_x, _y, d._x, d._y) > 0;
    }

    Direction_E2<T>
    operator-() const {
	return Direction_E2<T>(- _x, - _y);
    }
};

template<typename T>
Point_E2<T>
operator+(const Point_E2<T>& p, const Direction_E2<T>& d)
{
    return Point_E2<T>(p.x() + d.x(), p.y() + d.y());
}

template<typename T>
T
dot_product(const Direction_E2<T>& d1, const Direction_E2<T>& d2)
{
    return d1.x() * d2.x() + d1.y() * d2.y();
}

template<typename T>
T
cross_product(const Direction_E2<T>& d1, const Direction_E2<T>& d2)
{
    return determinant(d1.x(), d1.y(), d2.x(), d2.y());
}

template<typename T>
std::ostream&
operator<<(std::ostream& os, const Direction_E2<T>& D)
{
    return os << D.x() << " " << D.y();
}

#endif // DIRECTION_E2_H
