/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef LINE_E2_H
#define LINE_E2_H

#include "point_e2.h"
#include "segment_e2.h"
#include "../misc/determinant.h"

template<typename T>
class Line_E2;

template<typename T>
ostream& operator<<(ostream& os, const Line_E2<T>& seg);

template<typename T>
class Line_E2
{
private:
    T _a, _b, _c;		// ax+by+c=0

    void init_abc(const Point_E2<T>& src, const Point_E2<T>& tgt)
    {
	_a = + determinant(src.y(), T(1.0), tgt.y(), T(1.0));
	_b = - determinant(src.x(), T(1.0), tgt.x(), T(1.0));
	_c = + determinant(src.x(), src.y(), tgt.x(), tgt.y());
    }
public:
    Line_E2() : _a(0), _b(1), _c(0) {}
    Line_E2(const Point_E2<T>& src, const Point_E2<T>& tgt)
    {
	init_abc(src, tgt);
    }
    Line_E2(const Segment_E2<T>& seg)
    {
	init_abc(seg.source(), seg.target());
    }

    T a() const { return _a; }
    T b() const { return _b; }
    T c() const { return _c; }

    bool operator==(const Line_E2<T>& L) const {
	return (this == &L) ||
	    are_dependent(_a, _b, _c, L._a, L._b, L._c) &&
	    inner_product(_a, _b, _c, L._a, L._b, L._c) > 0;
    }

    bool operator!=(const Line_E2<T>& L) const {
	return ! operator==(L);
    }

    T slope() const {
	return _b / _a;
    }

    bool is_parallel(const Line_E2<T>& L) const
    {
	return slope() == L.slope();
    }

    friend ostream& operator<< <>(ostream& os, const Line_E2<T>& seg);
};

template<typename T>
ostream& operator<<(ostream& os, const Line_E2<T>& seg)
{
    return os << seg._a << " " << seg._b << " " << seg._c;
}

#endif // LINE_E2_H
