/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef PREDICATES_E2_H
#define PREDICATES_E2_H

#include <iostream>
using namespace std;

#include "point_e2.h"
#include "segment_e2.h"
#include "line_e2.h"
#include "polygon_e2.h"

#include "../misc/orientation.h"
#include "../misc/determinant.h"
#include "../misc/dominant.h"

template<typename T>
Oriented_side
oriented_side(const Point_E2<T>& p1,
	      const Point_E2<T>& p2,
	      const Point_E2<T>& p3)
{
    T unity(static_cast<T>(1));
    T d = determinant(p1.x(), p1.y(), unity,
		      p2.x(), p2.y(), unity,
		      p3.x(), p3.y(), unity);

    return enum_Oriented_side(d);
}

template<typename T>
Oriented_side
oriented_side(const Line_E2<T>& L,
	      const Point_E2<T>& p)
{
    T d = L.a() * p.x() + L.b() * p.y() + L.c();

    return enum_Oriented_side(d);
}

template<typename T>
Oriented_side
oriented_side(const Segment_E2<T>& seg,
	      const Point_E2<T>& p)
{
    return oriented_side(seg.source(), seg.target(), p);
}

template<typename T>
bool
are_linearly_dependent(const Vector_E2<T>& v1,
		       const Vector_E2<T>& v2)
{
    return (determinant(v1.x(), v1.y(),
			v2.x(), v2.y()) == T(0));
}

template<typename T>
bool
are_parallel(const Segment_E2<T>& seg1,
	     const Segment_E2<T>& seg2)
{
    const Vector_E2<T> v1 = seg1.get_Vector_E2();
    const Vector_E2<T> v2 = seg2.get_Vector_E2();
    return are_linearly_dependent(v1, v2);
}

template<typename T>
Orientation_to_circle
inside_circle(const Point_E2<T>& p0,
	      const Point_E2<T>& p1,
	      const Point_E2<T>& p2,
	      const Point_E2<T>& p3)
{
    T unity(static_cast<T>(1));
    T d1 = determinant(p0.x() * p0.x(), p0.x(), p0.y(), unity,
		       p1.x() * p1.x(), p1.x(), p1.y(), unity,
		       p2.x() * p2.x(), p2.x(), p2.y(), unity,
		       p3.x() * p3.x(), p3.x(), p3.y(), unity);
    T d2 = determinant(p1.x(), p1.y(), unity, 
		       p2.x(), p2.y(), unity, 
		       p3.x(), p3.y(), unity);
    T d = d1 * d2;

    if(d > 0)
	return INSIDE_CIRCLE;
    else if(d < 0)
	return OUTSIDE_CIRCLE;
    else
	return COCIRCULAR;
}

template<typename T>
bool
is_inside(const Polygon_E2<T>& polygon,
	  const Point_E2<T>& point)
{
    Point_E2<T> inf(99999, point.y());
    int count = 0;
    for(unsigned int i=0; i<polygon.vertices.size(); ++i) {
	Point_E2<T> source(polygon.vertices[i]);
	Point_E2<T> target(polygon.vertices[(i==polygon.vertices.size()-1) ? 0 : i+1]);

	Oriented_side o1 = oriented_side(source, target, point);
	Oriented_side o2 = oriented_side(source, target, inf);

	Oriented_side o3 = oriented_side(point, inf, source);	    
	Oriented_side o4 = oriented_side(point, inf, target);	    

	if(o1 * o2 == -1 && o3 * o4 == -1)
	    count++;
    }
    return (count%2 == 1);
}

//----------------------------------------------------------------
template<typename T>
Dominant_E2
dominant(const Segment_E2<T>& segment)
{
    T dx = segment.target().x() - segment.source().x();
    T dy = segment.target().y() - segment.source().y();

    return dominant(dx, dy);
}
//----------------------------------------------------------------

#endif // PREDICATES_E2_H
