/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef TRANSFORMATION_E2_H
#define TRANSFORMATION_E2_H

#include <cassert>
#include <cmath>
using std::cos;
using std::sin;

#include "point_e2.h"
#include "segment_e2.h"
#include "vector_e2.h"
#include "line_e2.h"

#include "../geometry_lib/transformations.h"

extern Identity IDENTITY;
extern Scale SCALE;
extern Rotation ROTATION;
extern Translation TRANSLATION;

template<typename T>
class Transformation_E2
{
protected:
    T m00, m01, m02;
    T m10, m11, m12;

    // m20 = 0 , m21 = 0 , m22 = 1
    void setToIdentity();
public:    
    Transformation_E2();
    Transformation_E2(const Identity& t);
    Transformation_E2(const Scale& s, const Vector_E2<T>& v);
    Transformation_E2(const Translation& t, const Vector_E2<T>& v);
    Transformation_E2(const Rotation& r, const T& angle);

    Transformation_E2(const T& m00, const T& m01,
		      const T& m10, const T& m11)
	: m00(m00), m01(m01), m02(0),
	  m10(m10), m11(m11), m12(0)
    {}
    Transformation_E2(const T& m00, const T& m01, const T& m02,
		      const T& m10, const T& m11, const T& m12)
	: m00(m00), m01(m01), m02(m02),
	  m10(m10), m11(m11), m12(m12)
    {}

    T get_m00() { return m00; }
    T get_m01() { return m01; }
    T get_m02() { return m02; }
    T get_m10() { return m10; }
    T get_m11() { return m11; }
    T get_m12() { return m12; }

    Point_E2<T>  transform(const Point_E2<T>& p) const;
    Line_E2<T>  transform(const Line_E2<T>& l) const;
    Vector_E2<T>  transform(const Vector_E2<T>& p) const;
    Segment_E2<T> transform(const Segment_E2<T>& s) const;
    Point_E2<T>  operator()(const Point_E2<T>& p) const;
    Line_E2<T>  operator()(const Line_E2<T>& l) const;
    Vector_E2<T>  operator()(const Vector_E2<T>& v) const;

    Transformation_E2<T> operator*(const Transformation_E2<T>& TR) const;
};

template<typename T>
void
Transformation_E2<T>::setToIdentity()
{
    m00 = m11 = 1;
    m01 = m10 = m02 = m12 = 0;
}
template<typename T>
Transformation_E2<T>::Transformation_E2()
{
    setToIdentity();
}

template<typename T>
Transformation_E2<T>::Transformation_E2(const Identity& t)
{
    setToIdentity();
}

template<typename T>
Transformation_E2<T>::Transformation_E2(const Scale& s, const Vector_E2<T>& v)
{
    setToIdentity();
    m00 = v.x();
    m11 = v.y();
}

template<typename T>
Transformation_E2<T>::Transformation_E2(const Rotation& r, const T& angle)
{
    setToIdentity();
    const T c = cos(angle);
    const T s = sin(angle);
    m00 = c;
    m01 = -s;
    m10 = s;
    m11 = c;
}

template<typename T>
Transformation_E2<T>::Transformation_E2(const Translation& t, const Vector_E2<T>& v)
{
    setToIdentity();
    m02 = v.x();
    m12 = v.y();
}

template<typename T>
Point_E2<T>
Transformation_E2<T>::transform(const Point_E2<T>& p) const
{
    return Point_E2<T>(m00 * p.x() + m01 * p.y() + m02 ,
		       m10 * p.x() + m11 * p.y() + m12 );
}

template<typename T>
Line_E2<T>
Transformation_E2<T>::transform(const Line_E2<T>& l) const
{
    assert(false);
    return Line_E2<T>();
}

template<typename T>
Vector_E2<T>
Transformation_E2<T>::transform(const Vector_E2<T>& v) const
{
    return Vector_E2<T>(m00 * v.x() + m01 * v.y(),
			m10 * v.x() + m11 * v.y());
}

template<typename T>
Segment_E2<T>
Transformation_E2<T>::transform(const Segment_E2<T>& seg) const
{
    Point_E2<T> s = transform(seg.source());
    Point_E2<T> t = transform(seg.target());

    return Segment_E2<T>(s,t);
}

template<typename T>
Point_E2<T>
Transformation_E2<T>::operator()(const Point_E2<T>& p) const
{
    return transform(p);
}

template<typename T>
Line_E2<T>
Transformation_E2<T>::operator()(const Line_E2<T>& l) const
{
    return transform(l);
}

template<typename T>
Vector_E2<T>
Transformation_E2<T>::operator()(const Vector_E2<T>& v) const
{
    return transform(v);
}

template<typename T>
Transformation_E2<T>
Transformation_E2<T>::operator*(const Transformation_E2<T>& TR) const
{
    // Notice the order of composition:
    //      *this is applied second and TR is applied first
    return Transformation_E2<T>(m00 * TR.m00 + m01 * TR.m10         ,
				m00 * TR.m01 + m01 * TR.m11         ,
				m00 * TR.m02 + m01 * TR.m12  +  m02 ,

				m10 * TR.m00 + m11 * TR.m10         ,
				m10 * TR.m01 + m11 * TR.m11         ,
				m10 * TR.m02 + m11 * TR.m12  +  m12 );
}

#endif // TRANSFORMATION_E2_H
