/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#include <cassert>

#include "bbox_e3.h"

#include "epsilon_double.h"
#include "transformation_e3.h"
#include "predicates_e3.h"

typedef Epsilon_double MyDouble;

typedef Point_E3<MyDouble>   Point_E3d;
typedef Vector_E3<MyDouble> Vector_E3d;
typedef Segment_E3<MyDouble> Segment_E3d;
typedef Transformation_E3<MyDouble> Transformation_E3d;

typedef Bbox_E3<MyDouble>   Bbox_E3d;

void test_Transformation_E3()
{
    Point_E3d P1(5,7,9);
    Vector_E3d V1(11,13,17);

    Point_E3d P2 = P1 + V1;

    Transformation_E3d T(ORTHOGONAL, Point_E3d(4,5,6), Point_E3d(7,8,9));

    Point_E3d P1t = T(P1);
    Vector_E3d V1t = T(V1);

    Point_E3d P2t = T(P2);

    assert( P2t == P1t + V1t );
}

void test_Transformation_E3_operator_mult()
{
    Point_E3d P(3,4,5);

    Transformation_E3d T1(2,3,4,5, 6,7,8,9, 10,11,12,13);
    Transformation_E3d T2(14,15,16,17, 2,3,4,5, 6,7,8,9);

    Transformation_E3d T21 = T2 * T1; // T1 is applied first
	
    assert(T2(T1(P)) == T21(P));
}

void test_Bbox_E3_operator_incr()
{
    Point_E3d P1(-5,-4,-3);
    Point_E3d P2( 7, 6, 5);

    Bbox_E3d box(P1);
    box += P2;

    assert( box == Bbox_E3d(P1, P2) );

    assert( box.center() == Point_E3d(1,1,1) );
}	

void test_dominant()
{
    const Point_E3d A(5,5,5);
    const Point_E3d B(8,7,6);
    const Point_E3d C(7,8,6);
    const Point_E3d D(6,7,8);

    const Segment_E3d AB(A,B);
    const Segment_E3d AC(A,C);
    const Segment_E3d AD(A,D);

    assert( dominant(AB) == E3_POSX );
    assert( dominant(AC) == E3_POSY );
    assert( dominant(AD) == E3_POSZ );

    const Vector_E3d v1(1,1,0.01);
    assert( least_dominant(v1) == E3_POSZ );
}

int main()
{
    test_Transformation_E3();
    test_Transformation_E3_operator_mult();
    test_Bbox_E3_operator_incr();
    test_dominant();
}
